/* SPDX-License-Identifier: GPL-3.0 */

#ifndef __HASH_UTILS_H__
#define __HASH_UTILS_H__

#include "../global/global.h"

#include <stdio.h>
#include <errno.h>
#include <execinfo.h>
#include <string.h>

#define NEW_STRING(size) malloc(size)
#define NEW_STRING_EMPTY(size) calloc(size, sizeof(char))

#define IN_MEMORY_COPY(dest, src) strcpy(dest, src)
#define IN_MEMORY_COPY_N(dest, src, n) strncpy(dest, src, n)
#define STARTS_WITH(str1, str2) (strncmp(str1, str2, strlen(str1)) == 0)
#define ENDS_WITH(str1, str2) (strlen(str2) > strlen(str1) && \
	(strncmp(str1, str2 + strlen(str2) - strlen(str1), strlen(str1)) == 0))

#define SET_FLAG(flags, flag) { flags |= flag; }
#define UNSET_FLAG(flags, flag) do { \
		flags &= ~(flag); \
	} while (FALSE)
#define IS_SET_FLAG(flags, flag) ((flags) & (flag))

#define MAX(a, b) ((a) > (b) ? (a) : (b))
#define MIN(a, b) ((a) < (b) ? (a) : (b))

typedef char                            *string;

#define SUCCESS 0
#define EFAIL 1

#define LINE_SIZE 256

#define DELIMITERS "\t []{}<>=+-*/%!&|^.,:;()\\"

#define ASSERT(test, err_code) do {\
		if (!(test)) {\
			err_code\
		} \
	} while (FALSE)

#define FREE(ptr) do {\
		if (ptr)\
			free(ptr);\
		ptr = NULL; \
	} while (FALSE)

#define LOG(msg, ...)  do {\
		fprintf(stdout, "[%s][%s:%d] ", __FILE__, __func__, __LINE__);\
		fprintf(stdout, msg, __VA_ARGS__);\
		fprintf(stdout, "\n\n");\
	} while (FALSE);

#define TRACE(trace) do {\
		if (!trace) break; \
		int size = 128; \
		void* callstack[size];\
		int i, frames = backtrace(callstack, size);\
		char** strs = backtrace_symbols(callstack, frames);\
		printf("***DVT_DEBUG_TRACE\n"); \
		for (i = 0; i < frames; ++i) \
			fprintf(stdout, "\t%s\n", strs[i]);\
		printf("\n\n"); \
		free(strs);\
	} while (FALSE)	

typedef struct {
    char *str;
    size_t size;
    size_t capacity;
} str_builder;

static void str_builder_init(str_builder *sb) {
    sb->str = malloc(128);
    sb->size = 0;
    sb->capacity = 128;
}

static void str_builder_append(str_builder *sb, const char *fmt, ...) {
    va_list args;
    va_start(args, fmt);
    
    size_t current_size = sb->size;
    size_t required_size = 256;

    size_t new_required = current_size + required_size;
    if (new_required > sb->capacity) {
        size_t new_capacity = new_required;
        new_capacity = (new_capacity < 512) ? 512 : new_capacity;

        char* temp = realloc(sb->str, new_capacity);
        if (temp == NULL) {
            free(sb->str);
            sb->str = NULL;
            sb->size = 0;
            sb->capacity = 0;
            va_end(args);
            return;
        }
        sb->str = temp;
        sb->capacity = new_capacity;
    }

    va_list copy_args;
    va_copy(copy_args, args);
    int bytes_written = vsnprintf(sb->str + current_size, sb->capacity - current_size, fmt, copy_args);
    va_end(copy_args);

    if (bytes_written >= 0) {
        sb->size = current_size + (size_t)bytes_written;
    } else {
        sb->size = current_size;
    }

    va_end(args);
}
static void str_builder_free(str_builder *sb) {
    free(sb->str);
}

static void log_code_trace(str_builder *sb, int trace, int _error_code, char *msg) {
    if (msg == NULL || strcmp(msg, "Success") == 0) {
    	TRACE(trace);
        return; 
    }
    
    str_builder_append(sb, "\\n");
    str_builder_append(sb, "\tError: %s\\n", msg);
    
    if (_error_code != 0)
        str_builder_append(sb, "\tCode error: %d\\n", _error_code);
       
   	TRACE(trace);
}

#define ERROR_LOG_F(trace, msg, ...) do { \
    str_builder sb = {0};         \
    str_builder_init(&sb);        \
    str_builder_append(&sb, "*** Error: [%s][%s:%d] ", __FILE__, __func__, __LINE__);\
    str_builder_append(&sb, msg, __VA_ARGS__);\
    log_code_trace(&sb, trace, errno, strerror(errno));\
	str_builder_append(&sb, "\n"); \
    fprintf(stderr, "%s", sb.str); \
    str_builder_free(&sb);        \
} while (FALSE)

#define ERROR_LOG_DL(trace, msg, ...) do {\
    str_builder sb = {0};         \
    str_builder_init(&sb);        \
    str_builder_append(&sb, "*** Error: [%s][%s:%d] ", __FILE__, __func__, __LINE__);\
    str_builder_append(&sb, msg, __VA_ARGS__);\
    log_code_trace(&sb, trace, 0, dlerror());\
    str_builder_append(&sb, "\n"); \
	fprintf(stderr, "%s", sb.str); \
    str_builder_free(&sb);        \
} while (FALSE)

#define ERROR_LOG(trace, msg, ...) do { \
    str_builder sb = {0};         \
    str_builder_init(&sb);        \
    str_builder_append(&sb, "*** Error: [%s][%s:%d] ", __FILE__, __func__, __LINE__);\
    str_builder_append(&sb, msg);\
    log_code_trace(&sb, trace, errno, strerror(errno));\
	str_builder_append(&sb, "\n"); \
    fprintf(stderr, "%s", sb.str); \
    str_builder_free(&sb);        \
} while (FALSE)

#define F_WRITE_LINE 1
#define F_ALWAYS_RESET_WRITE_LINE (1 << 2)
#define F_LAST_COND_VALUE (1 << 3)

#endif
