/* SPDX-License-Identifier: GPL-3.0 */

#ifndef HASH_MAP_H
#define HASH_MAP_H

#include "vector.h"
#include "svdpi.h"
#include "utils.h"

#include <stdlib.h>
#include <string.h>

#define HASHMAP_SUCCESS 0
#define HASHMAP_ERR_ALLOC 1
#define HASHMAP_W_NOTFOUND 2

#define DEFAULT_LOAD_FACTOR 0.75
#define DEFAULT_LEN 16

#define PRIME 31

#define HASHMAP_INITIALIZER \
	{ NULL, NULL, NULL, NULL, NULL, 0, 0, 0 }

/***** STRUCTURES *****/

typedef struct Node {
	void *key;
	void *value;
	struct Node *next;
} Node;

typedef struct Hashmap {
	Node **table;

	uint64_t(*hash_fn)(void *key);
	bool (*eq_fn)(void *p1, void *p2);
	void *(*el_cst_fn)(void *el);
	void *(*key_cst_fn)(void *el);
	uint64_t table_len;
	float load_factor;
	uint64_t element_ct;

	int32_t (*put)(struct Hashmap *, void *, void *);
	void *(*get)(struct Hashmap *, void *);

	Vector *(*keys)(struct Hashmap *);

} Hashmap;

/***** METHODS *****/

int32_t hashmap_init(Hashmap *map, uint64_t(*hash_fn)(void *key),
	bool (*eq_fn)(void *p1, void *p2),
	void *(*el_cst_fn)(void *el), void *(*key_cst_fn)(void *el),
	uint64_t start_len, float load_factor);
void hashmap_destroy(Hashmap *map);

int32_t put(Hashmap *map, void *key, void *value);
void *get(Hashmap *map, void *key);
Vector *keys(struct Hashmap * self);

int32_t hashmap_drop(Hashmap *map, void *key);

/***** UTILITY *****/

uint64_t string_hash(void *key);

bool string_eq(void *p1, void *p2);

void *string_copy(void *el);

uint64_t pointer_hash(void *p);

bool pointer_eq(void *p1, void *p2);

/***** PRIVATE *****/
int32_t _hashmap_resize(Hashmap *map);


#endif
