#!/bin/bash

##############################################################################
### Prerequisites
##############################################################################

set -o nounset

: "${_DEBUG:=FALSE}"

if [ "$_DEBUG" = "TRUE" ] ; then
	PS4='+\[\e[1;31m\]${LINENO}\[\e[0m\]\n'
	export PS4
	set -x
	#trap read debug
fi

REQUIRED_CMDS="pwd dirname mktemp cp mv rm mkdir cat touch sed"
MISSING_CMDS=0
for cmd in $REQUIRED_CMDS; do
	if ! hash "$cmd" >/dev/null 2>&1 ; then
		echo "[ERROR] Command not found in PATH: $cmd" >&2
	MISSING_CMDS=$((MISSING_CMDS + 1))
	fi
done
if [ $MISSING_CMDS -gt 0 ] ; then
	echo "[ERROR] Minimum $MISSING_CMDS commands are missing in PATH" >&2
	exit 1
fi

cd $(dirname $0)
export BD=$(pwd)

function msg() {
    echo -e "$1"
}

function fatal() {
 	msg "$1" >&2
  	exit 1;
}
 
function error() {
	msg "$1" >&2
}

function escape() {
	echo ${!1} | sed -e 's/[\/&]/\\&/g'
}

##############################################################################
### Variables
##############################################################################

: "${JAVA_BIN:=}"
[ -n "$JAVA_BIN" ] || fatal "JAVA_BIN not set"

YUI_COMPRESSOR="$BD/../../../../../../../../../ro.amiq.dvt.releng/utils/yuicompressor.jar"
CLOSURE_COMPILER_COMPRESSOR="$BD/../../../../../../../../../ro.amiq.dvt.releng/utils/closure-compiler-v20210202.jar"
VLOG_RESOURCES_VERISSIMO_RULESET_EDITOR="$BD/../../../../../../../../resources/verissimo_ruleset_editor"
VLOG_RESOURCES_LIB="$BD/../../../../../../../../resources/lib"

##############################################################################
### Tasks
##############################################################################

function minify_files() {

	local tmp_style=$(mktemp /tmp/style.XXXX) || fail "Can't create temp style file"
	if   [ $1 == "CSS" ] ; then
		for each in "${@:2}" ; do
			cat "$each" >> $tmp_style
		done
		$JAVA_BIN -jar $YUI_COMPRESSOR --type css --line-break 180 -o "$VLOG_RESOURCES_VERISSIMO_RULESET_EDITOR/ruleset_editor.css" $tmp_style
	fi
	rm $tmp_style

	local tmp_script=$(mktemp /tmp/script.XXXX) || fail "Can't create temp script file"	
	if   [ $1 == "JS" ] ; then
		for each in "${@:2}" ; do
			cat "$each" >> $tmp_script
		done
		$JAVA_BIN -jar $CLOSURE_COMPILER_COMPRESSOR --warning_level QUIET --js $tmp_script --js_output_file "$VLOG_RESOURCES_VERISSIMO_RULESET_EDITOR/rulesetEditor.js"
	fi
	rm $tmp_script

}

function build() {
	msg "Compressing JS Code..."
	minify_files     "JS"   "$VLOG_RESOURCES_LIB/jquery/jquery.js" \
							"$VLOG_RESOURCES_LIB/jquery/jquery-ui-effects.js" \
							"$VLOG_RESOURCES_LIB/selectize/selectize.js" \
							"$VLOG_RESOURCES_VERISSIMO_RULESET_EDITOR/FileSaver.js" \
							"$VLOG_RESOURCES_VERISSIMO_RULESET_EDITOR/xml-formatter.js" \
							"$BD/src/rulesetEditor.js" \
							"$BD/src/searchFilter.js"
	msg "Compressing CSS Code..."
	minify_files     "CSS" 	"$VLOG_RESOURCES_LIB/selectize/selectize.css" \
							"$VLOG_RESOURCES_LIB/selectize/selectize.bootstrap3.css" \
							"$BD/src/ruleset_editor.css"
}

arg="${1:-}"

if [ "$arg" = "build" ] ; then
	build
else
	fatal "Unknown task $arg, try [build]"
fi

msg "Finished"
