var C = {

	XML_INDENTATION: 2,

	STATE: {
		SELECT_RULES: 0,
		EDIT_RULESET: 1
	},

	SELECTION_MODE: {
		ACTIVE: 0,
		NOT_ACTIVE: 1
	},

	BUTTON_TEXT: {
		ADD_TO_RULESET: 'Add to Ruleset',
		REMOVE_FROM_RULESET: 'Remove from Ruleset',
		SELECT: 'Select',
		UNSELECT: 'Unselect',
		SHOW_DESCRIPTION_APPEND: '&#x2610;',
		HIDE_DESCRIPTION_APPEND: '&#10697;'
	},

	TOOLTIP: {
		SHOW_DESCRIPTION_APPEND: 'Show description-append',
		HIDE_DESCRIPTION_APPEND: 'Hide description-append'
	},

	ANNOTATION_TYPE: {
		FS: {
			NAME: 'FS',
			FIELD_NAME: 'fs-field',
			FIELD_VALUES: ['last-modified']
		},
		GIT: {
			NAME: 'GIT',
			FIELD_NAME: 'git-field',
			FIELD_VALUES: ['author', 'author-mail', 'author-time', 'committer', 'committer-mail', 'committer-time', 'SHA1', 'summary']
		},
		P4: {
			NAME: 'P4',
			FIELD_NAME: 'p4-field',
			FIELD_VALUES: ['author', 'revision-number', 'date']
		},
		SVN: {
			NAME: 'SVN',
			FIELD_NAME: 'svn-field',
			FIELD_VALUES: ['author', 'commit', 'date']
		},
		CC: {
			NAME: 'CC',
			FIELD_NAME: 'cc-field',
			FIELD_VALUES: ['username', 'full-username', 'date', 'comment']
		},
		TAG: {
			NAME: 'TAG',
			FIELD_NAMES: 'tag-key',
		},
		LABEL: {
			NAME: 'LABEL',
		},
	},

	COMPILATION_RULES: {
		IDS: ['SYNTACTIC_PROBLEM', 'SEMANTIC_PROBLEM', 'NON_STANDARD'],
		DEFAULT_SEVERITY: 'Severity of the compilation failures'
	}
}

let state;
let selectionMode = C.SELECTION_MODE.NOT_ACTIVE;
let shouldHideAddedRules = false;
const rulesWithCopiesInRuleset = new Set();
const categoriesToRulesInRuleset = new Map();
const nameIndexes = new Map();
const currentSelection = new Set();
const tagSuggestions = new Map();
const tagKeys = new Set();

const categoryTemplateString = `<div class="category" id="" data-name=""><div class="categoryHead"><h2><input class="editableCategoryName" type="text" placeholder="UNNAMED CATEGORY" value="" autocomplete="off"/></h2><button class="button" id="categorySelectAll">Select Category</button><button class="button" id="categoryUnselectAll" style="display:none;">Clear Selection</button></div><div class="categoryContent"></div></div>`;
let categoryTemplate = document.createElement('div');
categoryTemplate.innerHTML = categoryTemplateString;
categoryTemplate = categoryTemplate.firstElementChild;

const allDefaultCategories = Array.from(document.querySelectorAll('div.category[data-name=""]'));
const allRulesInDefaultCategories = Array.from(document.querySelectorAll('div.category[data-name=""] div.rule'));
const selectRulesStateTitle = document.querySelector('div#titleHeader h1#selectRulesStateTitle');
const editRulesetStateTitle = document.querySelector('div#titleHeader h1#editRulesetStateTitle');
const tocContainer = document.querySelector('div#toc ul');
const categoriesContainer = document.getElementById('content');
const filterResults = document.getElementById('filterResults');
const moveToCategoryButton = document.getElementById('moveToCategoryButton');
const hideAddedRulesButton = document.getElementById('showAddedRules');
const selectRulesStateButton = document.getElementById('selectRulesState');
const editRulesetStateButton = document.getElementById('editRulesetState');
const generateRulesetButton = document.getElementById('generateRulesetButton');
const removeRulesButton = document.getElementById('removeAllFromCustomRuleset');
const allUnselectAllButton = document.getElementById('allUnselectAll');
const scrollToTop = document.getElementById("scrollToTop");
const filterInput = document.getElementById("filterInput");
const confidential = document.getElementById("confidential");
const editConfigurationButton = document.getElementById('editConfigurationButton');
const editXMLPreferences = document.getElementById('editXMLPreferences');

const modalOptions = new Set();
modalOptions.add('modalExistingCategories');
modalOptions.add('modalNewCategoryName');
const modal = document.getElementById('modal');
const addToDefaultCategoriesOption = document.getElementById('addToDefaultCategories');
const addToExistingCategoryOption = document.getElementById('radioModalExistingCategories');
const addToNewCategoryOption = document.getElementById('radioModalNewCategory');
const displayCurrentSelectionElement = document.getElementById('currentSelectionElement');
const newCategoryNameInput = document.getElementById('modalNewCategoryName');

Selectize.define('dvt-addons', function (options) {

	this.hideInput = (function () {
		return function () {
			const self = this;
			self.setTextboxValue('');
			self.$control_input.css({
				opacity: 0,
				position: 'relative',
				left: 0
			});
			self.isInputHidden = true;
		};
	})();

	this.showInput = (function () {
		return function () {
			this.$control_input.css({
				opacity: 1,
				position: 'relative',
				left: 0
			});
			this.isInputHidden = false;
		};
	})();

	this.getValue = (function () {
		return function () {
			return this.items;
		};
	})();

	this.addItems = (function () {
		return function (values) {
			const items = $.isArray(values) ? values : [values];
			for (let i = 0, n = items.length; i < n; i++) {
				this.isPending = (i < n - 1);
				if (this.options.hasOwnProperty(items[i]))
					this.addItem(items[i]);
				else
					this.addNewItem(items[i]);
			}
			this.isPending = false;
		};
	})();

	this.addNewItem = (function () {

		const hash_key = function (value) {
			if (typeof value === 'undefined' || value === null)
				return '';
			if (typeof value === 'boolean')
				return value ? '1' : '0';
			return value + '';
		};

		return function (input) {
			const self = this;
			const data = {};
			data[self.settings.labelField] = input;
			data[self.settings.valueField] = input;

			self.unlock();

			if (!data || typeof data !== 'object')
				return;
			const value = hash_key(data[self.settings.valueField]);
			if (!value)
				return;

			self.addOption(data);
			self.addItem(value);
			self.refreshOptions(false);
		};
	})();

	this.removeAllItems = (function () {
		return function () {
			const selectedItems = this.items;
			while (selectedItems.length > 0)
				this.removeItem(selectedItems[0]);
		}
	})();

});

let editingCustomCategoryName;
let editingAnnotationName;
let editingRuleName;
let editingTagKeyName;

const categoriesFilterInitialData = getCategoriesFilterData(C.STATE.SELECT_RULES).map(item => item.text);
let [categoriesFilterComponent, categoriesFilterValues] = initFilter("categoriesFilterInput", "categoriesRemoveAll", getCategoriesFilterData(state));
let [labelsFilterComponent, labelsFilterValues] = initFilter("labelsFilterInput", "labelsRemoveAll", getLabelsFilterData(state));

function getCategoriesFilterData(currentState) {
	let categoriesFilterData = [];
	if (currentState == C.STATE.SELECT_RULES) {
		allDefaultCategories.forEach(category => {
			categoriesFilterData.push({
				text: category.id
			});
		});
	} else {
		categoriesToRulesInRuleset.forEach((rules, category) => {
			categoriesFilterData.push({
				text: category.dataset.name
			});
		});
	}

	return categoriesFilterData;
}

function areFiltersActive() {
	return filterInputValue !== '' || (categoriesFilterValues !== null && categoriesFilterValues.length > 0) || (labelsFilterValues !== null && labelsFilterValues.length > 0);
}

function notify(message) {
	if (!message) {
		message = 'Internal error!';
	}

	if (!window['vscodeOpen']) {
		alert(message);
	} else {
		vscode.postMessage({
			type: 'NOTIFICATION',
			data: {
				eventType: 'ALERT',
				context: {
					message: message
				}
			},
		});
	}
}

async function receiveConfirmation(message) {
	return new Promise((resolve) => {
		const handler = (event) => {
			const response = event.data;
			if (response.type === 'NOTIFICATION' && response.data.eventType === 'MISSING_MANDATORY_PARAMETER_VALUE') {
				window.removeEventListener('message', handler);
				resolve(response.data.context.answer);
			}
		};

		window.addEventListener('message', handler);

		vscode.postMessage({
			type: 'NOTIFICATION',
			data: {
				eventType: 'MISSING_MANDATORY_PARAMETER_VALUE',
				context: {
					message: message
				}
			},
		});
	});
}

async function request(message) {
	if (!message) {
		notify('Internal error!');
	}

	if (!window['vscodeOpen']) {
		return confirm(message);
	} else {
		return await receiveConfirmation(message);
	}
}

function scrollFunction() {
	if (document.body.scrollTop > 800 || document.documentElement.scrollTop > 800) {
		scrollToTop.style.display = "block";
	} else {
		scrollToTop.style.display = "none";
	}
}

function scrollToTopFunction() {
	window.scrollTo(0, 0);
}

function adjustTextareaHeight(textarea) {
	textarea.style.height = '';
	textarea.style.height = textarea.scrollHeight + 3 + 'px';
	textarea.dataset.adjusted = 'true';
}

function changeTableColor(table, added) {
	const ths = table.querySelectorAll('th');
	const tds = table.querySelectorAll('td');
	let className = '';

	if (added) {
		className = 'added';
	}

	ths.forEach(th => {
		th.className = className;
	})
	tds.forEach(td => {
		td.className = className;
	})
}

function createAnnotationSelect(values) {
	const select = document.createElement('select');
	for (const value of values) {
		const option = document.createElement('option');
		option.value = value;
		option.text = value
		select.append(option);
	}
	return select;
}

function getNextIndexedName(currentName) {
	let index = nameIndexes.get(currentName);
	if (!index) {
		index = 1;
	} else {
		index = index + 1;
	}
	nameIndexes.set(currentName, index);

	return `${currentName}_${index}`;
}

function highlightListItem(listItems, currentFocus, list) {
	unhighlightListItem(listItems);
	if (currentFocus >= listItems.length)
		currentFocus = 0;
	if (currentFocus < 0)
		currentFocus = (listItems.length - 1);

	listItems[currentFocus].classList.add("autocomplete-active");
	list.dataset.focus = currentFocus.toString();
}

function closeOtherSuggestions(currentElement) {
	const listElements = document.getElementsByClassName("autocomplete-items");
	for (const listElement of listElements)
		if (!currentElement || listElement !== currentElement)
			listElement.parentNode.removeChild(listElement);
}

function unhighlightListItem(listItems) {
	for (const listItem of listItems) {
		listItem.classList.remove("autocomplete-active");
	}
}

function toggleElement(element, button) {
	if (button.className === 'button') {
		element.style.display = '';
		button.className = 'clickedButton';
		if (button.id !== 'helpButton') {
			document.getElementById('areaWrapper').style.display = '';
		} else {
			document.getElementById('areaWrapper').style.display = 'none';
		}
	} else {
		element.style.display = 'none';
		button.className = 'button';
		document.getElementById('areaWrapper').style.display = 'none';
	}
}

function hideConfigurationAreas(event) {
	if (!event.target.matches('.wrap *') && event.target.id !== 'editConfigurationButton' && event.target.id !== 'helpButton' && event.target.id !== 'editXMLPreferences' && event.target.id !== 'annotationDelete') {
		document.getElementById('helpDiv').style.display = 'none';
		document.querySelector('button#helpButton').className = 'button';
		document.getElementById('areaWrapper').style.display = 'none';
		document.querySelector('button#editConfigurationButton').className = 'button';
		document.querySelector('button#editXMLPreferences').className = 'button';
	}
}

function disableOtherModalOptions(currentModalOptionId) {
	if (currentModalOptionId !== 'noid') {
		const currentModalOption = document.getElementById(currentModalOptionId);
		if (currentModalOption) {
			currentModalOption.disabled = false;
		}
	}

	modalOptions.forEach(modalOption => {
		if (modalOption !== currentModalOptionId) {
			const otherOption = document.getElementById(modalOption);
			if (otherOption) {
				otherOption.disabled = true;
			}
		}
	});
}

function changeStates() {
	scrollToTopFunction();

	categoriesFilterComponent.clearOptions();
	categoriesFilterComponent.addOption(getCategoriesFilterData(state));
	categoriesFilterComponent.refreshOptions(false);

	if (state === C.STATE.SELECT_RULES && nOfAllChecks === null) {
		filterSearch();
	}

	if (rulesWithCopiesInRuleset !== null && rulesWithCopiesInRuleset.size < 1) {
		if (state === C.STATE.SELECT_RULES) {
			document.getElementById('infoMessage').style.display = 'none';
		} else {
			document.getElementById('infoMessage').style.display = '';
		}
	}

	const defaultCategoryDisplay = state === C.STATE.SELECT_RULES ? '' : 'none';
	const rulesetCategoryDisplay = state === C.STATE.SELECT_RULES ? 'none' : '';

	if (state === C.STATE.SELECT_RULES && shouldHideAddedRules) {
		allDefaultCategories.forEach(category => {
			const rules = category.querySelectorAll('div.rule');
			for (const rule of rules) {
				if (!rulesWithCopiesInRuleset.has(rule)) {
					category.style.display = defaultCategoryDisplay;
					const li = document.querySelector(`li[data-id='${category.id}']`);
					li.style.display = defaultCategoryDisplay;
					break;
				}
			}
		});
	} else {
		allDefaultCategories.forEach(category => {
			category.style.display = defaultCategoryDisplay;
			const li = document.querySelector(`li[data-id='${category.id}']`);
			li.style.display = defaultCategoryDisplay;
		});
	}

	let nofRulesInRuleset = 0;
	categoriesToRulesInRuleset.forEach((rules, category) => {
		nofRulesInRuleset += rules.size;
		category.style.display = rulesetCategoryDisplay;
		const li = document.querySelector(`li[data-id='${category.id}']`);
		li.style.display = rulesetCategoryDisplay;

		if (state === C.STATE.EDIT_RULESET) {
			rules.forEach(rule => {
				const textarea = rule.querySelector('textarea');
				if (!textarea.dataset.adjusted || textarea.dataset.adjusted === 'false') {
					adjustTextareaHeight(textarea);
				}
			});
		}
	});

	nOfChecks = state === C.STATE.SELECT_RULES ? (shouldHideAddedRules ? nOfAllChecks - rulesWithCopiesInRuleset.size : nOfAllChecks) : nofRulesInRuleset;
	filterResults.textContent = nOfChecks + ' checks';
}

function resetFilters() {
	filterInput.value = '';
	if (categoriesFilterComponent !== null)
		categoriesFilterComponent.removeAllItems();
	if (labelsFilterComponent !== null)
		labelsFilterComponent.removeAllItems();
	parseInput();
	filterSearch();
}

function addCustomCategoryToFilterOptions(categoryName) {
	if (categoryName !== null && state === C.STATE.EDIT_RULESET) {
		categoriesFilterComponent.addOption({ text: categoryName });
		categoriesFilterComponent.refreshOptions(false);
	}
}

function removeCustomCategoryFromFilterOptions(categoryName) {
	if (categoryName !== null && state === C.STATE.EDIT_RULESET) {
		categoriesFilterComponent.removeOption(categoryName);
		categoriesFilterComponent.refreshOptions(false);
	}
}

function editRulesetState() {
	if (selectionMode === C.SELECTION_MODE.ACTIVE) {
		clearCurrentSelection();
	}

	if (areFiltersActive()) {
		resetFilters();
	}

	state = C.STATE.EDIT_RULESET;
	moveToCategoryButton.textContent = 'Move to Category';
	addToDefaultCategoriesOption.style.display = 'none';
	hideAddedRulesButton.style.display = 'none';
	selectRulesStateTitle.style.display = 'none';
	editRulesetStateTitle.style.display = '';
	confidential.style.display = 'none';
	selectRulesStateButton.className = 'button';
	editRulesetStateButton.className = 'clickedButton';
	editXMLPreferences.style.visibility = 'visible';
	editConfigurationButton.style.visibility = 'visible';

	if (rulesWithCopiesInRuleset !== null && rulesWithCopiesInRuleset.size !== 0) {
		generateRulesetButton.style.display = '';
	}

	changeStates();
}

function selectRulesState() {
	if (selectionMode === C.SELECTION_MODE.ACTIVE) {
		clearCurrentSelection();
	}

	if (areFiltersActive()) {
		resetFilters();
	}

	state = C.STATE.SELECT_RULES;
	moveToCategoryButton.textContent = 'Add to Category';
	addToDefaultCategoriesOption.style.display = '';
	hideAddedRulesButton.style.display = '';
	selectRulesStateTitle.style.display = '';
	editRulesetStateTitle.style.display = 'none';
	confidential.style.display = 'inline';
	selectRulesStateButton.className = 'clickedButton';
	editRulesetStateButton.className = 'button';
	generateRulesetButton.style.display = 'none';
	editXMLPreferences.style.visibility = 'hidden';
	editConfigurationButton.style.visibility = 'hidden';

	changeStates();
}

function checkIfCategoryShouldBeHidden(category, categoryTOC) {
	const categoryContent = category.querySelector('div.categoryContent');
	const rules = categoryContent.children;
	let hideCategory = true;
	for (const rule of rules) {
		if (rule.style.display !== 'none') {
			hideCategory = false;
			break;
		}
	}

	if (hideCategory) {
		category.style.display = 'none';
		categoryTOC.style.display = 'none';
	}
}

function checkIfCategoriesShouldBeHidden() {
	allDefaultCategories.forEach(category => {
		const categoryTOC = document.querySelector(`li[data-id="${category.id}"]`);
		checkIfCategoryShouldBeHidden(category, categoryTOC)
	});
}

function hideRule(rule, category, checkCategory) {
	rule.style.display = 'none';
	const ruleTOC = document.querySelector(`li[data-id="${rule.id}"]`);
	ruleTOC.style.display = 'none';

	if (checkCategory) {
		const categoryTOC = document.querySelector(`li[data-id="${category.id}"]`);
		checkIfCategoryShouldBeHidden(category, categoryTOC);
	}
}

function showRule(rule, category) {
	rule.style.display = '';
	const ruleTOC = document.querySelector(`li[data-id="${rule.id}"]`);
	ruleTOC.style.display = '';

	if (category.style.display === 'none') {
		category.style.display = '';
		const categoryTOC = document.querySelector(`li[data-id="${category.id}"]`);
		categoryTOC.style.display = '';
	}
}

function hideAddedRules(hide) {
	let nofRules = 0;
	if (hide) {
		hideAddedRulesButton.className = 'clickedButton';
		shouldHideAddedRules = true;
		rulesWithCopiesInRuleset.forEach(rule => {
			const category = rule.closest('div.category');
			hideRule(rule, category, false);
			nofRules++;
		});

		if (!areFiltersActive()) {
			checkIfCategoriesShouldBeHidden();
		}
	} else {
		shouldHideAddedRules = false;
		hideAddedRulesButton.className = 'button';
		rulesWithCopiesInRuleset.forEach(rule => {
			const category = rule.closest('div.category');
			showRule(rule, category);
			nofRules++;
		});
	}

	if (areFiltersActive()) {
		filterSearch();
	} else {
		updateFilterResults(!hide, nofRules);
	}
}

function createEditableRuleCopy(rule) {
	const ruleCopy = rule.cloneNode(true);
	ruleCopy.querySelector('button#duplicateRule').style.visibility = 'visible';
	ruleCopy.querySelector('table').style.cursor = '';
	changeIdForRule(ruleCopy);

	const nameData = ruleCopy.querySelector('tr#nameData');
	nameData.lastElementChild.style.display = '';
	nameData.removeChild(nameData.firstElementChild.nextElementSibling);

	const titleData = ruleCopy.querySelector('tr#titleData');
	titleData.lastElementChild.style.display = '';
	titleData.removeChild(titleData.firstElementChild.nextElementSibling);

	const severityData = ruleCopy.querySelector('tr#severityData');
	severityData.lastElementChild.style.display = '';
	severityData.removeChild(severityData.firstElementChild.nextElementSibling);

	const descriptionData = ruleCopy.querySelector('tr#descriptionData');
	descriptionData.lastElementChild.style.display = '';
	descriptionData.removeChild(descriptionData.firstElementChild.nextElementSibling);
	descriptionData.querySelector('div.centered').style.display = '';

	const tagsData = ruleCopy.querySelector('tr#tagsData');
	if (tagsData) {
		const tagsInfo = tagsData.querySelector('div#tagsInfo');
		if (tagsInfo) {
			tagsData.style.display = '';
		}
	}

	const parametersDescriptionData = ruleCopy.querySelector('tr#parametersDescriptionData');
	if (parametersDescriptionData) {
		parametersDescriptionData.firstElementChild.textContent = 'Parameters Description';
		const parametersValuesData = ruleCopy.querySelector('tr#parametersValuesData');
		parametersValuesData.style.display = '';
	}

	const autocorrectEnabledData = ruleCopy.querySelector('tr#autocorrectEnabledData');
	if (autocorrectEnabledData) {
		autocorrectEnabledData.style.display = '';
		const radioInputs = autocorrectEnabledData.querySelectorAll(`td input`);
		radioInputs.forEach(radioInput => radioInput.name = ruleCopy.id);
	}

	const autocorrectParametersDescriptionData = ruleCopy.querySelector('tr#autocorrectParametersDescriptionData');
	if (autocorrectParametersDescriptionData) {
		autocorrectParametersDescriptionData.firstElementChild.textContent = 'Autocorrect Parameters Description';
		const autocorrectParametersValuesData = ruleCopy.querySelector('tr#autocorrectParametersValuesData');
		autocorrectParametersValuesData.style.display = '';
	}

	return ruleCopy;
}

function createRuleTocCopy(rule, ruleCopy, setTextContent) {
	const ruleTocCopy = document.querySelector(`li[data-id='${rule.id}']`).cloneNode(true);
	ruleTocCopy.dataset.id = ruleCopy.id;
	const nextLocationLink = ruleTocCopy.querySelector('a');
	nextLocationLink.href = `#${ruleCopy.id}`;
	if (setTextContent) {
		const name = ruleCopy.querySelector(`tr#nameData td input`).value;
		nextLocationLink.textContent = name;
	}
	return ruleTocCopy;
}

function addToCategoriesInRuleset(rule, category) {
	const rulesInCategory = categoriesToRulesInRuleset.get(category);
	if (rulesInCategory) {
		rulesInCategory.add(rule);
	} else {
		const rules = new Set();
		rules.add(rule);
		categoriesToRulesInRuleset.set(category, rules);
	}
}

function removeFromCategoriesInRuleset(rule, category) {
	const rulesInCategory = categoriesToRulesInRuleset.get(category);
	if (rulesInCategory && rulesInCategory.size === 1) {
		categoriesToRulesInRuleset.delete(category);
		return true;
	} else if (rulesInCategory && rulesInCategory.size !== 1) {
		rulesInCategory.delete(rule);
		return false;
	}
}

function removeRuleSectionFromCategory(rule, category, onlyCheckIfCategoryShouldBeRemoved) {
	const categoryToc = document.querySelector(`li[data-id="${category.id}"]`);
	if (removeFromCategoriesInRuleset(rule, category)) {
		removeCustomCategoryFromFilterOptions(category.dataset.name);
		categoriesContainer.removeChild(category);
		tocContainer.removeChild(categoryToc);
	} else if (!onlyCheckIfCategoryShouldBeRemoved) {
		const categoryContent = category.querySelector('div.categoryContent');
		categoryContent.removeChild(rule);

		const categoryTocChildren = categoryToc.querySelector('ul');
		const ruleToc = document.querySelector(`li[data-id="${rule.id}"]`);
		categoryTocChildren.removeChild(ruleToc);

		if (areFiltersActive()) {
			checkIfCategoryShouldBeHidden(category, categoryToc);
		}
	}
}

function addRuleSectionToCategory(rule, ruleToc, category, nextSibling, nextSiblingToc) {
	addToCategoriesInRuleset(rule, category);
	const categoryContent = category.querySelector('div.categoryContent');
	const categoryToc = document.querySelector(`li[data-id="${category.id}"] ul`);

	if (nextSibling && nextSiblingToc) {
		categoryContent.insertBefore(rule, nextSibling);
		categoryToc.insertBefore(ruleToc, nextSiblingToc);
	} else {
		categoryContent.appendChild(rule);
		categoryToc.appendChild(ruleToc);
	}
}

function createAndAddRuleCopy(rule, table, addOrRemoveRuleButton, selectRuleButton, currentCategory, destinationCategory) {
	if (rulesWithCopiesInRuleset.has(rule))
		return;

	addOrRemoveRuleButton.textContent = C.BUTTON_TEXT.REMOVE_FROM_RULESET;
	addOrRemoveRuleButton.className = 'clickedButton';
	addOrRemoveRuleButton.style.visibility = 'visible';

	const ruleCopy = createEditableRuleCopy(rule);
	const ruleTocCopy = createRuleTocCopy(rule, ruleCopy, false);
	addRuleSectionToCategory(ruleCopy, ruleTocCopy, destinationCategory, null, null);

	rulesWithCopiesInRuleset.add(rule);
	selectRuleButton.style.display = 'none';
	changeTableColor(table, true);

	if (areFiltersActive()) {
		unhighlightRule(ruleCopy.getElementsByTagName('tr'));
	}

	if (shouldHideAddedRules) {
		hideRule(rule, currentCategory, true);
		updateFilterResults(false, 1);
	}

	return ruleCopy;
}

function markRuleInDefaultCategoryRemoved(rule, table, addOrRemoveRuleButton, selectRuleButton) {
	addOrRemoveRuleButton.textContent = C.BUTTON_TEXT.ADD_TO_RULESET;
	addOrRemoveRuleButton.className = 'button';
	rulesWithCopiesInRuleset.delete(rule);
	selectRuleButton.style.display = '';
	changeTableColor(table, false);
}

function markRuleInDefaultCategoryAdded(rule) {
	const addOrRemoveRuleButton = rule.querySelector('button#addRuleButton');
	addOrRemoveRuleButton.textContent = C.BUTTON_TEXT.REMOVE_FROM_RULESET;
	addOrRemoveRuleButton.className = 'clickedButton';
	addOrRemoveRuleButton.style.visibility = 'visible';
	rulesWithCopiesInRuleset.add(rule);
	rule.querySelector('button#selectRuleButton').style.display = 'none';
	changeTableColor(rule.querySelector('table'), true);
}

function changeIdForRule(rule, originalRule) {
	rule.id = getNextIndexedName(rule.dataset.originalId);

	if (originalRule) {
		const inputName = rule.querySelector(`tr#nameData td input`);
		const originalInputName = originalRule.querySelector(`tr#nameData td input`);
		let originalName = originalInputName.value;
		if (!originalName)
			originalName = originalInputName.placeholder;
		const nextIndexedName = `${originalName}_${nameIndexes.get(rule.dataset.originalId)}`;
		inputName.placeholder = nextIndexedName;
		inputName.value = nextIndexedName;
	}
}

function getRulesetEditorCategory(categoryName) {
	let rulesetEditorCategory = document.querySelector(`div.category[data-name='${categoryName}']`);
	if (!rulesetEditorCategory) {
		return createCategory(categoryName, state === C.STATE.SELECT_RULES);
	}
	return rulesetEditorCategory;
}

function duplicateRule(rule) {
	const category = rule.closest('div.category');
	const ruleCopy = rule.cloneNode(true);
	changeIdForRule(ruleCopy, rule);
	const ruleTocCopy = createRuleTocCopy(rule, ruleCopy, true);
	const autocorrectEnabledData = ruleCopy.querySelector('tr#autocorrectEnabledData');
	if (autocorrectEnabledData) {
		const radioInputs = autocorrectEnabledData.querySelectorAll(`td input`);
		radioInputs.forEach(radioInput => radioInput.name = ruleCopy.id);
	}

	addRuleSectionToCategory(ruleCopy, ruleTocCopy, category, rule.nextElementSibling, document.querySelector(`li[data-id="${rule.id}"]`).nextElementSibling);
	updateFilterResults(true, 1);
}

function addOrRemoveRule(rule) {
	const addOrRemoveRuleButton = rule.querySelector('button#addRuleButton');
	const selectRuleButton = rule.querySelector('button#selectRuleButton');
	const action = addOrRemoveRuleButton.textContent;
	const table = rule.querySelector('table');
	const currentCategory = rule.closest('div.category');

	if (action === C.BUTTON_TEXT.ADD_TO_RULESET) {
		createAndAddRuleCopy(rule, table, addOrRemoveRuleButton, selectRuleButton, currentCategory, getRulesetEditorCategory(currentCategory.id));
	} else if (action === C.BUTTON_TEXT.REMOVE_FROM_RULESET) {
		if (state === C.STATE.SELECT_RULES) {
			markRuleInDefaultCategoryRemoved(rule, table, addOrRemoveRuleButton, selectRuleButton);

			// remove all rule copies
			const ruleCopies = Array.from(document.querySelectorAll(`div.rule[data-original-id="${rule.id}"]`));
			if (ruleCopies) {
				ruleCopies.forEach(ruleCopy => {
					if (ruleCopy === rule)
						return;

					const ruleCopyCategory = ruleCopy.closest('div.category');
					removeRuleSectionFromCategory(ruleCopy, ruleCopyCategory, false);
				});
			}
		} else if (state === C.STATE.EDIT_RULESET) {
			removeRuleSectionFromCategory(rule, currentCategory, false);
			updateFilterResults(false, 1);

			// check if the rule in the default category should be marked as removed
			const ruleCopies = document.querySelectorAll(`div.rule[data-original-id="${rule.dataset.originalId}"]`);
			if (ruleCopies && ruleCopies.length === 1) {
				const originalRule = document.querySelector(`div.rule[id="${rule.dataset.originalId}"]`);
				if (originalRule) {
					markRuleInDefaultCategoryRemoved(originalRule, originalRule.querySelector('table'), originalRule.querySelector('button#addRuleButton'), originalRule.querySelector('button#selectRuleButton'));
					if (shouldHideAddedRules) {
						originalRule.style.display = '';
						const originalRuleToc = document.querySelector(`li[data-id="${rule.dataset.originalId}"]`);
						originalRuleToc.style.display = '';
					}
				}
			}

			if (rulesWithCopiesInRuleset === null || rulesWithCopiesInRuleset.size === 0) {
				generateRulesetButton.style.display = 'none';
				document.getElementById('infoMessage').style.display = '';
			}
		}
	}
}

function getCurrentCategoriesNames() {
	const selectElement = document.createElement('select');
	const currentCategoryNames = new Set();
	categoriesToRulesInRuleset.forEach((rules, category) => {
		let categoryName = category.id;
		if (category.dataset.name !== undefined && category.dataset.name !== '') {
			categoryName = category.dataset.name;
		}
		if (currentCategoryNames.has(categoryName)) {
			return;
		}
		currentCategoryNames.add(categoryName);
		const optionElement = document.createElement('option');
		optionElement.value = categoryName;
		optionElement.text = categoryName;
		selectElement.appendChild(optionElement);
	});

	if (state === C.STATE.SELECT_RULES) {
		allDefaultCategories.forEach(category => {
			let categoryName = category.id;
			if (category.dataset.name !== undefined && category.dataset.name !== '') {
				categoryName = category.dataset.name;
			}
			if (currentCategoryNames.has(categoryName)) {
				return;
			}
			currentCategoryNames.add(categoryName);
			const optionElement = document.createElement('option');
			optionElement.value = categoryName;
			optionElement.text = categoryName;
			selectElement.appendChild(optionElement);
		});
	}

	selectElement.firstElementChild.selected = true;
	selectElement.setAttribute('id', 'modalExistingCategories');
	return selectElement;
}

function setInfoForCategoryTemplate(categoryName) {
	const newCategory = categoryTemplate.cloneNode(true);
	newCategory.id = getNextIndexedName(categoryName);
	newCategory.dataset.name = categoryName;
	const categoryHeadH2 = newCategory.querySelector('div.categoryHead').firstElementChild;
	categoryHeadH2.lastElementChild.value = categoryName;
	return newCategory;
}

function createCategory(categoryName, hide) {
	const newCategory = setInfoForCategoryTemplate(categoryName);
	const newCategoryToc = document.createElement('li');
	newCategoryToc.dataset.id = newCategory.id;
	const newCategoryLink = document.createElement('a');
	newCategoryLink.href = `#${newCategory.id}`;
	newCategoryLink.textContent = categoryName;
	const newCategoryTocChildren = document.createElement('ul');
	newCategoryToc.appendChild(newCategoryLink);
	newCategoryToc.appendChild(newCategoryTocChildren);

	if (hide) {
		newCategory.style.display = 'none';
		newCategoryToc.style.display = 'none';
	}

	tocContainer.prepend(newCategoryToc);
	categoriesContainer.prepend(newCategory);

	const editableInput = newCategory.querySelector('.editableCategoryName');
	if (!hide) {
		addCustomCategoryToFilterOptions(categoryName);
	}

	return newCategory;
}

function displayMoveToCategoryModal() {
	if (currentSelection.size === 0) {
		notify('No selected rules!');
		return;
	}

	const modalContent = modal.querySelector('div.modal-content');
	const existingCategoriesOption = modalContent.querySelector('div#moveToExistingCategories');
	const selectElement = getCurrentCategoriesNames();
	selectElement.addEventListener('onclick', function (event) {
		existingCategoriesOption.firstElementChild.click();
	});
	existingCategoriesOption.appendChild(selectElement);

	const modalTitle = modalContent.querySelector('h2');
	modalTitle.textContent = moveToCategoryButton.textContent;

	const displayCurrentSelectionElementList = document.createElement('ul');
	currentSelection.forEach(rule => {
		const listItem = document.createElement('li');
		let name;
		if (state === C.STATE.SELECT_RULES) {
			name = rule.querySelector('tr#nameData td').textContent;
		} else {
			name = rule.querySelector('tr#nameData td input').value;
		}
		const textNode = document.createTextNode(name);
		listItem.appendChild(textNode);
		displayCurrentSelectionElementList.appendChild(listItem);
	});

	displayCurrentSelectionElement.appendChild(displayCurrentSelectionElementList);
	newCategoryNameInput.value = '';
	modal.style.display = 'block';
}

function clickSelectButtonForRule(rule, action) {
	if (rule.style.display !== 'none') {
		if ((state === C.STATE.SELECT_RULES && !rulesWithCopiesInRuleset.has(rule)) || (state === C.STATE.EDIT_RULESET)) {
			const button = rule.querySelector('button#selectRuleButton');
			if (button.textContent === action) {
				button.click();
			}
		}
	}
}

function clearCurrentSelection() {
	currentSelection.forEach(rule => {
		const button = rule.querySelector('button#selectRuleButton');
		if (button.textContent === C.BUTTON_TEXT.UNSELECT) {
			button.click();
		}
	});
}

function closeModal(done, existingCategories) {
	existingCategories.removeChild(existingCategories.lastElementChild);
	displayCurrentSelectionElement.removeChild(displayCurrentSelectionElement.firstElementChild);
	addToExistingCategoryOption.click();
	modal.style.display = 'none';

	if (done) {
		clearCurrentSelection();
	}
}

function performModalAction(done) {
	const existingCategories = addToExistingCategoryOption.parentNode;
	if (done) {
		let destinationCategoryName;
		if (addToExistingCategoryOption.checked) {
			destinationCategoryName = existingCategories.lastElementChild.value;
		} else if (addToNewCategoryOption.checked) {
			if (newCategoryNameInput.value !== '') {
				destinationCategoryName = newCategoryNameInput.value;
			} else {
				destinationCategoryName = newCategoryNameInput.placeholder;
			}
		} else {
			destinationCategoryName = null;
		}

		let destinationCategory;
		if (destinationCategoryName !== null) {
			destinationCategory = getRulesetEditorCategory(destinationCategoryName);
		}

		const newSelection = new Set();
		currentSelection.forEach(currentRule => {
			const currentCategory = currentRule.closest('div.category');
			if ((destinationCategoryName === null) || (currentCategory.id !== destinationCategory.id)) {
				let ruleCopy = currentRule;

				if (state === C.STATE.SELECT_RULES) {
					if (destinationCategoryName === null) {
						destinationCategory = getRulesetEditorCategory(currentCategory.id);
					}

					ruleCopy = createAndAddRuleCopy(currentRule, currentRule.querySelector('table'), currentRule.querySelector('button#addRuleButton'), currentRule.querySelector('button#selectRuleButton'), currentCategory, destinationCategory);
					newSelection.add(ruleCopy);
				} else {
					const ruleTocCopy = document.querySelector(`li[data-id="${currentRule.id}"]`);
					addRuleSectionToCategory(ruleCopy, ruleTocCopy, destinationCategory, null, null);
					removeRuleSectionFromCategory(currentRule, currentCategory, true);
				}
			}
		});

		newSelection.forEach(rule => currentSelection.add(rule));
	}

	closeModal(done, existingCategories);
}

function setSelectionMode(setActive, display, visibility, cursor) {
	selectionMode = setActive ? C.SELECTION_MODE.ACTIVE : C.SELECTION_MODE.NOT_ACTIVE;

	moveToCategoryButton.style.display = display;
	allUnselectAllButton.style.display = display;
	if (state === C.STATE.SELECT_RULES) {
		allDefaultCategories.forEach(category => category.querySelector('button#categoryUnselectAll').style.display = display);
		allRulesInDefaultCategories.forEach(rule => {
			rule.querySelector('button#addRuleButton').style.visibility = visibility;
			if (!setActive || !rulesWithCopiesInRuleset.has(rule)) {
				rule.querySelector('table').style.cursor = cursor;
			}
		});
	} else {
		generateRulesetButton.style.display = setActive ? 'none' : '';
		removeRulesButton.style.display = display;
		categoriesToRulesInRuleset.forEach((rules, category) => {
			category.querySelector('button#categoryUnselectAll').style.display = display;
			rules.forEach(rule => {
				rule.querySelector('button#addRuleButton').style.visibility = visibility;
				rule.querySelector('button#duplicateRule').style.visibility = visibility;
				rule.querySelector('table').style.cursor = cursor;
			});
		});
	}
}

function escapeCommentForXML(value) {
	return value.replaceAll(/--/g, '&#x2D;&#x2D;');
}

function createParameterXML(isAutocorrectParameter, doc, parameterValue, parameterName) {
	let property;
	if (!isAutocorrectParameter) {
		property = doc.createElement('property');
	} else {
		property = doc.createElement('autocorrect-input');
	}

	property.setAttribute('key', parameterName);
	property.setAttribute('value', parameterValue);
	return property;
}

function createParameterInfoXML(doc, parametersInfo, parameterName) {
	const parameterInfo = parametersInfo.get(parameterName);
	const parameterInfoXML = doc.createComment(` ${escapeCommentForXML(parameterInfo)} `);
	return parameterInfoXML;
}

async function addParametersToRuleXML(parametersEntries, doc, ruleXML, originalRule, isAutocorrectParameter, shouldAddAllParameters, shouldCommentParameters, shouldAddParametersInfo, generateAnyway) {
	const parametersDefaultValues = computeParametersDefaultValues(originalRule, !isAutocorrectParameter ? 'tr#parametersValuesData div.parameterValueEntry' : 'tr#autocorrectParametersValuesData div.parameterValueEntry');
	let parametersInfo;
	if (shouldAddParametersInfo) {
		parametersInfo = computeParametersInfo(originalRule, !isAutocorrectParameter ? 'tr#parametersDescriptionData td pre' : 'tr#autocorrectParametersDescriptionData td pre')
	}

	let addParametersDelimitator = true;
	for (const parameterEntry of parametersEntries.children) {
		const toSplit = parameterEntry.firstElementChild.textContent;
		const splitText = toSplit.split(' ');
		let parameterName = splitText[0];
		if (toSplit.includes('[new]')) {
			parameterName = splitText[1];
		}

		const parameterInput = parameterEntry.lastElementChild;
		if (parameterInput.classList.contains('required')) {
			if (!generateAnyway) {
				const respose = await request('There is at least one mandatory parameter that has no value!\nWhen linting, the rules that have mandatory parameters with no values will not run.\n\nDo you want to generate the ruleset XML file anyway?');
				if (!respose) {
					parameterInput.focus();
					throw new Error('Interrupt generation');
				}

				generateAnyway = true;
			}
		}

		const parameterValue = parameterInput.value;
		if (!shouldAddAllParameters && parametersDefaultValues.get(parameterName) === parameterValue) {
			if (shouldCommentParameters) {
				const parameter = createParameterXML(isAutocorrectParameter, doc, parameterValue, parameterName);
				const serializer = new XMLSerializer();
				const parameterString = serializer.serializeToString(parameter);
				const commentedParameter = doc.createComment(` ${parameterString} `);
				if (addParametersDelimitator) {
					ruleXML.appendChild(doc.createComment(` ${isAutocorrectParameter ? 'Autocorrect ' : ''}Parameters: `));
					addParametersDelimitator = false;
				}
				if (shouldAddParametersInfo) {
					const parameterInfo = createParameterInfoXML(doc, parametersInfo, parameterName);
					ruleXML.appendChild(parameterInfo);
				}
				ruleXML.appendChild(commentedParameter);
			}

			continue;
		}

		if (addParametersDelimitator) {
			ruleXML.appendChild(doc.createComment(` ${isAutocorrectParameter ? 'Autocorrect ' : ''}Parameters: `));
			addParametersDelimitator = false;
		}
		if (shouldAddParametersInfo) {
			const parameterInfo = createParameterInfoXML(doc, parametersInfo, parameterName);
			ruleXML.appendChild(parameterInfo);
		}
		const parameter = createParameterXML(isAutocorrectParameter, doc, parameterValue, parameterName);
		ruleXML.appendChild(parameter);
	}

	return generateAnyway;
}

function addTagsToRuleXML(tags, doc, ruleXML) {
	for (const tagEntry of tags) {
		const tagKey = tagEntry.dataset.name;
		const tagValuesInputs = tagEntry.querySelectorAll('div.tagValue input');

		let tagValues = [];
		for (const tagValueInput of tagValuesInputs) {
			const tagValue = tagValueInput.value;
			if (!tagValue) {
				continue;
			}

			tagValues.push(tagValue);
		}

		if (tagValues.length > 0) {
			const tag = doc.createElement('tag');
			tag.setAttribute('key', tagKey);
			tag.setAttribute('value', tagValues.join(', '));
			ruleXML.appendChild(tag);
		}
	}
}

function download(data, filename, type) {
	if (!window['vscodeOpen']) {
		const file = new Blob([data], { type: type });
		saveAs(file, filename);
	} else {
		vscode.postMessage({
			type: 'NOTIFICATION',
			data: {
				eventType: 'GENERATE',
				context: {
					content: data
				}
			},
		});
	}
}

function deleteRuleTag(tagInput) {
	let tagKey = tagInput.value;
	if (!tagKey) {
		tagKey = tagInput.placeholder;
	}

	const tagEntries = document.querySelectorAll(`div#tagsInfo[data-name="${tagKey}"]`);
	for (const tagEntry of tagEntries) {
		const tagEntryContainer = tagEntry.parentNode;
		tagEntryContainer.removeChild(tagEntry);

		const otherTagEntry = tagEntryContainer.querySelector('div#tagsInfo');
		if (!otherTagEntry) {
			tagEntryContainer.closest('tr').style.display = "none";
		}
	}
}

function deleteAnnotation(annotation) {
	const annotationArea = annotation.parentNode;
	const input = annotation.lastElementChild;

	annotationArea.removeChild(annotation);
	if (input && input.tagName === 'INPUT') {
		deleteRuleTag(input);
	}
}

function addTagEntry(rule, nextIndexedName, showTagsData) {
	const tagsData = rule.querySelector('tr#tagsData');
	if (showTagsData) {
		tagsData.style.display = '';
	}

	const td = tagsData.querySelector('td');
	const tagEntryTemplate = td.firstElementChild;
	const tagEntry = tagEntryTemplate.cloneNode(true);
	tagEntry.querySelector('label').textContent = `${nextIndexedName} :`;
	tagEntry.style.display = '';
	tagEntry.id = 'tagsInfo';
	tagEntry.dataset.name = nextIndexedName;
	td.appendChild(tagEntry);
}

function isDuplicateTagKey(newTagKey, oldTagKey) {
	if (oldTagKey === newTagKey) {
		return false;
	}
	const rule = document.querySelector('div.rule');
	const tagKeys = rule.querySelectorAll('div#tagsInfo');
	for (const tagKey of tagKeys) {
		const tagKeyName = tagKey.dataset.name;
		if (tagKeyName !== newTagKey) {
			continue;
		}
		return true;
	}
	return false;
}

function isDuplicateRuleName(ruleName, input) {
	for (const [, rules] of categoriesToRulesInRuleset) {
		for (const rule of rules) {
			const ruleInput = rule.querySelector('tr#nameData input');
			if (ruleInput !== input && ((ruleInput.value && ruleInput.value === ruleName) || (!ruleInput.value && ruleInput.placeholder === ruleName)))
				return true;
		}
	}

	return false;
}

function unescapeXML(value) {
	return value.replaceAll(/(&#10;|&lt;|&gt;|&amp;|&quot;|&apos;)/g, (match) => {
		switch (match) {
			case '&#10;': return '\n';
			case '&lt;': return '<';
			case '&gt;': return '>';
			case '&amp;': return '&';
			case '&quot;': return '"';
			case '&apos;': return "'";
			default: return match;
		}
	});
}

function computeParametersInfo(originalRule, originalParametersDescriptionSelector) {
	const parametersInfo = new Map();
	const parametersDescription = originalRule.querySelector(originalParametersDescriptionSelector);
	if (!parametersDescription) {
		return null;
	}

	const parametersInfoArray = parametersDescription.innerHTML.replace(/<br\s*\/?>/gi, '\n').split('\n').filter(line => line.length > 0).map(line => line.replace(/<\/?i>/gi, ''));
	for (const parameterInfo of parametersInfoArray) {
		const parameterName = parameterInfo.split(' : ')[0];
		parametersInfo.set(parameterName, unescapeXML(parameterInfo));
	}

	return parametersInfo;
}

function computeParametersDefaultValues(originalRule, originalParametersSelector) {
	const defaultValues = new Map();
	const originalParameters = originalRule.querySelectorAll(originalParametersSelector);
	for (const originalParameter of originalParameters) {
		const originalParameterName = originalParameter.firstElementChild.textContent.split(' ')[0];
		defaultValues.set(originalParameterName, originalParameter.lastElementChild.value)
	}

	return defaultValues;
}

function attributeHasDefaultValue(attributeValue, originalRule, originalElementSelector) {
	const originalElement = originalRule.querySelector(originalElementSelector);
	return attributeValue === originalElement.value;
}

function isCheckedPreference(idPrefix) {
	const checkbox = document.getElementById(`${idPrefix}Preference`);
	return checkbox.checked;
}

function getCompleteRuleInfo(indent, id, name, title, severity, labels, description, descriptionAppend, parametersInfoArray, autocorrectParametersInfoArray) {
	const innerIndent = ' '.repeat(indent + C.XML_INDENTATION);
	const info = [
		['ID', id],
		['NAME', name],
		['TITLE', title],
		['SEVERITY', severity],
		...(!!labels ? [['LABEL', labels]] : []),
		['DESCRIPTION', `\n${innerIndent}${description.concat(descriptionAppend || '').replace(/\n/gi, `\n${innerIndent}`)}`],
	].filter(
		([, value]) => !!value
	);

	if (parametersInfoArray) {
		info.push([`PARAMETERS`, `\n${innerIndent}${Array.from(parametersInfoArray.values()).join(`\n${innerIndent}`)}`]);
	}
	if (autocorrectParametersInfoArray) {
		info.push([`AUTOCORRECT PARAMETERS`, `\n${innerIndent}${Array.from(autocorrectParametersInfoArray.values()).join(`\n${innerIndent}`)}`]);
	}

	return info.map(([key, value]) => `${' '.repeat(indent)}${key}: ${escapeCommentForXML(value)}`).join('\n');
}

function customFormatXML(xmlString, indentOfRuleAttributes) {
	const lines = xmlString.split('\n');
	const result = [];

	result.push(lines[0]);
	for (let i = 1; i < lines.length; i++) {
		const isCommentedNode = lines[i].trim().startsWith('<!-- <');
		const isComment = lines[i].trim().startsWith('<!--');

		if (isCommentedNode) {
			if ((lines[i - 1].trim().includes('<!-- Parameters: -->') || lines[i - 1].trim().includes('<!-- Autocorrect Parameters: -->')) && result.length > 0) {
				result.push('');
			}
			result.push(lines[i]);
		} else {
			if ((isComment && !lines[i - 1].trim().startsWith('<category') && (!(lines[i].trim().includes('<!-- Parameters: -->') || lines[i].trim().includes('<!-- Autocorrect Parameters: -->')) || !lines[i - 1].trim().startsWith('<rule')))
				|| (lines[i].trim().startsWith('<property') && lines[i - 1].trim().includes('<!-- Parameters: -->'))
				|| (lines[i].trim().startsWith('<autocorrect-input') && lines[i - 1].trim().includes('<!-- Autocorrect Parameters: -->'))) {
				result.push('');
			}

			if (lines[i].trim().startsWith('<rule ')) {
				const match = lines[i].trim().match(/^<(\w+)\s+([^>]+)>$/);
				const tagName = match[1];
				const attrs = match[2].trim().split(/\s+(?=[a-zA-Z_:][\w:.-]*=\")/);

				const firstLine = `<${tagName} ${attrs[0]}`;
				result.push(`${indentOfRuleAttributes.slice(0, -C.XML_INDENTATION)}${firstLine}`);

				if (attrs.length > 1) {
					const rest = attrs.slice(1).join(`\n${indentOfRuleAttributes}`);
					result.push(`${indentOfRuleAttributes}${rest}`);
				}

				if (!lines[i].trim().endsWith('/>')) {
					result.push(`${indentOfRuleAttributes.slice(0, -C.XML_INDENTATION)}>`);
				} else {
					result[result.length - 1] = result[result.length - 1].slice(0, -1);
					result.push(`${indentOfRuleAttributes.slice(0, -C.XML_INDENTATION)}>`);
					result.push(`${indentOfRuleAttributes.slice(0, -C.XML_INDENTATION)}</rule>`);
				}

				if (lines[i + 1].trim().startsWith('<rule')) {
					result.push('');
				}
				continue;
			}

			result.push(lines[i]);

			if (lines[i].trim().startsWith('</rule') && (i + 1 > lines.length - 1 || !lines[i + 1].trim().startsWith('<!--'))) {
				result.push('');
			}
		}
	}

	return result.join('\n');
}

window.onclick = function (event) {
	if (event.target === modal) {
		performModalAction(false);
	}
}

window.onscroll = function () {
	scrollFunction();
};

document.body.addEventListener('click', hideConfigurationAreas);

categoriesContainer.addEventListener('click', function (event) {
	if (event.target.tagName === 'BUTTON') {
		const rule = event.target.closest('div.rule');
		const id = event.target.id;

		if (id === 'duplicateRule') {
			duplicateRule(rule);
		}

		if (id === 'addRuleButton') {
			addOrRemoveRule(rule);
		}

		if (id === 'categorySelectAll') {
			const category = event.target.closest('div.category');
			const rules = category.querySelectorAll('div.rule');
			rules.forEach(rule => {
				clickSelectButtonForRule(rule, C.BUTTON_TEXT.SELECT);
			});
		}

		if (id === 'categoryUnselectAll') {
			const category = event.target.closest('div.category');
			const rules = category.querySelectorAll('div.rule');
			rules.forEach(rule => {
				clickSelectButtonForRule(rule, C.BUTTON_TEXT.UNSELECT);
			});
		}

		if (id === 'selectRuleButton') {
			const rule = event.target.closest('div.rule');
			const table = rule.querySelector('table');
			if (event.target.textContent === C.BUTTON_TEXT.SELECT) {
				currentSelection.add(rule);
				event.target.textContent = C.BUTTON_TEXT.UNSELECT;
				event.target.className = 'clickedButton';
				table.className = 'selected';

				if (selectionMode === C.SELECTION_MODE.NOT_ACTIVE) {
					setSelectionMode(true, '', 'hidden', 'pointer');
				}
			} else {
				currentSelection.delete(rule);
				event.target.textContent = C.BUTTON_TEXT.SELECT;
				event.target.className = 'button';
				table.className = '';

				if (currentSelection.size === 0) {
					setSelectionMode(false, 'none', 'visible', '');
				}
			}
		}

		if (id === 'descriptionAppendButton') {
			const descriptionAppendButton = event.target;
			const tbody = descriptionAppendButton.closest('tbody');
			const descriptionAppend = tbody.querySelector('tr#descriptionAppendData');

			if (descriptionAppend.style.display !== 'none') {
				descriptionAppend.style.display = 'none';
				descriptionAppendButton.title = C.TOOLTIP.SHOW_DESCRIPTION_APPEND;
				descriptionAppendButton.innerHTML = C.BUTTON_TEXT.SHOW_DESCRIPTION_APPEND;
			} else {
				descriptionAppend.style.display = '';
				descriptionAppendButton.title = C.TOOLTIP.HIDE_DESCRIPTION_APPEND;
				descriptionAppendButton.innerHTML = C.BUTTON_TEXT.HIDE_DESCRIPTION_APPEND;
			}
		}

		if (id === 'addTagValue') {
			const addTagValueButton = event.target;
			const tagValuesContainer = addTagValueButton.parentNode;
			let tagValueContainer = addTagValueButton.previousElementSibling;
			let shouldAddNewValue = true;

			while (tagValueContainer && tagValueContainer.className === 'tagValue') {
				if (tagValueContainer.firstElementChild.value === '') {
					tagValueContainer.firstElementChild.focus();
					shouldAddNewValue = false;
					break;
				}

				tagValueContainer = tagValueContainer.previousElementSibling;
			}

			if (shouldAddNewValue) {
				const tagValueContainerCopy = addTagValueButton.previousElementSibling.cloneNode(true);
				tagValueContainerCopy.firstElementChild.value = '';
				const autocompleteItems = tagValueContainerCopy.querySelector('div.autocomplete-items');
				if (autocompleteItems) {
					tagValueContainerCopy.removeChild(autocompleteItems);
				}
				tagValuesContainer.insertBefore(tagValueContainerCopy, addTagValueButton);
			}
		}
	}

	if (event.target.tagName === 'INPUT') {
		event.target.select();
		if (event.target.className === 'editableCategoryName' && event.target.value.length > 0) {
			event.target.size = event.target.value.length;

			const category = event.target.closest('div.category');
			if (category !== null) {
				const categoryName = category.dataset.name;
				editingCustomCategoryName = categoryName;
			}
		} else if (event.target.className === 'inputText' && event.target.closest('tr').id === 'nameData') {
			editingRuleName = event.target.value;
			if (!editingRuleName)
				editingRuleName = event.target.placeholder;
		}
	}

	if (selectionMode === C.SELECTION_MODE.ACTIVE) {
		const table = event.target.closest('table');
		if (table) {
			const rule = table.closest('div.rule');
			if ((state === C.STATE.SELECT_RULES && !rulesWithCopiesInRuleset.has(rule)) || (state === C.STATE.EDIT_RULESET)) {
				const button = rule.querySelector('button#selectRuleButton');
				button.click();
			}
		}
	}
});

categoriesContainer.addEventListener('focusout', function (event) {
	if (event.target.tagName === 'INPUT') {
		if (event.target.parentNode.className === 'tagValue') {
			const tagValueContainer = event.target.parentNode;
			const value = event.target.value;

			if (value === '' && (tagValueContainer.nextElementSibling.className === 'tagValue' || tagValueContainer.previousElementSibling.className === 'tagValue')) {
				tagValueContainer.parentNode.removeChild(tagValueContainer);
			}

			if (value !== '') {
				const label = tagValueContainer.parentNode.firstElementChild;
				const suggestionsSet = tagSuggestions.get(label.textContent);
				if (suggestionsSet) {
					suggestionsSet.add(value);
				} else {
					const suggestionsSet = new Set();
					suggestionsSet.add(value);
					tagSuggestions.set(label.textContent, suggestionsSet);
				}
			}
		} else if (event.target.className === 'inputText' && event.target.closest('tr').id === 'nameData') {
			const input = event.target;
			let newRuleName = input.value;
			if (newRuleName === '') {
				newRuleName = input.placeholder;
			}

			if (newRuleName === editingRuleName)
				return;

			if (isDuplicateRuleName(newRuleName, input)) {
				notify(`Duplicate rule name '${newRuleName}'! Please set a new name!`);
				input.value = editingRuleName;
				const ruleId = input.closest('div.rule').id;
				const ruleToc = document.querySelector(`li[data-id="${ruleId}"]`);
				ruleToc.firstElementChild.textContent = editingRuleName;
				return;
			}

			editingRuleName = undefined;
		} else if (event.target.className === 'editableCategoryName') {
			const category = event.target.closest('div.category');
			if (category !== null) {
				const categoryName = category.dataset.name;
				if (editingCustomCategoryName !== categoryName) {
					removeCustomCategoryFromFilterOptions(editingCustomCategoryName);
					addCustomCategoryToFilterOptions(categoryName);
				}
			}
			editingCustomCategoryName = undefined;
		}
	}
});

categoriesContainer.addEventListener('input', function (event) {
	if (event.target.tagName === 'INPUT') {
		if (event.target.className === 'editableCategoryName') {
			const category = event.target.closest('div.category');
			const categoryToc = document.querySelector(`li[data-id="${category.id}"]`);
			let newName = event.target.value;
			if (newName !== '') {
				categoryToc.firstElementChild.textContent = newName;
				category.dataset.name = newName;
			} else {
				categoryToc.firstElementChild.textContent = 'UNNAMED CATEGORY';
				category.dataset.name = newName;
				newName = 'UNNAMED CATEGORY';
			}
			event.target.size = newName.length;

		} else {
			const tr = event.target.closest('tr');
			const inputValue = event.target.value;
			if (tr.id === 'nameData') {
				const ruleId = event.target.closest('div.rule').id;
				const ruleToc = document.querySelector(`li[data-id="${ruleId}"]`);
				if (inputValue !== '') {
					ruleToc.firstElementChild.textContent = inputValue;
				} else {
					ruleToc.firstElementChild.textContent = event.target.placeholder;
				}
			} else if (tr.id === 'tagsData') {
				closeOtherSuggestions();

				const input = event.target;
				const label = input.parentNode.parentNode.firstElementChild;
				if (!inputValue || !tagSuggestions.has(label.textContent) || tagSuggestions.get(label.textContent).size === 0) {
					return false;
				}

				const divElement = document.createElement("DIV");
				divElement.setAttribute("class", "autocomplete-items");
				divElement.dataset.focus = "-1";
				input.parentNode.appendChild(divElement);

				const suggestions = tagSuggestions.get(label.textContent);
				for (const item of suggestions) {
					if (item.slice(0, inputValue.length).toUpperCase() === inputValue.toUpperCase()) {
						const entry = document.createElement("DIV");
						entry.innerHTML = "<strong>" + item.slice(0, inputValue.length) + "</strong>";
						entry.innerHTML += item.slice(inputValue.length);
						entry.innerHTML += "<input type='hidden' value='" + item + "'>";
						entry.addEventListener("click", function (e) {
							const hiddenInput = e.target.querySelector('input');
							if (hiddenInput) {
								input.value = hiddenInput.value;
							}
							closeOtherSuggestions();
						});
						divElement.appendChild(entry);
					}
				}
			} else if (tr.id === 'parametersValuesData' || tr.id === 'autocorrectParametersValuesData') {
				if (event.target.classList.contains('required')) {
					if (inputValue !== '') {
						event.target.classList.remove('required');
					}
				} else if (event.target.required && event.target.value === '') {
					event.target.classList.add('required');
				}
			}
		}
	} else if (event.target.tagName === 'TEXTAREA') {
		adjustTextareaHeight(event.target);
	}
});

categoriesContainer.addEventListener('keydown', function (event) {
	if (event.target.tagName === 'INPUT') {
		const tr = event.target.closest('tr');
		if (tr && tr.id === 'tagsData') {
			const input = event.target;
			const list = input.parentNode.querySelector('div.autocomplete-items');
			if (!list) {
				return;
			}

			const listItems = list.querySelectorAll('div');
			if (!listItems || listItems.length === 0) {
				return;
			}

			let currentFocus = parseInt(list.dataset.focus);
			if (event.key === 'ArrowDown') {
				event.preventDefault();
				currentFocus++;
				highlightListItem(listItems, currentFocus, list);
			} else if (event.key === 'ArrowUp') {
				event.preventDefault();
				currentFocus--;
				highlightListItem(listItems, currentFocus, list);
			} else if (event.key === 'Enter') {
				if (currentFocus > -1) {
					listItems[currentFocus].click();
				}
			}
		}
	}
});

document.getElementById("editConfigurationDiv").addEventListener('click', (event) => {
	if (event.target.tagName === 'BUTTON') {
		if (event.target.id === 'addAnnotation') {
			const annotationModel = document.getElementById('annotationModel');
			const newAnnotation = annotationModel.cloneNode(true);
			newAnnotation.id = "";
			newAnnotation.style.display = "";
			const nextIndexedName = getNextIndexedName('Annotation Name');
			const newAnnotationName = newAnnotation.querySelector('input');
			newAnnotationName.placeholder = nextIndexedName;
			newAnnotationName.value = nextIndexedName;
			event.target.parentNode.insertBefore(newAnnotation, annotationModel);
		}
	} else if (event.target.id === 'annotationDelete') {
		deleteAnnotation(event.target.parentNode);
	} else if (event.target.tagName === 'INPUT') {
		if (event.target.name === 'name') {
			editingAnnotationName = event.target.value;
			if (!editingAnnotationName)
				editingAnnotationName = event.target.placeholder;
		} else if (event.target.type === 'checkbox') {
			if (event.target.name === 'details') {
				const annotationArea = event.target.parentNode;
				const checkboxes = annotationArea.querySelectorAll('input[type="checkbox"]');
				let canUncheckDefault = false;

				for (const checkbox of checkboxes) {
					if (checkbox.name !== 'details' && checkbox.checked) {
						canUncheckDefault = true;
					}
				}

				if (!canUncheckDefault && !event.target.checked) {
					event.target.checked = true;
				}
			} else {
				const annotationArea = event.target.parentNode;
				const checkboxes = annotationArea.querySelectorAll('input[type="checkbox"]');
				let shouldCheckDefault = true;
				let defaultCheckbox;
				for (const checkbox of checkboxes) {
					if (checkbox.checked) {
						shouldCheckDefault = false;
					}
					if (checkbox.name === 'details') {
						defaultCheckbox = checkbox;
					}
				}

				if (shouldCheckDefault) {
					defaultCheckbox.checked = true;
				}
			}
		} else if (event.target.type === 'text' && event.target.className !== 'annotationInput') {
			editingTagKeyName = event.target.value;
			if (!editingTagKeyName)
				editingTagKeyName = event.target.placeholder;
		}
	}
});

document.getElementById("editConfigurationDiv").addEventListener('focusout', (event) => {
	if (event.target.tagName === 'INPUT') {
		if (event.target.name === 'name') {
			let annotationName = event.target.value;
			if (!annotationName)
				annotationName = event.target.placeholder;

			if (editingAnnotationName !== annotationName) {
				const annotationAreas = document.querySelectorAll('div.annotationArea');
				if (annotationAreas && annotationAreas.length > 1) {
					for (const annotationArea of annotationAreas) {
						if (annotationArea.id === 'annotationModel')
							continue;

						const annotationNameInput = annotationArea.querySelector('input[name="name"]');
						if (annotationNameInput === event.target)
							continue;

						let currentAnnotationName = annotationNameInput.value;
						if (!currentAnnotationName)
							currentAnnotationName = annotationNameInput.placeholder;

						if (currentAnnotationName === annotationName) {
							notify(`Annotation name '${currentAnnotationName}' is duplicated! Please set a new annotation name!`);
							event.target.value = editingAnnotationName;
							break;
						}
					}
				}
			}

			editingAnnotationName = undefined;
		} else if (event.target.type === 'text' && event.target.className !== 'annotationInput') {
			const input = event.target;
			let newTagKey = input.value;
			if (newTagKey === '')
				newTagKey = input.placeholder;
	
			if (isDuplicateTagKey(newTagKey, editingTagKeyName)) {
				notify(`Duplicate tag key '${newTagKey}'! Please set a new tag key!`);
				input.value = editingTagKeyName;
				setTimeout(() => {
					generateRulesetButton.disabled = false;
				}, 300);
				return;
			}
	
			const tagEntries = document.querySelectorAll(`div#tagsInfo[data-name='${editingTagKeyName}']`);
			for (const tagEntry of tagEntries) {
				tagEntry.dataset.name = newTagKey;
				tagEntry.firstElementChild.textContent = `${newTagKey} : `;
			}

			editingTagKeyName = undefined;
		}
	}
});

document.getElementById("editConfigurationDiv").addEventListener('change', (event) => {
	if (event.target.tagName === 'SELECT') {
		if (event.target.id === 'annotationType') {
			const value = event.target.value;
			const typeSelect = event.target;
			const annotation = typeSelect.parentNode;
			let fieldName;
			let fieldValueSelect;
			let dvtShowInOptionDisplay = 'hidden';

			switch (value) {
				case C.ANNOTATION_TYPE.FS.NAME:
					fieldName = C.ANNOTATION_TYPE.FS.FIELD_NAME + ': ';
					fieldValueSelect = createAnnotationSelect(C.ANNOTATION_TYPE.FS.FIELD_VALUES);
					break;
				case C.ANNOTATION_TYPE.GIT.NAME:
					fieldName = C.ANNOTATION_TYPE.GIT.FIELD_NAME + ': ';
					fieldValueSelect = createAnnotationSelect(C.ANNOTATION_TYPE.GIT.FIELD_VALUES);
					break;
				case C.ANNOTATION_TYPE.P4.NAME:
					fieldName = C.ANNOTATION_TYPE.P4.FIELD_NAME + ': ';
					fieldValueSelect = createAnnotationSelect(C.ANNOTATION_TYPE.P4.FIELD_VALUES);
					break;
				case C.ANNOTATION_TYPE.SVN.NAME:
					fieldName = C.ANNOTATION_TYPE.SVN.FIELD_NAME + ': ';
					fieldValueSelect = createAnnotationSelect(C.ANNOTATION_TYPE.SVN.FIELD_VALUES);
					break;
				case C.ANNOTATION_TYPE.CC.NAME:
					fieldName = C.ANNOTATION_TYPE.CC.FIELD_NAME + ': ';
					fieldValueSelect = createAnnotationSelect(C.ANNOTATION_TYPE.CC.FIELD_VALUES);
					break;
				case C.ANNOTATION_TYPE.TAG.NAME:
					fieldName = C.ANNOTATION_TYPE.TAG.FIELD_NAMES + ': ';
					fieldValueSelect = document.createElement('input');
					let nextIndexedName = getNextIndexedName('Tag Key');
					while (isDuplicateTagKey(nextIndexedName)) {
						nextIndexedName = getNextIndexedName('Tag Key');
					}
					fieldValueSelect.placeholder = nextIndexedName;
					fieldValueSelect.name = nextIndexedName;
					dvtShowInOptionDisplay = 'visible';

					categoriesToRulesInRuleset.forEach((rules, category) => {
						rules.forEach(rule => {
							addTagEntry(rule, nextIndexedName, true);
						});
					});
					allRulesInDefaultCategories.forEach(rule => {
						addTagEntry(rule, nextIndexedName, false);
					});
					break;
				case C.ANNOTATION_TYPE.LABEL.NAME:
					fieldName = '';
					dvtShowInOptionDisplay = 'visible';
					break;
			}

			annotation.querySelector("div#dvtShowInOption").style.visibility = dvtShowInOptionDisplay;
			const fieldLabel = typeSelect.nextElementSibling;
			fieldLabel.textContent = fieldName;
			fieldLabel.style.display = '';
			const oldFieldValue = fieldLabel.nextElementSibling;
			if (oldFieldValue) {
				if (oldFieldValue.tagName === 'INPUT') {
					deleteRuleTag(oldFieldValue);
				}
				annotation.removeChild(oldFieldValue);
			}

			if (fieldValueSelect) {
				annotation.appendChild(fieldValueSelect);
			}
		}
	}
});

document.getElementById("editConfigurationDiv").addEventListener('input', (event) => {
	if (event.target.tagName === 'INPUT' && event.target.type === 'text' && event.target.className !== 'annotationInput') {
		const input = event.target;
		let newTagKey = input.value;
		if (newTagKey === '') {
			newTagKey = input.placeholder;
		}

		if (isDuplicateTagKey(newTagKey, input.name)) {
			generateRulesetButton.disabled = true;
		} else {
			generateRulesetButton.disabled = false;
		}
	}
});

editRulesetStateButton.addEventListener('click', () => {
	if (state === C.STATE.SELECT_RULES) {
		editRulesetState();
	}
});

selectRulesStateButton.addEventListener('click', () => {
	if (state === C.STATE.EDIT_RULESET) {
		selectRulesState();
	}
});

removeRulesButton.addEventListener('click', () => {
	currentSelection.forEach(rule => {
		addOrRemoveRule(rule);
	});
	setSelectionMode(false, 'none', 'visible', '');
	generateRulesetButton.style.display = 'none';
	currentSelection.clear();
});

allUnselectAllButton.addEventListener('click', clearCurrentSelection);

hideAddedRulesButton.addEventListener('click', function () {
	hideAddedRules(!shouldHideAddedRules);
})

moveToCategoryButton.addEventListener('click', displayMoveToCategoryModal);

document.getElementById('doneChoosingCategories').addEventListener('click', function () {
	performModalAction(true);
});

document.getElementById('cancelChoosingCategories').addEventListener('click', function () {
	performModalAction(false);
});

document.getElementById('modalClose').addEventListener('click', function () {
	performModalAction(false);
});

document.querySelector('label[for="radioModalDefaultCategories"]').addEventListener('click', function () {
	this.parentNode.firstElementChild.click();
});

document.querySelector('label[for="modalExistingCategories"]').addEventListener('click', function () {
	this.parentNode.firstElementChild.click();
});

document.querySelector('label[for="modalNewCategoryName"]').addEventListener('click', function () {
	this.parentNode.firstElementChild.click();
});

addToNewCategoryOption.addEventListener('click', function () {
	disableOtherModalOptions('modalNewCategoryName');
});

document.getElementById('radioModalDefaultCategories').addEventListener('click', function () {
	disableOtherModalOptions('noid');
});

addToExistingCategoryOption.addEventListener('click', function () {
	disableOtherModalOptions('modalExistingCategories');
});

document.querySelector('button#helpButton').addEventListener('click', (event) => {
	toggleElement(document.getElementById('helpDiv'), event.target);
	document.getElementById('editConfigurationDiv').style.display = 'none';
	document.getElementById('editXMLPreferencesDiv').style.display = 'none';

	document.querySelector('button#editConfigurationButton').className = 'button';
	document.querySelector('button#editXMLPreferences').className = 'button';
});

document.querySelector('button#editConfigurationButton').addEventListener('click', (event) => {
	toggleElement(document.getElementById('editConfigurationDiv'), event.target);
	document.getElementById('helpDiv').style.display = 'none';
	document.getElementById('editXMLPreferencesDiv').style.display = 'none';

	document.querySelector('button#helpButton').className = 'button';
	document.querySelector('button#editXMLPreferences').className = 'button';
});

document.querySelector('button#editXMLPreferences').addEventListener('click', (event) => {
	toggleElement(document.getElementById('editXMLPreferencesDiv'), event.target);
	document.getElementById('helpDiv').style.display = 'none';
	document.getElementById('editConfigurationDiv').style.display = 'none';

	document.querySelector('button#helpButton').className = 'button';
	document.querySelector('button#editConfigurationButton').className = 'button';
});

document.querySelector('input#parametersPreference').addEventListener('click', (event) => {
	const commentedParametersPreference = document.querySelector('input#commentedParametersPreference');
	if (commentedParametersPreference.checked) {
		commentedParametersPreference.checked = false;
	}
});

document.querySelector('input#commentedParametersPreference').addEventListener('click', (event) => {
	const parametersPreference = document.querySelector('input#parametersPreference');
	if (parametersPreference.checked) {
		parametersPreference.checked = false;
	}
});

filterInput.addEventListener('keydown', function (event) {
	if (event.key === 'Enter') {
		parseInput();
		filterSearch();
	}
});

document.getElementById('allSelectAll').addEventListener('click', function () {
	const action = C.BUTTON_TEXT.SELECT;
	if (state === C.STATE.SELECT_RULES) {
		allRulesInDefaultCategories.forEach(rule => {
			clickSelectButtonForRule(rule, action);
		});
	} else {
		categoriesToRulesInRuleset.forEach((rules, category) => {
			rules.forEach(rule => {
				clickSelectButtonForRule(rule, action);
			});
		});
	}
});

document.addEventListener("click", function (e) {
	closeOtherSuggestions(e.target);
});

generateRulesetButton.addEventListener('click', async function () {
	const doc = document.implementation.createDocument(null, null);
	const pi = doc.createProcessingInstruction('xml', 'version="1.0" encoding="UTF-8"');
	doc.appendChild(pi);

	const ruleset = doc.createElement('ruleset');
	ruleset.setAttribute('version', '2');
	const rulesetTitle = editRulesetStateTitle.firstElementChild;
	if (rulesetTitle.value && rulesetTitle.value !== '') {
		ruleset.setAttribute('name', rulesetTitle.value);
	} else {
		ruleset.setAttribute('name', rulesetTitle.placeholder);
	}

	try {
		const configuration = doc.createElement('configuration');
		const annotationAreas = document.querySelectorAll('div.annotationArea');
		if (annotationAreas && annotationAreas.length > 1) {
			for (const annotationArea of annotationAreas) {
				if (annotationArea.id === 'annotationModel') {
					continue;
				}

				const annotationXML = doc.createElement('annotation');
				const annotationNameInput = annotationArea.querySelector('input[name="name"]');

				let annotationName;
				if (annotationNameInput.value) {
					annotationName = annotationNameInput.value;
				} else {
					annotationName = annotationNameInput.placeholder;
				}

				annotationXML.setAttribute('name', annotationName);

				const showInArray = [];
				const checkBoxes = annotationArea.querySelectorAll('input[type="checkbox"]');
				for (const checkbox of checkBoxes) {
					if (checkbox.checked) {
						showInArray.push(checkbox.name);
					}
				}
				if (!(showInArray.length === 1 && showInArray[0] === 'details') && showInArray.length >= 1) {
					annotationXML.setAttribute('show-in', showInArray.join(', '));
				}

				const annotationTypeSelect = annotationArea.querySelector('select#annotationType');
				if (!annotationTypeSelect.value) {
					throw new Error(`Annotation type for '${annotationName}' is invalid! Remove the annotation or select a valid type.`);
				}
				annotationXML.setAttribute('type', annotationTypeSelect.value);

				const fieldName = annotationArea.querySelector('label[for="annotationField"]').textContent.trim().slice(0, -1);
				let fieldValue = annotationArea.lastElementChild.value;
				if (!fieldValue) {
					fieldValue = annotationArea.lastElementChild.placeholder;
				}
				if (fieldName && fieldValue) {
					annotationXML.setAttribute(fieldName, fieldValue);
				}

				configuration.appendChild(annotationXML);
			}

			ruleset.appendChild(configuration);
		}

		let generateAnyway = false;
		const isParametersPreferenceChecked = isCheckedPreference('parameters');
		const isCommentedParametersPreferenceChecked = isCheckedPreference('commentedParameters');
		const isParametersInfoPreferenceChecked = isCheckedPreference('parametersInfo');
		const isCompleteRuleInfoPreferenceChecked = isCheckedPreference('completeRuleInfo');

		for (const [category, rules] of categoriesToRulesInRuleset) {
			let categoryName = category.dataset.name;
			if (categoryName === '') {
				throw new Error('Internal error!');
			}
			const categoryXML = doc.createElement('category');
			categoryXML.setAttribute('name', categoryName);

			for (const rule of rules) {
				const ruleXML = doc.createElement('rule');
				const id = rule.dataset.originalId;
				ruleXML.setAttribute('id', id);

				const name = rule.querySelector('tr#nameData td input');
				let ruleName = name.value;
				if (ruleName === '') {
					ruleName = name.placeholder;
				}

				const originalRule = document.querySelector(`div.rule[id="${rule.dataset.originalId}"]`);
				if (isCheckedPreference('name') || !attributeHasDefaultValue(ruleName, originalRule, 'tr#nameData td input')) {
					ruleXML.setAttribute('name', ruleName);
				}

				const title = rule.querySelector('tr#titleData td input');
				if (isCheckedPreference('title') || !attributeHasDefaultValue(title.value, originalRule, 'tr#titleData td input') || title.value === '') {
					ruleXML.setAttribute('title', title.value === '' ? title.placeholder : title.value);
				}

				const severity = rule.querySelector('tr#severityData td select');
				if ((C.COMPILATION_RULES.IDS.includes(id) && severity.value !== C.COMPILATION_RULES.DEFAULT_SEVERITY) ||
					(!C.COMPILATION_RULES.IDS.includes(id) && (isCheckedPreference('severity') || severity.value !== 'ERROR'))) {
					ruleXML.setAttribute('severity', severity.value);
				}

				const description = rule.querySelector('tr#descriptionData td textarea');
				if (isCheckedPreference('description') || !attributeHasDefaultValue(description.value, originalRule, 'tr#descriptionData td textarea')) {
					ruleXML.setAttribute('description', description.value);
				}

				const descriptionAppend = rule.querySelector('tr#descriptionAppendData td textarea');
				if (descriptionAppend && descriptionAppend.value !== '') {
					ruleXML.setAttribute('description-append', descriptionAppend.value);
				}

				const autocorrectEnabler = rule.querySelector('tr#autocorrectEnabledData td');
				if (autocorrectEnabler) {
					if ((autocorrectEnabler.firstElementChild.checked)) {
						ruleXML.setAttribute('autocorrect', 'on');
					}
				}

				const tags = rule.querySelectorAll('tr#tagsData td div#tagsInfo');
				if (tags) {
					addTagsToRuleXML(tags, doc, ruleXML);
				}

				const parameters = rule.querySelector('tr#parametersValuesData td');
				if (parameters) {
					generateAnyway = await addParametersToRuleXML(parameters, doc, ruleXML, originalRule, false, isParametersPreferenceChecked, isCommentedParametersPreferenceChecked, isParametersInfoPreferenceChecked, generateAnyway);
				}

				const autocorrectParameters = rule.querySelector('tr#autocorrectParametersValuesData td');
				if (autocorrectParameters) {
					addParametersToRuleXML(autocorrectParameters, doc, ruleXML, originalRule, true, isParametersPreferenceChecked, isCommentedParametersPreferenceChecked, isParametersInfoPreferenceChecked);
				}

				if (isCompleteRuleInfoPreferenceChecked) {
					const labels = rule.querySelector('tr#labelData td');
					const labelText = labels ? labels.textContent : '';
					const completerRuleInfo = getCompleteRuleInfo(3 * C.XML_INDENTATION, id, ruleName, title.value, severity.value, labelText, description.value, descriptionAppend.value, computeParametersInfo(originalRule, 'tr#parametersDescriptionData td pre'), computeParametersInfo(originalRule, 'tr#autocorrectParametersDescriptionData td pre'));
					categoryXML.appendChild(doc.createComment(`\n${completerRuleInfo}\n${' '.repeat(2 * C.XML_INDENTATION)}`));
				}
				categoryXML.appendChild(ruleXML);
			}

			ruleset.appendChild(categoryXML);
		}
	} catch (e) {
		if (e.message === 'Interrupt generation') {
			return;
		}
		notify(e.message);
		return;
	}

	doc.appendChild(ruleset);

	try {
		const serializer = new XMLSerializer();
		const xmlString = serializer.serializeToString(doc);
		const prettyXmlText = customFormatXML(xmlFormatter(xmlString, { indentation: ' '.repeat(C.XML_INDENTATION), collapseContent: false, lineSeparator: '\n' }), ' '.repeat(3 * C.XML_INDENTATION));
		download(prettyXmlText, 'verissimo_ruleset.xml', 'text/plain');
	} catch (e) {
		notify(e.message);
		return;
	}
});
