<#assign emptyAnnotations = []>
<#function nl2br content>
  <#return content?replace("\n","<br />") >
</#function>
<#function nl3br content>
    <#assign result = content?replace("^(\\w+) ", "<i>$1 </i>","rm")>
     <#return result?replace("\n","<br />") >
</#function>
<#macro compress_single_line>
    <#local captured><#nested></#local>
${ captured?replace("^\\s+|\\s+$|\\n|\\r", "", "rm") }
</#macro>
<#macro Logo>
<div style="margin-top: 60px;">
  <div style="display: inline-block;"><a href="http://eda.amiq.com" target="_new"><b>Verissimo Linter<sup>TM</sup><br></>Version ${all.version}</b></a></div>
  <div style="display: inline-block;float: right;">
    <a href="http://www.amiq.com" target="_new">
      <img width="91" height="45" alt="AMIQ EDA" src="data:image/png;base64,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" />
    </a>
  </div>
</div>
</#macro>
<#macro Toc children isInCustomRuleset>
    <#list children as child>
        <#if child.type == "category">
            <li
                <#if isInCustomRuleset>
                    data-id="${child.name?html}_c">
                        <a href="#${child.name?html}_c">${child.name?html}</a>
                <#else>
                    data-id="${child.name?html}" <#if custom??>style="display: none;"</#if>>
                        <a href="#${child.name?html}">${child.name?html}</a>
                </#if>
                <ul>
                    <@Toc children=child.children isInCustomRuleset=isInCustomRuleset/>
                </ul>
            </li>
        </#if>
        <#if child.type == "rule" && child.id != "NONEXISTENT_ID_ERROR">
            <li
                data-id="${child.id?html}<#if isInCustomRuleset>_r<#if child.rulesetId??>_${child.rulesetId}</#if></#if>">
                    <a href="#${child.id?html}<#if isInCustomRuleset>_r<#if child.rulesetId??>_${child.rulesetId}</#if></#if>">${child.name?html}</a>
            </li>
        </#if>
    </#list>
</#macro>
<#macro CategoriesFilter>
	<div class="tocHeader">
		<h2 class="tocEntry">Filter by Category</h2>
		<button title="Remove All" class="tocRemoveButton" id="categoriesRemoveAll">&times;</button>
	</div>
	<input type="text" id="categoriesFilterInput">
</#macro>
<#macro LabelsFilter>
	<div class="tocHeader">
		<h2 class="tocEntry">Filter by Labels</h2>
		<button title="Remove All" class="tocRemoveButton" id="labelsRemoveAll">&times;</button>
	</div>
	<input type="text" id="labelsFilterInput">
</#macro>
<#macro Content children xvm isInCustomRuleset annotations>
    <#list children as child>
        <#if child.type == "category">
            <div class="category"
                <#if isInCustomRuleset>
                    id="${child.name?html}_c" data-name="${child.name?html}"
                <#else>
                    id="${child.name?html}" data-name="" <#if custom??>style="display: none;"</#if>
                </#if>>
                <div class="categoryHead">
                    <h2>
                        <#if isInCustomRuleset>
                            <input class="editableCategoryName" type="text" placeholder="UNNAMED CATEGORY" value="${child.name?html}" autocomplete="off"/>
                        <#else>
                            <span>${child.name?html}</span>
                        </#if>
                    </h2>
                    <button class="button" id="categorySelectAll">Select Category</button>
                    <button class="button" id="categoryUnselectAll" style="display:none;">Clear Selection</button>
                    <#if child.description != "">
                        <h3>Description: ${nl2br(child.description?html)}</h3>
                    </#if>
                </div>
                <div class="categoryContent">
                    <@Content children=child.children xvm="${xvm}" isInCustomRuleset=isInCustomRuleset annotations=annotations/>
                </div>
            </div>
        <#elseif child.type == "rule">
            <div class="rule" id="${child.id?html}<#if isInCustomRuleset && child.id != "NONEXISTENT_ID_ERROR">_r<#if child.rulesetId??>_${child.rulesetId}</#if></#if>" data-original-id="${child.id?html}">
                <button class="button" id="selectRuleButton">Select</button>
                <#if isInCustomRuleset>
                    <button class="clickedButton" id="addRuleButton">Remove from Ruleset</button>
                <#else>
                    <button class="button" id="addRuleButton">Add to Ruleset</button>
                </#if>
                <button class="button" id="duplicateRule" <#if !isInCustomRuleset>style="visibility:hidden;"</#if>>Duplicate Rule</button>
                <table summary="${child.id?html}">
                    <tr id="nameData">
                        <th>Name</th>
                        <@compress_single_line>
                        <#if !isInCustomRuleset>
                        <td>
                            <b>${child.name?html}</b>
                        </td>
                        </#if>
                        <td <#if !isInCustomRuleset>style="display:none"</#if>>
                            <input class="inputText" type="text" placeholder="${child.name?html}" value="${child.name?html}" autocomplete="off">
                        </td>
                        </@compress_single_line>
                    </tr>
                    <tr>
                        <th>ID</th>
                        <td>${child.id?html}</td>
                    </tr>
                    <tr id="titleData">
                        <th>Title</th>
                        <@compress_single_line>
                        <#if !isInCustomRuleset>
                        <td>
                            ${child.title?html}
                        </td>
                        </#if>
                        <td <#if !isInCustomRuleset>style="display:none"</#if>>
                            <input class="inputText" type="text" placeholder="${child.title?html}" value="${child.title?html}" autocomplete="off">
                        </td>
                        </@compress_single_line>
                    </tr>
					<#if child.label??>
	                <tr id="labelData">
	                    <th>Label</th>
	                    <td><#list child.label as label>${label?html}<#sep>, </#sep></#list></td>
	                </tr>
            		</#if>
                    <#if child.date??>
                    <tr hidden>
                        <th>Date</th>
                        <td>${child.date?html}</td>
                    </tr>
                    </#if>
                    <#if child.version??>
                    <tr hidden>
                        <th>Version</th>
                        <td>${child.version?html}</td>
                    </tr>
                    </#if>
                    <tr id="severityData">
                        <th>Severity</th>
                        <@compress_single_line>
                        <#if !isInCustomRuleset>
                        <td>
                            ${child.severity?html}
                        </td>
                        </#if>
                        <td <#if !isInCustomRuleset>style="display:none"</#if>>
                            <select autocomplete="off"><#if child.id == "SYNTACTIC_PROBLEM" || child.id="SEMANTIC_PROBLEM" || child.id == "NON_STANDARD"><option <#if child.severity == "Severity of the compilation failures">selected="true"</#if> value="Severity of the compilation failures">Severity of the compilation failures</option></#if><option <#if child.severity == "ERROR">selected="true"</#if> value="ERROR">ERROR</option><option <#if child.severity == "WARNING">selected="true"</#if> value="WARNING">WARNING</option><option <#if child.severity == "INFO">selected="true"</#if> value="INFO">INFO</option></select>
                        </td>
                        </@compress_single_line>
                    </tr>
                    <tr id="descriptionData">
                        <@compress_single_line>
                        <th>
                            Description
                            <div class="centered" <#if !isInCustomRuleset>style="display:none;"</#if>>
                            <button class="button" id="descriptionAppendButton" title="<#if child.descriptionAppend?? && child.descriptionAppend != "">Hide<#else>Show</#if> description-append"><#if child.descriptionAppend?? && child.descriptionAppend != "">&#10697;<#else>&#x2610;</#if></button>
                            </div>
                        </th>
                        </@compress_single_line>
                        <#if !isInCustomRuleset>
                        <td>
                            <pre>${nl2br(child.description?html)}</pre>
                        </td>
                        </#if>
                        <td <#if !isInCustomRuleset>style="display:none"</#if>>
                            <textarea class="inputText" <#if isInCustomRuleset>data-adjusted="true"</#if> value="" autocomplete="off">${child.description?html}</textarea>
                        </td>
                    </tr>
                    <tr id="descriptionAppendData" <#if !isInCustomRuleset || !child.descriptionAppend?? || child.descriptionAppend == "" >style="display:none;"</#if>>
                        <th>Description Append</th>
                        <td>
                            <textarea class="inputText" value="" autocomplete="off"><#if child.descriptionAppend??>${child.descriptionAppend?html}</#if></textarea>
                        </td>
                    </tr>
                    <tr id="tagsData" <#if !child.tags?? && (!annotations?? || !annotations?has_content || !isInCustomRuleset)>style="display: none;"</#if>>
                        <th>Tags</th>
                        <td>
                            <@compress_single_line>
                            <div style="display:none">
                                <label class="parameterLabel"></label>
                                <div class="tagValue">
                                    <input type="text" placeholder="Enter tag value...">
                                </div>
                                <button class="button" id="addTagValue" title="Add new tag value">&#43;</button>
                            </div>
                            </@compress_single_line>
                            <#if child.tags??>
                            <@compress_single_line>
                            <#list child.tags as tags>
                                <div id="tagsInfo" data-name="${tags.tagKey}">
                                    <#if tags.tagKey??>
                                    <label class="parameterLabel">${tags.tagKey} :</label>
                                    </#if>
                                    <#if tags.tagValues??>
                                    <#list tags.tagValues as tagValue>
                                    <div class="tagValue">
                                    <input type="text" placeholder="Enter tag value..." value="${tagValue}">
                                    </div>
                                    </#list>
                                    </#if>
                                    <button class="button" id="addTagValue" title="Add new tag value">&#43;</button>
                                </div>
                            </#list>
                            </@compress_single_line>
                            </#if>
                            <#if annotations?? && (annotations?size > 0)>
                            <@compress_single_line>
                            <#list annotations as annotation>
                                <#if annotation.type == "TAG">
                                <#assign addAnnotation=true>
                                <#if child.tags??>
                                <#list child.tags as tags>
                                    <#if tags.tagKey == annotation.arg>
                                        <#assign addAnnotation=false>
                                        <#break>
                                    </#if>
                                </#list>
                                </#if>
                                <#if addAnnotation?? && addAnnotation>
                                <div id="tagsInfo" data-name="${annotation.arg}">
                                    <label class="parameterLabel">${annotation.arg} :</label>
                                    <div class="tagValue">
                                    <input type="text" placeholder="Enter tag value...">
                                    </div>
                                    <button class="button" id="addTagValue" title="Add new tag value">&#43;</button>
                                </div>
                                </#if>
                                </#if>
                            </#list>
                            </@compress_single_line>
                            </#if>
                        </td>
                    </tr>
                    <#if child.parameters != "" && child.parameterMap?exists>
                        <tr id="parametersDescriptionData">
                            <@compress_single_line>
                            <th>
                                <#if isInCustomRuleset>
                                Parameters Description
                                <#else>
                                Parameters
                                </#if>
                            </th>
                            </@compress_single_line>
                            <td><pre>${nl3br(child.parameters?html)}</pre></td>
                        </tr>
                        <tr id="parametersValuesData" <#if !isInCustomRuleset>style="display:none"</#if>>
                            <th>Parameters Values</th>
                            <td>
                                <#list child.parameterMap as parameterMapList>
                                    <div class="parameterValueEntry">
                                        <label class="parameterLabel<#if isInCustomRuleset && parameterMapList["rulesetOverriden"] == "false"> bold</#if>"><#if isInCustomRuleset && parameterMapList["rulesetOverriden"] == "false">[new] </#if>${parameterMapList["name"]} : </label>
                                        <#if parameterMapList["type"]??>
                                            <#if parameterMapList["type"] == "BOOLEAN">
                                                <select>
                                                    <option <#if (parameterMapList["userValue"]?? && (parameterMapList["userValue"] == "true" || (parameterMapList["userValue"] != "false" && parameterMapList["defaultValue"] == "true"))) || (!parameterMapList["userValue"]?? && parameterMapList["defaultValue"] == "true")>selected="selected"</#if> value="true">true</option>
                                                    <option <#if (parameterMapList["userValue"]?? && (parameterMapList["userValue"] == "false" || (parameterMapList["userValue"] != "true" && parameterMapList["defaultValue"] == "false"))) || (!parameterMapList["userValue"]?? && parameterMapList["defaultValue"] == "false")>selected="selected"</#if> value="false">false</option>
                                                </select>
                                            <#else>
                                                <input class="inputText <#if parameterMapList["required"] == "mandatory" && parameterMapList["defaultValue"] == '' && (!parameterMapList["userValue"]?? || parameterMapList["userValue"] == '')>required</#if>" type="text" style="flex:1" value="<#if parameterMapList["userValue"]??>${parameterMapList["userValue"]}<#else><#if parameterMapList["type"] != "INTEGER">${parameterMapList["defaultValue"].replace('xvm', xvm)}<#else>${parameterMapList["defaultValue"]}</#if></#if>" autocomplete="off" <#if parameterMapList["required"] == "mandatory">required</#if>/>
                                            </#if>
                                        <#else>
                                            <input class="inputText" type="text" style="flex:1" value="<#if parameterMapList["userValue"]??>${parameterMapList["userValue"]}<#else>${parameterMapList["defaultValue"].replace('xvm', xvm)}</#if>" autocomplete="off"/>
                                        </#if>
                                    </div>
                                </#list>
                            </td>
                        </tr>
                    </#if>
                    <#if child.isAutocorrectable?? && child.isAutocorrectable>
                        <tr id="autocorrectEnabledData" <#if !isInCustomRuleset>style="display:none"</#if>>
                            <th>Autocorrect</th>
                            <td>
                                <input type="radio" name="autocorrectEnabler${child.id?html}<#if isInCustomRuleset && child.id != "NONEXISTENT_ID_ERROR">_r<#if child.rulesetId??>_${child.rulesetId}</#if></#if>" <#if child.hasAutocorrectEnabled?? && child.hasAutocorrectEnabled>checked</#if> autocomplete="off" value="true"/>
                                <label for="autocorrectEnabled">On</label>
                                <input type="radio" name="autocorrectEnabler${child.id?html}<#if isInCustomRuleset && child.id != "NONEXISTENT_ID_ERROR">_r<#if child.rulesetId??>_${child.rulesetId}</#if></#if>" <#if (child.hasAutocorrectEnabled?? && !child.hasAutocorrectEnabled) || !isInCustomRuleset>checked</#if> autocomplete="off" value="false"/>
                                <label for="autocorrectDisabled">Off</label>
                            </td>
                        </tr>
                    </#if>
                    <#if child.autocorrectParameters != "">
                        <tr id="autocorrectParametersDescriptionData">
                            <@compress_single_line>
                            <th>
                                <#if isInCustomRuleset>
                                Autocorrect Parameters Description
                                <#else>
                                Autocorrect Parameters
                                </#if>
                            </th>
                            </@compress_single_line>
                            <td><pre>${nl3br(child.autocorrectParameters?html)}</pre></td>
                        </tr>
                        <tr id="autocorrectParametersValuesData" <#if !isInCustomRuleset>style="display:none"</#if>>
                            <th>Autocorrect Parameters Values</th>
                            <td>
                                <#list child.autocorrectParameterMap as autocorrectParameterMapList>
                                    <div class="parameterValueEntry">
                                        <label class="parameterLabel<#if isInCustomRuleset && autocorrectParameterMapList["rulesetOverriden"] == "false"> bold</#if>"><#if isInCustomRuleset && autocorrectParameterMapList["rulesetOverriden"] == "false">[new] </#if>${autocorrectParameterMapList["name"]} : </label>
                                        <#if autocorrectParameterMapList["type"]??>
                                            <#if autocorrectParameterMapList["type"] == "BOOLEAN">
                                                <select>
                                                    <option <#if (autocorrectParameterMapList["userValue"]?? && (autocorrectParameterMapList["userValue"] == "true" || (autocorrectParameterMapList["userValue"] != "false" && autocorrectParameterMapList["defaultValue"] == "true"))) || (!autocorrectParameterMapList["userValue"]?? && autocorrectParameterMapList["defaultValue"] == "true")>selected="selected"</#if> value="true">true</option>
                                                    <option <#if (autocorrectParameterMapList["userValue"]?? && (autocorrectParameterMapList["userValue"] == "false" || (autocorrectParameterMapList["userValue"] != "true" && autocorrectParameterMapList["defaultValue"] == "false"))) || (!autocorrectParameterMapList["userValue"]?? && autocorrectParameterMapList["defaultValue"] == "false")>selected="selected"</#if> value="false">false</option>
                                                </select>
                                            <#else>
                                                <input class="inputText" type="text" style="flex:1" value="<#if autocorrectParameterMapList["userValue"]??>${autocorrectParameterMapList["userValue"]}<#else><#if autocorrectParameterMapList["type"] != "INTEGER">${autocorrectParameterMapList["defaultValue"].replace('xvm', xvm)}<#else>${autocorrectParameterMapList["defaultValue"]}</#if></#if>" autocomplete="off"/>
                                            </#if>
                                        <#else>
                                            <input class="inputText" type="text" style="flex:1" value="<#if autocorrectParameterMapList["userValue"]??>${autocorrectParameterMapList["userValue"]}<#else>${autocorrectParameterMapList["defaultValue"].replace('xvm', xvm)}</#if>" autocomplete="off"/>
                                        </#if>
                                    </div>
                                </#list>
                            </td>
                        </tr>
                    </#if>
                </table>
            </div>
        </#if>
    </#list>
</#macro>
<#macro JSContent children>
    let categoryContent;
    let category;
    let nonexistentIdsInfo = "";
    <#list children as child>
        categoryContent = document.querySelector(`div.category[data-name="${child.name?html}"] div.categoryContent`);
        category = categoryContent.parentNode;
        if (!!category) {
            const rules = Array.from(categoryContent.children);
            rules.forEach(rule => {
            	if (rule.id !== "NONEXISTENT_ID_ERROR") {
		            addToCategoriesInRuleset(rule, category);
		 			adjustTextareaHeight(rule.querySelector('textarea'));
	            	const originalRule = document.querySelector(`div.rule[id='` + rule.dataset.originalId + `']`);
	            	if (!!originalRule) {
		                markRuleInDefaultCategoryAdded(originalRule);
	 				}
 				} else {
 					nonexistentIdsInfo = nonexistentIdsInfo.concat(rule.querySelector('textarea').value, '\n');
 					rule.parentNode.removeChild(rule);
 				}
            });

            categoriesFilterComponent.addOption({ text: category.dataset.name });
        }
    </#list>
    categoriesFilterComponent.refreshOptions(false);
    if (nonexistentIdsInfo !== "") {
    	notify(nonexistentIdsInfo);
    }
    const tagsInfo = document.querySelectorAll('tr#tagsData td div#tagsInfo');
    for (const tagInfo of tagsInfo) {
        const label = tagInfo.querySelector('label');
        const values = tagInfo.querySelectorAll('div.tagValue');
        if (!values || values.length === 0) {
            continue;
        }

        let valuesSet;
        if (tagSuggestions.has(label.textContent)) {
            valuesSet = tagSuggestions.get(label.textContent);
        } else {
            valuesSet = new Set();
        }

        for (const value of values) {
            const input = value.firstElementChild;
            valuesSet.add(input.value);
        }

        if (!tagSuggestions.has(label.textContent)) {
            tagSuggestions.set(label.textContent, valuesSet);
        }
    }
</#macro>
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN" "http://www.w3.org/TR/html4/loose.dtd">
<html>
    <head>
        <style type="text/css">
            <#include "${all.resources_path}/ruleset_editor.css">
        </style>
        <title><#if !custom??>${all.ruleset}<#else>${custom.ruleset}</#if></title>
    </head>
    <body oncontextmenu="return false;">
        <div id="container">
            <div id="header">
                <button class="<#if !custom??>clickedButton<#else>button</#if>" id="selectRulesState">Rule Select Mode</button>
                <button class="<#if custom??>clickedButton<#else>button</#if>" id="editRulesetState">Rule Edit Mode</button>
                <div id="titleHeader">
                    <h1 id="selectRulesStateTitle" <#if custom??>style="display: none;"</#if>>${all.ruleset}</h1>
                    <h1 id="editRulesetStateTitle" <#if !custom??>style="display: none;"</#if>>
                        <input class="editableCategoryName" type="text" placeholder="Custom Ruleset" value="<#if custom??>${custom.ruleset}<#else>Custom Ruleset</#if>" autocomplete="off"/>
                    </h1>
                </div>
                <button class="button" id="generateRulesetButton" <#if !custom??>style="display: none;"</#if>>Generate Ruleset</button>
                <button class="button" id="allUnselectAll" style="display:none;">Clear Selection</button>
                <button class="button" id="allSelectAll">Select All</button>
                <button class="button" id="showAddedRules" <#if custom??>style="display: none;"</#if>>Hide Added Rules</button>
                <button class="button" id="removeAllFromCustomRuleset" style="display:none;">Remove rules</button>
                <button class="button" id="moveToCategoryButton" style="display:none;"><#if custom??>Move<#else>Add</#if> to Category</button>
                <div id="modal" class="modal">
                    <div class="modal-content">
                        <span id="modalClose" class="close">&times;</span>
                        <h2 id="modalTitle" style="padding-top: 10px;padding-bottom: 20px; text-align:center"></h2>
                        <div class="displayElement" id="displayLabel">
                            <label for="currentSelectionElement">Currently selected rules:</label>
                        </div>
                        <div class="displayElement" id="currentSelectionElement">
                        </div>
                        <div id="addToDefaultCategories" style="padding-bottom: 5px; <#if custom??>display: none;</#if>">
                            <input type="radio" id="radioModalDefaultCategories" name="modalChooseCategory" autocomplete="off">
                            <label for="radioModalDefaultCategories">Use default category for each rule</label>
                        </div>
                        <div id="moveToExistingCategories" style="padding-bottom: 5px;">
                            <input type="radio" id="radioModalExistingCategories" name="modalChooseCategory" checked="checked" autocomplete="off">
                            <label for="modalExistingCategories">Choose from existing categories:</label>
                        </div>
                        <div id="moveToNewCategory" style="padding-bottom: 20px;">
                            <input type="radio" id="radioModalNewCategory" name="modalChooseCategory" autocomplete="off">
                            <label for="modalNewCategoryName">Choose new category:</label>
                            <input id="modalNewCategoryName" type="text" placeholder="UNNAMED CATEGORY" disabled="disabled" autocomplete="off"/>
                        </div>
                        <div style="text-align:center">
                            <button class="button" id="doneChoosingCategories" style="margin-bottom:10px">Done</button>
                            <button class="button" id="cancelChoosingCategories" style="margin-bottom:10px">Cancel</button>
                        </div>
                    </div>
                </div>
                <div class="wrap">
                    <div class="filter">
                        <button type="submit" class="button" id="helpButton">
                            ?
                        </button>

                        <input type="text" id="filterInput" class="filterTerm" name="search" placeholder="Filter checks..." autocomplete="off">
                        <div class="textBlock">
                            <span id="filterResults"></span>
                        </div>
                    </div>
                    <div id="helpDiv" class="configurationArea" style="display: none">
                        You can filter the list of checks using the following filtering syntax:<br>
                        <br>
                          ( name | title | description | parameter | parameter_description | since | autocorrect_parameter | autocorrect_parameter_description ): [-]( value | "value with whitespaces" )<br>
                          ( id | label | severity | parameter_name | autocorrect_parameter_name ): ( [-]value | "[-]value1, ... , [-]valueN" )<br>
                          free-from text<br>
                        <br>
                        Notes:<br>
                        <br>
                          Placing "-" character before any value will filter-out matching checks.<br>
                          Filtering based on free-form text results in a list of checks that contain all the words from that free-form text anywhere in their content.<br>
                          Combining multiple attribute-based filters together or with free-form text results in a list of checks that match each attribute-based filter AND all the free-form text words.<br>
                          Free-from text must not contain ":" character.<br>
                          Filtering is case-insensitive.<br>
                        <br>
                        Examples:<br>
                        <table class="filterExamples" id="filterExamples" style="width:100%">
                          <tr>
                            <td style="width:20%">id:UVM</td>
                            <td style="width:80%">show checks containing "UVM" in their ID</td>
                          </tr>
                          <tr>
                            <td style="width:20%">id:-SVTB</td>
                            <td style="width:80%">filter-out checks containing "SVTB" in their ID</td>
                          </tr>
                          <tr>
                            <td style="width:20%">description:"Check supports auto-correcting"</td>
                            <td style="width:80%">show checks containing "Check supports auto-correcting" in their description</td>
                          </tr>
                          <tr>
                            <td style="width:20%">title:deprecated parameter:comments</td>
                            <td style="width:80%">show checks containing "deprecated" in their title and "comments" in their parameter name or parameter description</td>
                          </tr>
                          <tr>
                            <td style="width:20%">id:"SVTB, -SVTB.1"</td>
                            <td style="width:80%">show checks containing "SVTB" in their ID but not "SVTB.1"</td>
                          </tr>
                          <tr>
                            <td style="width:20%">connect uvm_active_passive_enum title:-driver</td>
                            <td style="width:80%">show checks containing "connect" and "uvm_active_passive_enum" anywhere in their content and not containing "driver" in their title</td>
                          </tr>
                          <tr>
                            <td style="width:20%">since:2022-10-01</td>
                            <td style="width:80%">show checks that were created after 2022-10-01</td>
                          </tr>
                          <tr>
                            <td style="width:20%">since:-2022-10-01</td>
                            <td style="width:80%">show checks that were created before 2022-10-01</td>
                          </tr>
                          <tr>
						    <td style="width:20%">since:24.1.5</td>
						    <td style="width:80%">show checks that were added starting with version 24.1.5</td>
						  </tr>
						  <tr>
						    <td style="width:20%">since:-24.1.5</td>
						    <td style="width:80%">show checks that were added before version 24.1.5</td>
						  </tr>
                        </table>
                      </div>
                </div>
                <div class="wrap">
                    <div class="config">
                        <button class="button" id="editConfigurationButton" title="Edit the configuration of the ruleset" <#if !custom??>style="visibility: hidden;"</#if>>Configuration</button>
                        <p style="display: inline;">CONFIDENTIAL</p><p id="confidential" <#if custom??>style="display: none"<#else>style="display: inline"</#if>> - Copyright (C) 2005-2026 AMIQ EDA s.r.l.</p>
                        <button class="button" id="editXMLPreferences" title="Customize the generated XML using preferences" <#if !custom??>style="visibility: hidden;"</#if>>Generation Preferences</button>
                    </div>
                    <div id="areaWrapper" class="configurationArea" style="display: none">
                        <div id="editConfigurationDiv" style="display: none">
                            Edit the configuration of the ruleset.<br>
                            <button class="button" id="addAnnotation"><b>&#43;</b> Annotation</button>
                            <div id="annotationModel" class="configurationArea annotationArea" style="display: none">
                                <input class="annotationInput" placeholder="" value="" name="name" autocomplete="off"></input>
                                <span id="annotationDelete" class="close annotationClose" title="Delete Annotation">&times;</span>
                                <label for="annotationShowIn">Show in:</label>
                                <input type="checkbox" name="details" checked autocomplete="off">
                                <label for="detailsCheckbox">details</label>
                                <input type="checkbox" name="filters" autocomplete="off">
                                <label for="filtersCheckbox">filters</label>
                                <input type="checkbox" name="table" autocomplete="off">
                                <label for="tableCheckbox">table</label>
                                <div id="dvtShowInOption" style="display:inline;visibility:hidden;">
                                    <input type="checkbox" name="dvt" autocomplete="off">
                                    <label for="dvtCheckbox">dvt</label>
                                </div>
                                <label style="margin-left: 4rem;" for="annotationSelect">Type:</label>
                                <select id="annotationType" class="annotationSelect" autocomplete="off">
                                    <option value="" disabled selected>Select type...</option>
                                    <option value="FS">FS</option>
                                    <option value="GIT">GIT</option>
                                    <option value="P4">P4</option>
                                    <option value="SVN">SVN</option>
                                    <option value="CC">CC</option>
                                    <option value="TAG">TAG</option>
                                    <option value="LABEL">LABEL</option>
                                </select>
                                <label style="margin-left: 6rem;" for="annotationField"></label>
                            </div>
                            <#if custom?? && custom.annotations??>
                                <#list custom.annotations as annotation>
                                    <div class="configurationArea annotationArea">
                                    <input class="annotationInput" placeholder="${annotation.name?html}" value="${annotation.name?html}" name="name" autocomplete="off"></input>
                                    <span id="annotationDelete" class="close annotationClose" title="Delete Annotation">&times;</span>
                                    <label for="annotationShowIn">Show in:</label>
                                    <input type="checkbox" name="details" <#if annotation.locations?seq_contains("DETAILS")>checked</#if> autocomplete="off">
                                    <label for="detailsCheckbox">details</label>
                                    <input type="checkbox" name="filters" <#if annotation.locations?seq_contains("FILTERS")>checked</#if> autocomplete="off">
                                    <label for="filtersCheckbox">filters</label>
                                    <input type="checkbox" name="table" <#if annotation.locations?seq_contains("TABLE")>checked</#if> autocomplete="off">
                                    <label for="tableCheckbox">table</label>
                                    <div id="dvtShowInOption" style="display:inline;visibility:<#if annotation.type == "LABEL" || annotation.type == "TAG">visible<#else>hidden</#if>;">
                                        <input type="checkbox" name="dvt" <#if annotation.locations?seq_contains("DVT")>checked</#if> autocomplete="off">
                                        <label for="dvtCheckbox">dvt</label>
                                    </div>
                                    <label style="margin-left: 4rem;" for="annotationSelect">Type:</label>
                                    <select id="annotationType" class="annotationSelect" autocomplete="off">
                                        <option value="" disabled>Select type...</option>
                                        <option value="FS" <#if annotation.type == "FS">selected</#if>>FS</option>
                                        <option value="GIT" <#if annotation.type == "GIT">selected</#if>>GIT</option>
                                        <option value="P4" <#if annotation.type == "P4">selected</#if>>P4</option>
                                        <option value="SVN" <#if annotation.type == "SVN">selected</#if>>SVN</option>
                                        <option value="CC" <#if annotation.type == "CC">selected</#if>>CC</option>
                                        <option value="TAG" <#if annotation.type == "TAG">selected</#if>>TAG</option>
                                        <option value="LABEL" <#if annotation.type == "LABEL">selected</#if>>LABEL</option>
                                    </select>
                                    <label style="margin-left: 6rem;" for="annotationField">
                                    <#if annotation.type == "FS">
                                        fs-field: 
                                    <#elseif annotation.type == "GIT">
                                        git-field: 
                                    <#elseif annotation.type == "P4">
                                        p4-field: 
                                    <#elseif annotation.type == "SVN">
                                        svn-field: 
                                    <#elseif annotation.type == "CC">
                                        cc-field: 
                                    <#elseif annotation.type == "TAG">
                                        tag-key: 
                                    </#if>
                                    </label>
                                    <#if annotation.type == "FS">
                                        <select><option value="last-modified">last-modified</option></select>
                                    <#elseif annotation.type == "GIT">
                                        <select><option value="author" <#if annotation.arg == "author">selected</#if>>author</option><option value="author-mail" <#if annotation.arg == "author-mail">selected</#if>>author-mail</option><option value="author-time" <#if annotation.arg == "author-time">selected</#if>>author-time</option><option value="committer" <#if annotation.arg == "committer">selected</#if>>committer</option><option value="committer-mail" <#if annotation.arg == "committer-mail">selected</#if>>committer-mail</option><option value="committer-time" <#if annotation.arg == "committer-time">selected</#if>>committer-time</option><option value="SHA1" <#if annotation.arg == "sha1">selected</#if>>SHA1</option><option value="summary" <#if annotation.arg == "summary">selected</#if>>summary</option></select>
                                    <#elseif annotation.type == "P4">
                                        <select><option value="author" <#if annotation.arg == "author">selected</#if>>author</option><option value="revision-number" <#if annotation.arg == "revision-number">selected</#if>>revision-number</option><option value="date" <#if annotation.arg == "date">selected</#if>>date</option></select>
                                    <#elseif annotation.type == "SVN">
                                        <select><option value="author" <#if annotation.arg == "author">selected</#if>>author</option><option value="commit" <#if annotation.arg == "commit">selected</#if>>commit</option><option value="date" <#if annotation.arg == "date">selected</#if>>date</option></select>
                                    <#elseif annotation.type == "CC">
                                        <select><option value="username" <#if annotation.arg == "username">selected</#if>>username</option><option value="full-username" <#if annotation.arg == "full-username">selected</#if>>full-username</option><option value="date" <#if annotation.arg == "date">selected</#if>>date</option><option value="comment" <#if annotation.arg == "comment">selected</#if>>comment</option></select>
                                    <#elseif annotation.type == "TAG">
                                        <input placeholder="${annotation.arg}" value="${annotation.arg}" name="tag" autocomplete="off">
                                    </#if>
                                </div>
                                </#list>
                            </#if>
                        </div>
                        <div id="editXMLPreferencesDiv" style="display: none">
                            Select the preferences used for generating the ruleset XML file.<br><br>
                            By default, attributes for each rule are added to the ruleset only if they do not have the default value.<br>
                            You can choose to add attributes that have the default value for all rules by checking the attribute preferences.<br>
                            You can choose to add additional information for all rules by checking the information preferences.<br><br>
                            <table class="filterExamples" style="width:100%">
                                <tr>
                                    <td colspan="3">
                                    Attribute preferences:
                                    </td>
                                </tr>
                                <tr>
                                    <td><input id="namePreference" type="checkbox" name="name" autocomplete="off"></td>
                                    <td>name</td>
                                    <td><i>Include the name for all rules.</i></td>
                                </tr>
                                <tr>
                                    <td><input id="titlePreference" type="checkbox" name="title" autocomplete="off"></td>
                                    <td>title</td>
                                    <td><i>Include the title for all rules.</i></td>
                                </tr>
                                <tr>
                                    <td><input id="severityPreference" type="checkbox" name="severity" autocomplete="off"></td>
                                    <td>severity</td>
                                    <td><i>Include the severity for all rules.</i></td>
                                </tr>
                                <tr>
                                    <td><input id="descriptionPreference" type="checkbox" name="description" autocomplete="off"></td>
                                    <td>description</td>
                                    <td><i>Include the description for all rules.</i></td>
                                </tr>
                                <tr>
                                    <td><input id="parametersPreference" type="checkbox" name="parameters" autocomplete="off"></td>
                                    <td>parameters</td>
                                    <td><i>Include all the parameters for all rules.</i></td>
                                </tr>
                                <tr>
                                    <td><input id="commentedParametersPreference" type="checkbox" name="commented_parameters" autocomplete="off"></td>
                                    <td>commented_parameters</td>
                                    <td><i>Include the parameters that have default values for all rules, as comments.</i></td>
                                </tr>
                                <tr>
                                    <td colspan="3">
                                    </td>
                                </tr>
                                <tr>
                                    <td colspan="3">
                                    Information preferences:
                                    </td>
                                </tr>
                                <tr>
                                    <td><input id="completeRuleInfoPreference" type="checkbox" name="complete_rule_info" autocomplete="off"></td>
                                    <td>complete_rule_info</td>
                                    <td><i>Include all rule information as a comment.</i></td>
                                </tr>
                                <tr>
                                    <td><input id="parametersInfoPreference" type="checkbox" name="parameters_info" autocomplete="off"></td>
                                    <td>parameters_info</td>
                                    <td><i>Include the description for the already included parameters, as comments.</i></td>
                                </tr>
                            </table>
                        </div>
                    </div>
                    <button class="button" onclick="scrollToTopFunction()" id="scrollToTop" style="display: none;">Back to Top &uarr;</button>
                </div>
            </div>
            <div id="toc">
            	<div class="tocFilterContainer" id="tocCategoriesFilter">
            		<@CategoriesFilter/>
            	</div>
            	<div class="tocFilterContainer" id="tocLabelsFilter">
            		<@LabelsFilter/>
            	</div>
            	<div id="tocQuickNavigation" style="margin-top: 10px;">
	            	<h2 class="tocEntry" style="font-weight: bold;">Quick Navigation</h2>
	                <ul>
	                	<#if custom??>
	                    	<@Toc children=custom.children isInCustomRuleset=true/>
	                    </#if>
	                    <@Toc children=all.children isInCustomRuleset=false/>
	                </ul>
	            </div>
                <@Logo/>
                <p>Copyright (C) 2005-2026 AMIQ EDA s.r.l.</p>
            </div>
            <div id="content">
                <div id="infoMessage" style="display:none">
                    <p>⚠️ <b>The ruleset currently contains no rules</b>. To add rules, go to <b>Rule Select Mode</b> and select the ones you wish to add to the ruleset.</p>
                </div>
	            <#if custom??>
	                <@Content children=custom.children xvm="${all.xvm}" isInCustomRuleset=true annotations=custom.annotations!emptyAnnotations/>
                </#if>
                <@Content children=all.children xvm="${all.xvm}" isInCustomRuleset=false annotations=emptyAnnotations/>
            </div>
            <div id="footer">
                <p>Generated by <b>${all.user}</b> on ${all.date} using <a href="http://eda.amiq.com/" target="_new">Verissimo Linter<sup>TM</sup></a> Version ${all.version}</p>
                <p>Copyright (C) 2005-2026 AMIQ EDA s.r.l.</p>
            </div>
        </div>

        <script>
			function getLabelsFilterData() {
				var data = [];
				<#list all.labels as label>
				    data.push({
				        text : "${label}"
				    });
				</#list>
				return data;
			}
			
            <#include "${all.resources_path}/rulesetEditor.js">
        </script>
        <script type="text/javascript">
            window.onload = function () {
            	state = <#if custom??>C.STATE.EDIT_RULESET<#else>C.STATE.SELECT_RULES</#if>;

                <#if custom??>
                    <@JSContent custom.children/>
                <#else>
                </#if>

                filterSearch();

                if (!window['vscodeOpen']) {
                    var DARK_THEME_CSS = "body {background-color: #474645;} body, h1, h2, h3, h4, h5, h6, p, table, td, caption, th, ul, ol, dl, li, dd, dt {color: #e5e5e4;} th, td {border: 1px solid #3c3b40;} th {background-color: #3c3b40;} .categoryHead {border-top: 4px solid #474645; border-bottom: 1px solid #3c3b40;} .categoryHead a {color: #3c3b40;} #header {border-bottom: 1px dashed #3c3b40; background-color: #474645;} #toc {border-right: 1px dashed #3c3b40;} #footer {border-top: 1px dashed #3c3b40;} .filterTerm {border: 3px solid #3c3b40; background: #474645; color: #e5e5e4;} background: #3c3b40; color: #e5e5e4;} #helpDiv {border: 3px solid #3c3b40; color: #e5e5e4;} .textBlock {border: 3px solid #3c3b40; background-color: #474645; color: #e5e5e4;}";
                    var CLASSIC_THEME_CSS = "body, h1, h2, h3, h4, h5, h6, p, table, td, caption, th, ul, ol, dl, li, dd, dt {color: #333;} h1, h2, h3 {color: #333;} th, td {border: 1px solid #ccc;} th {background-color: #eee;} .categoryHead {border-top: 4px solid #fff; border-bottom: 1px solid #ccc;} .categoryHead a {color: #333;} #header {border-bottom: 1px dashed #ccc; background-color: white;} #toc {border-right: 1px dashed #ccc;} #footer {border-top: 1px dashed #ccc;} .filterTerm {border: 3px solid #eee; color: #333;} #helpDiv {border: 3px solid #eee; color: #333;} .textBlock {border: 3px solid #eee; background-color: white; color: #333;}";
                    var styleSheet = document.createElement('style');

                    var results = new RegExp('[\?&]' + "theme" + '=([^&#]*)').exec(window.location.search);
                    if (results == null) {
                        styleSheet.textContent = CLASSIC_THEME_CSS;
                        document.getElementsByTagName('head')[0].appendChild(styleSheet);
                        return;
                    }

                    if (results[1] === "css/dark-theme.css")
                        styleSheet.textContent = DARK_THEME_CSS;
                       else if (results[1] === "css/style.css")
                        styleSheet.textContent = CLASSIC_THEME_CSS;

                    document.getElementsByTagName('head')[0].appendChild(styleSheet);
                }
            }
        </script>
    </body>
</html>
