library ieee;

-- $Date: 2008-04-10 17:16:09 +0930 (Thu, 10 Apr 2008) $

use WORK.MATH_REAL.all;
use std.standard.all;
package MATH_COMPLEX is

  --
  -- Type Definitions
  --
  type COMPLEX is
  record
    RE : REAL;                          -- Real part
    IM : REAL;                          -- Imaginary part
  end record;

  subtype POSITIVE_REAL is REAL range 0.0 to REAL'high;

  subtype PRINCIPAL_VALUE is REAL range -MATH_PI to MATH_PI;

  type COMPLEX_POLAR is
  record
    MAG : POSITIVE_REAL;                -- Magnitude
    ARG : PRINCIPAL_VALUE;  -- Angle in radians; -MATH_PI is illegal
  end record;

  -- VHDL 93 --

	--complex vector is array of complex type
  	type COMPLEX_VECTOR is array (INTEGER range <>) of COMPLEX;

  	-- Purpose:
  	-- 		 Returns square root of Z
  	function CSQRT(Z: in COMPLEX ) return COMPLEX_VECTOR;

  -- End of VHDL 93 --

  --
  -- Constant Definitions
  --
  constant MATH_CBASE_1 : COMPLEX := COMPLEX'(1.0, 0.0);
  constant MATH_CBASE_J : COMPLEX := COMPLEX'(0.0, 1.0);
  constant MATH_CZERO   : COMPLEX := COMPLEX'(0.0, 0.0);


  --
  -- Overloaded equality and inequality operators for COMPLEX_POLAR
  -- (equality and inequality operators for COMPLEX are predefined)
  --

  function "=" (L : in COMPLEX_POLAR; R : in COMPLEX_POLAR) return BOOLEAN;
  -- Purpose:
  --         Returns TRUE if L is equal to R and returns FALSE otherwise
  -- Special values:
  --         COMPLEX_POLAR'(0.0, X) = COMPLEX_POLAR'(0.0, Y) returns TRUE
  --         regardless of the value of X and Y.
  -- Domain:
  --         L in COMPLEX_POLAR and L.ARG /= -MATH_PI
  --         R in COMPLEX_POLAR and R.ARG /= -MATH_PI
  -- Error conditions:
  --         Error if L.ARG = -MATH_PI
  --         Error if R.ARG = -MATH_PI
  -- Range:
  --         "="(L,R) is either TRUE or FALSE
  -- Notes:
  --         None

  function "/=" (L : in COMPLEX_POLAR; R : in COMPLEX_POLAR) return BOOLEAN;
  -- Purpose:
  --         Returns TRUE if L is not equal to R and returns FALSE
  --         otherwise
  -- Special values:
  --         COMPLEX_POLAR'(0.0, X) /= COMPLEX_POLAR'(0.0, Y) returns
  --         FALSE regardless of the value of X and Y.
  -- Domain:
  --         L in COMPLEX_POLAR and L.ARG /= -MATH_PI
  --         R in COMPLEX_POLAR and R.ARG /= -MATH_PI
  -- Error conditions:
  --         Error if L.ARG = -MATH_PI
  --         Error if R.ARG = -MATH_PI
  -- Range:
  --         "/="(L,R) is either TRUE or FALSE
  -- Notes:
  --         None

  --
  -- Function Declarations
  --
  function CMPLX(X : in REAL; Y : in REAL := 0.0) return COMPLEX;
  -- Purpose:
  --         Returns COMPLEX number X + iY
  -- Special values:
  --         None
  -- Domain:
  --         X in REAL
  --         Y in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         CMPLX(X,Y) is mathematically unbounded
  -- Notes:
  --         None

  function GET_PRINCIPAL_VALUE(X : in REAL) return PRINCIPAL_VALUE;
  -- Purpose:
  --         Returns principal value of angle X; X in radians
  -- Special values:
  --         None
  -- Domain:
  --         X in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         -MATH_PI < GET_PRINCIPAL_VALUE(X) <= MATH_PI
  -- Notes:
  --         None

  function COMPLEX_TO_POLAR(Z : in COMPLEX) return COMPLEX_POLAR;
  -- Purpose:
  --         Returns principal value COMPLEX_POLAR of Z
  -- Special values:
  --         COMPLEX_TO_POLAR(MATH_CZERO) = COMPLEX_POLAR'(0.0, 0.0)
  --         COMPLEX_TO_POLAR(Z) = COMPLEX_POLAR'(ABS(Z.IM),
  --                              SIGN(Z.IM)*MATH_PI_OVER_2) if Z.RE = 0.0
  -- Domain:
  --         Z in COMPLEX
  -- Error conditions:
  --         None
  -- Range:
  --         result.MAG >= 0.0
  --         -MATH_PI < result.ARG <= MATH_PI
  -- Notes:
  --         None

  function POLAR_TO_COMPLEX(Z : in COMPLEX_POLAR) return COMPLEX;
  -- Purpose:
  --         Returns COMPLEX value of Z
  -- Special values:
  --         None
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  -- Range:
  --         POLAR_TO_COMPLEX(Z) is mathematically unbounded
  -- Notes:
  --         None

  function "ABS"(Z : in COMPLEX) return POSITIVE_REAL;
  -- Purpose:
  --         Returns absolute value (magnitude) of Z
  -- Special values:
  --         None
  -- Domain:
  --         Z in COMPLEX
  -- Error conditions:
  --         None
  -- Range:
  --         ABS(Z) is mathematically unbounded
  -- Notes:
  --         ABS(Z) = SQRT(Z.RE*Z.RE + Z.IM*Z.IM)

  function "ABS"(Z : in COMPLEX_POLAR) return POSITIVE_REAL;
  -- Purpose:
  --         Returns absolute value (magnitude) of Z
  -- Special values:
  --         None
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  -- Range:
  --         ABS(Z) >= 0.0
  -- Notes:
  --         ABS(Z) = Z.MAG

  function ARG(Z : in COMPLEX) return PRINCIPAL_VALUE;
  -- Purpose:
  --         Returns argument (angle) in radians of the principal
  --         value of Z
  -- Special values:
  --         ARG(Z) = 0.0 if Z.RE >= 0.0 and Z.IM = 0.0
  --         ARG(Z) = SIGN(Z.IM)*MATH_PI_OVER_2 if Z.RE = 0.0
  --         ARG(Z) = MATH_PI if Z.RE < 0.0        and Z.IM = 0.0
  -- Domain:
  --         Z in COMPLEX
  -- Error conditions:
  --         None
  -- Range:
  --         -MATH_PI < ARG(Z) <= MATH_PI
  -- Notes:
  --         ARG(Z) = ARCTAN(Z.IM, Z.RE)

  function ARG(Z : in COMPLEX_POLAR) return PRINCIPAL_VALUE;
  -- Purpose:
  --         Returns argument (angle) in radians of the principal
  --         value of Z
  -- Special values:
  --         None
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  -- Range:
  --         -MATH_PI < ARG(Z) <= MATH_PI
  -- Notes:
  --         ARG(Z) = Z.ARG


  function "-" (Z : in COMPLEX) return COMPLEX;
  -- Purpose:
  --         Returns unary minus of Z
  -- Special values:
  --         None
  -- Domain:
  --         Z in COMPLEX
  -- Error conditions:
  --         None
  -- Range:
  --         "-"(Z) is mathematically unbounded
  -- Notes:
  --         Returns -x -jy for Z= x + jy

  function "-" (Z : in COMPLEX_POLAR) return COMPLEX_POLAR;
  -- Purpose:
  --         Returns principal value of unary minus of Z
  -- Special values:
  --         "-"(Z) = COMPLEX_POLAR'(Z.MAG, MATH_PI) if Z.ARG = 0.0
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  -- Range:
  --         result.MAG >= 0.0
  --         -MATH_PI < result.ARG <= MATH_PI
  -- Notes:
  --         Returns COMPLEX_POLAR'(Z.MAG, Z.ARG - SIGN(Z.ARG)*MATH_PI) if
  --                Z.ARG /= 0.0

  function CONJ (Z : in COMPLEX) return COMPLEX;
  -- Purpose:
  --         Returns complex conjugate of Z
  -- Special values:
  --         None
  -- Domain:
  --         Z in COMPLEX
  -- Error conditions:
  --         None
  -- Range:
  --         CONJ(Z) is mathematically unbounded
  -- Notes:
  --         Returns x -jy for Z= x + jy

  function CONJ (Z : in COMPLEX_POLAR) return COMPLEX_POLAR;
  -- Purpose:
  --         Returns principal value of complex conjugate of Z
  -- Special values:
  --         CONJ(Z) = COMPLEX_POLAR'(Z.MAG, MATH_PI) if Z.ARG = MATH_PI
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  -- Range:
  --         result.MAG >= 0.0
  --         -MATH_PI < result.ARG <= MATH_PI
  -- Notes:
  --         Returns COMPLEX_POLAR'(Z.MAG, -Z.ARG) if Z.ARG /= MATH_PI

  function SQRT(Z : in COMPLEX) return COMPLEX;
  -- Purpose:
  --         Returns square root of Z with positive real part
  --         or, if the real part is zero, the one with nonnegative
  --         imaginary part
  -- Special values:
  --         SQRT(MATH_CZERO) = MATH_CZERO
  -- Domain:
  --         Z in COMPLEX
  -- Error conditions:
  --         None
  -- Range:
  --         SQRT(Z) is mathematically unbounded
  -- Notes:
  --         None

  function SQRT(Z : in COMPLEX_POLAR) return COMPLEX_POLAR;
  -- Purpose:
  --         Returns square root of Z with positive real part
  --         or, if the real part is zero, the one with nonnegative
  --         imaginary part
  -- Special values:
  --         SQRT(Z) = COMPLEX_POLAR'(0.0, 0.0) if Z.MAG = 0.0
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  -- Range:
  --         result.MAG >= 0.0
  --         -MATH_PI < result.ARG <= MATH_PI
  -- Notes:
  --         None

  function EXP(Z : in COMPLEX) return COMPLEX;
  -- Purpose:
  --         Returns exponential of Z
  -- Special values:
  --         EXP(MATH_CZERO) = MATH_CBASE_1
  --         EXP(Z) = -MATH_CBASE_1 if Z.RE = 0.0 and ABS(Z.IM) = MATH_PI
  --         EXP(Z) = SIGN(Z.IM)*MATH_CBASE_J if Z.RE = 0.0 and
  --                                          ABS(Z.IM) =  MATH_PI_OVER_2
  -- Domain:
  --         Z in COMPLEX
  -- Error conditions:
  --         None
  -- Range:
  --         EXP(Z) is mathematically unbounded
  -- Notes:
  --         None



  function EXP(Z : in COMPLEX_POLAR) return COMPLEX_POLAR;
  -- Purpose:
  --         Returns principal value of exponential of Z
  -- Special values:
  --         EXP(Z) = COMPLEX_POLAR'(1.0, 0.0) if Z.MAG =0.0 and
  --                                                        Z.ARG = 0.0
  --         EXP(Z) = COMPLEX_POLAR'(1.0, MATH_PI) if Z.MAG = MATH_PI and
  --                                        ABS(Z.ARG) = MATH_PI_OVER_2
  --         EXP(Z) = COMPLEX_POLAR'(1.0, MATH_PI_OVER_2) if
  --                                        Z.MAG = MATH_PI_OVER_2 and
  --                                        Z.ARG = MATH_PI_OVER_2
  --         EXP(Z) = COMPLEX_POLAR'(1.0, -MATH_PI_OVER_2) if
  --                                        Z.MAG = MATH_PI_OVER_2 and
  --                                        Z.ARG = -MATH_PI_OVER_2
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  -- Range:
  --         result.MAG >= 0.0
  --         -MATH_PI < result.ARG <= MATH_PI
  -- Notes:
  --         None

  function LOG(Z : in COMPLEX) return COMPLEX;
  -- Purpose:
  --         Returns natural logarithm of Z
  -- Special values:
  --         LOG(MATH_CBASE_1) = MATH_CZERO
  --         LOG(-MATH_CBASE_1) = COMPLEX'(0.0, MATH_PI)
  --         LOG(MATH_CBASE_J) = COMPLEX'(0.0, MATH_PI_OVER_2)
  --         LOG(-MATH_CBASE_J) = COMPLEX'(0.0, -MATH_PI_OVER_2)
  --         LOG(Z) = MATH_CBASE_1 if Z = COMPLEX'(MATH_E, 0.0)
  -- Domain:
  --         Z in COMPLEX and ABS(Z) /= 0.0
  -- Error conditions:
  --         Error if ABS(Z) = 0.0
  -- Range:
  --         LOG(Z) is mathematically unbounded
  -- Notes:
  --         None

  function LOG2(Z : in COMPLEX) return COMPLEX;
  -- Purpose:
  --         Returns logarithm base 2 of Z
  -- Special values:
  --         LOG2(MATH_CBASE_1) = MATH_CZERO
  --         LOG2(Z) = MATH_CBASE_1 if Z = COMPLEX'(2.0, 0.0)
  -- Domain:
  --         Z in COMPLEX and ABS(Z) /= 0.0
  -- Error conditions:
  --         Error if ABS(Z) = 0.0
  -- Range:
  --         LOG2(Z) is mathematically unbounded
  -- Notes:
  --         None

  function LOG10(Z : in COMPLEX) return COMPLEX;
  -- Purpose:
  --         Returns logarithm base 10 of Z
  -- Special values:
  --         LOG10(MATH_CBASE_1) = MATH_CZERO
  --         LOG10(Z) = MATH_CBASE_1 if Z = COMPLEX'(10.0, 0.0)
  -- Domain:
  --         Z in COMPLEX and ABS(Z) /= 0.0
  -- Error conditions:
  --         Error if ABS(Z) = 0.0
  -- Range:
  --         LOG10(Z) is mathematically unbounded
  -- Notes:
  --         None

  function LOG(Z : in COMPLEX_POLAR) return COMPLEX_POLAR;
  -- Purpose:
  --         Returns principal value of natural logarithm of Z
  -- Special values:
  --         LOG(Z) = COMPLEX_POLAR'(0.0, 0.0) if Z.MAG = 1.0 and
  --                                             Z.ARG = 0.0
  --         LOG(Z) = COMPLEX_POLAR'(MATH_PI, MATH_PI_OVER_2) if
  --                              Z.MAG = 1.0 and Z.ARG = MATH_PI
  --         LOG(Z) = COMPLEX_POLAR'(MATH_PI_OVER_2, MATH_PI_OVER_2) if
  --                              Z.MAG = 1.0 and  Z.ARG = MATH_PI_OVER_2
  --         LOG(Z) = COMPLEX_POLAR'(MATH_PI_OVER_2, -MATH_PI_OVER_2) if
  --                              Z.MAG = 1.0 and  Z.ARG = -MATH_PI_OVER_2
  --         LOG(Z) = COMPLEX_POLAR'(1.0, 0.0) if Z.MAG = MATH_E and
  --                                             Z.ARG = 0.0
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  --         Z.MAG /= 0.0
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  --         Error if Z.MAG = 0.0
  -- Range:
  --         result.MAG >= 0.0
  --         -MATH_PI < result.ARG <= MATH_PI
  -- Notes:
  --         None

  function LOG2(Z : in COMPLEX_POLAR) return COMPLEX_POLAR;
  -- Purpose:
  --         Returns principal value of logarithm base 2 of Z
  -- Special values:
  --         LOG2(Z) = COMPLEX_POLAR'(0.0, 0.0) if Z.MAG = 1.0 and
  --                                              Z.ARG = 0.0
  --         LOG2(Z) = COMPLEX_POLAR'(1.0, 0.0) if Z.MAG = 2.0 and
  --                                             Z.ARG = 0.0
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  --         Z.MAG /= 0.0
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  --         Error if Z.MAG = 0.0
  -- Range:
  --         result.MAG >= 0.0
  --         -MATH_PI < result.ARG <= MATH_PI
  -- Notes:
  --        None

  function LOG10(Z : in COMPLEX_POLAR) return COMPLEX_POLAR;
  -- Purpose:
  --         Returns principal value of logarithm base 10 of Z
  -- Special values:
  --         LOG10(Z) = COMPLEX_POLAR'(0.0, 0.0) if Z.MAG = 1.0 and
  --                                               Z.ARG = 0.0
  --         LOG10(Z) = COMPLEX_POLAR'(1.0, 0.0) if Z.MAG = 10.0 and
  --                                               Z.ARG = 0.0
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  --         Z.MAG /= 0.0
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  --         Error if Z.MAG = 0.0
  -- Range:
  --         result.MAG >= 0.0
  --         -MATH_PI < result.ARG <= MATH_PI
  -- Notes:
  --         None

  function LOG(Z : in COMPLEX; BASE : in REAL) return COMPLEX;
  -- Purpose:
  --         Returns logarithm base BASE of Z
  -- Special values:
  --         LOG(MATH_CBASE_1, BASE) = MATH_CZERO
  --         LOG(Z,BASE) = MATH_CBASE_1 if Z = COMPLEX'(BASE, 0.0)
  -- Domain:
  --         Z in COMPLEX and ABS(Z) /= 0.0
  --         BASE > 0.0
  --         BASE /= 1.0
  -- Error conditions:
  --         Error if ABS(Z) = 0.0
  --         Error if BASE <= 0.0
  --         Error if BASE = 1.0
  -- Range:
  --         LOG(Z,BASE) is mathematically unbounded
  -- Notes:
  --         None

  function LOG(Z : in COMPLEX_POLAR; BASE : in REAL) return COMPLEX_POLAR;
  -- Purpose:
  --         Returns principal value of logarithm base BASE of Z
  -- Special values:
  --         LOG(Z, BASE) = COMPLEX_POLAR'(0.0, 0.0) if Z.MAG = 1.0 and
  --                                                Z.ARG = 0.0
  --         LOG(Z, BASE) = COMPLEX_POLAR'(1.0, 0.0) if Z.MAG = BASE and
  --                                                Z.ARG = 0.0
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  --         Z.MAG /= 0.0
  --         BASE > 0.0
  --         BASE /= 1.0
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  --         Error if Z.MAG = 0.0
  --         Error if BASE <= 0.0
  --         Error if BASE = 1.0
  -- Range:
  --         result.MAG >= 0.0
  --         -MATH_PI < result.ARG <= MATH_PI
  -- Notes:
  --         None

  function SIN (Z : in COMPLEX) return COMPLEX;
  -- Purpose:
  --         Returns sine of Z
  -- Special values:
  --         SIN(MATH_CZERO) = MATH_CZERO
  --         SIN(Z) = MATH_CZERO if Z = COMPLEX'(MATH_PI, 0.0)
  -- Domain:
  --         Z in COMPLEX
  -- Error conditions:
  --         None
  -- Range:
  --         ABS(SIN(Z)) <= SQRT(SIN(Z.RE)*SIN(Z.RE) +
  --                                         SINH(Z.IM)*SINH(Z.IM))
  -- Notes:
  --         None

  function SIN (Z : in COMPLEX_POLAR) return COMPLEX_POLAR;
  -- Purpose:
  --         Returns principal value of sine of Z
  -- Special values:
  --         SIN(Z) = COMPLEX_POLAR'(0.0, 0.0) if Z.MAG = 0.0 and
  --                                            Z.ARG = 0.0
  --         SIN(Z) = COMPLEX_POLAR'(0.0, 0.0) if Z.MAG = MATH_PI and
  --                                            Z.ARG = 0.0
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  -- Range:
  --         result.MAG >= 0.0
  --         -MATH_PI < result.ARG <= MATH_PI
  -- Notes:
  --         None

  function COS (Z : in COMPLEX) return COMPLEX;
  -- Purpose:
  --         Returns cosine of Z
  -- Special values:
  --         COS(Z) = MATH_CZERO if Z = COMPLEX'(MATH_PI_OVER_2, 0.0)
  --         COS(Z) = MATH_CZERO if Z = COMPLEX'(-MATH_PI_OVER_2, 0.0)
  -- Domain:
  --         Z in COMPLEX
  -- Error conditions:
  --         None
  -- Range:
  --         ABS(COS(Z)) <= SQRT(COS(Z.RE)*COS(Z.RE) +
  --                                         SINH(Z.IM)*SINH(Z.IM))
  -- Notes:
  --         None


  function COS (Z : in COMPLEX_POLAR) return COMPLEX_POLAR;
  -- Purpose:
  --         Returns principal value of cosine of Z
  -- Special values:
  --         COS(Z) = COMPLEX_POLAR'(0.0, 0.0) if Z.MAG = MATH_PI_OVER_2
  --                                               and Z.ARG = 0.0
  --         COS(Z) = COMPLEX_POLAR'(0.0, 0.0) if Z.MAG = MATH_PI_OVER_2
  --                                               and Z.ARG = MATH_PI
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  -- Range:
  --         result.MAG >= 0.0
  --         -MATH_PI < result.ARG <= MATH_PI
  -- Notes:
  --         None

  function SINH (Z : in COMPLEX) return COMPLEX;
  -- Purpose:
  --         Returns hyperbolic sine of Z
  -- Special values:
  --         SINH(MATH_CZERO) = MATH_CZERO
  --         SINH(Z) = MATH_CZERO if Z.RE = 0.0 and Z.IM = MATH_PI
  --         SINH(Z) = MATH_CBASE_J if Z.RE = 0.0 and
  --                                               Z.IM = MATH_PI_OVER_2
  --         SINH(Z) = -MATH_CBASE_J if Z.RE = 0.0 and
  --                                               Z.IM = -MATH_PI_OVER_2
  -- Domain:
  --         Z in COMPLEX
  -- Error conditions:
  --         None
  -- Range:
  --         ABS(SINH(Z)) <= SQRT(SINH(Z.RE)*SINH(Z.RE) +
  --                                         SIN(Z.IM)*SIN(Z.IM))
  -- Notes:
  --         None

  function SINH (Z : in COMPLEX_POLAR) return COMPLEX_POLAR;
  -- Purpose:
  --         Returns principal value of hyperbolic sine of Z
  -- Special values:
  --         SINH(Z) = COMPLEX_POLAR'(0.0, 0.0) if Z.MAG = 0.0 and
  --                                            Z.ARG = 0.0
  --         SINH(Z) = COMPLEX_POLAR'(0.0, 0.0) if Z.MAG = MATH_PI and
  --                                            Z.ARG = MATH_PI_OVER_2
  --         SINH(Z) = COMPLEX_POLAR'(1.0, MATH_PI_OVER_2) if Z.MAG =
  --                         MATH_PI_OVER_2 and Z.ARG = MATH_PI_OVER_2
  --         SINH(Z) = COMPLEX_POLAR'(1.0, -MATH_PI_OVER_2) if Z.MAG =
  --                         MATH_PI_OVER_2 and Z.ARG = -MATH_PI_OVER_2
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  -- Range:
  --         result.MAG >= 0.0
  --         -MATH_PI < result.ARG <= MATH_PI
  -- Notes:
  --         None

  function COSH (Z : in COMPLEX) return COMPLEX;
  -- Purpose:
  --         Returns hyperbolic cosine of Z
  -- Special values:
  --         COSH(MATH_CZERO) = MATH_CBASE_1
  --         COSH(Z) = -MATH_CBASE_1 if Z.RE = 0.0 and Z.IM = MATH_PI
  --         COSH(Z) = MATH_CZERO if Z.RE = 0.0 and Z.IM = MATH_PI_OVER_2
  --         COSH(Z) = MATH_CZERO if Z.RE = 0.0 and Z.IM = -MATH_PI_OVER_2
  -- Domain:
  --         Z in COMPLEX
  -- Error conditions:
  --         None
  -- Range:
  --         ABS(COSH(Z)) <= SQRT(SINH(Z.RE)*SINH(Z.RE) +
  --                                         COS(Z.IM)*COS(Z.IM))
  -- Notes:
  --         None


  function COSH (Z : in COMPLEX_POLAR) return COMPLEX_POLAR;
  -- Purpose:
  --         Returns principal value of hyperbolic cosine of Z
  -- Special values:
  --         COSH(Z) = COMPLEX_POLAR'(1.0, 0.0) if Z.MAG = 0.0 and
  --                                            Z.ARG = 0.0
  --         COSH(Z) = COMPLEX_POLAR'(1.0, MATH_PI) if Z.MAG = MATH_PI and
  --                                            Z.ARG = MATH_PI_OVER_2
  --         COSH(Z) = COMPLEX_POLAR'(0.0, 0.0) if Z.MAG =
  --                        MATH_PI_OVER_2 and Z.ARG = MATH_PI_OVER_2
  --         COSH(Z) = COMPLEX_POLAR'(0.0, 0.0) if Z.MAG =
  --                        MATH_PI_OVER_2 and Z.ARG = -MATH_PI_OVER_2
  -- Domain:
  --         Z in COMPLEX_POLAR and Z.ARG /= -MATH_PI
  -- Error conditions:
  --         Error if Z.ARG = -MATH_PI
  -- Range:
  --         result.MAG >= 0.0
  --         -MATH_PI < result.ARG <= MATH_PI
  -- Notes:
  --         None

  --
  -- Arithmetic Operators
  --
  function "+" (L : in COMPLEX; R : in COMPLEX) return COMPLEX;
  function "+" (L : in REAL; R : in COMPLEX) return COMPLEX;
  function "+" (L : in COMPLEX; R : in REAL) return COMPLEX;
  function "+" (L : in COMPLEX_POLAR; R : in COMPLEX_POLAR) return COMPLEX_POLAR;
  function "+" (L : in REAL; R : in COMPLEX_POLAR) return COMPLEX_POLAR;
  function "+" (L : in COMPLEX_POLAR; R : in REAL) return COMPLEX_POLAR;
  function "-" (L : in COMPLEX; R : in COMPLEX) return COMPLEX;
  function "-" (L : in REAL; R : in COMPLEX) return COMPLEX;
  function "-" (L : in COMPLEX; R : in REAL) return COMPLEX;
  function "-" (L : in COMPLEX_POLAR; R : in COMPLEX_POLAR) return COMPLEX_POLAR;
  function "-" (L : in REAL; R : in COMPLEX_POLAR) return COMPLEX_POLAR;
  function "-" (L : in COMPLEX_POLAR; R : in REAL) return COMPLEX_POLAR;
  function "*" (L : in COMPLEX; R : in COMPLEX) return COMPLEX;
  function "*" (L : in REAL; R : in COMPLEX) return COMPLEX;
  function "*" (L : in COMPLEX_POLAR; R : in COMPLEX_POLAR) return COMPLEX_POLAR;
  function "*" (L : in REAL; R : in COMPLEX_POLAR) return COMPLEX_POLAR;
  function "*" (L : in COMPLEX_POLAR; R : in REAL) return COMPLEX_POLAR;
  function "/" (L : in COMPLEX; R : in COMPLEX) return COMPLEX;
  function "/" (L : in REAL; R : in COMPLEX) return COMPLEX;
  function "/" (L : in COMPLEX; R : in REAL) return COMPLEX;
  function "/" (L : in COMPLEX_POLAR; R : in COMPLEX_POLAR) return COMPLEX_POLAR;
  function "/" (L : in REAL; R : in COMPLEX_POLAR) return COMPLEX_POLAR;
  function "/" (L : in COMPLEX_POLAR; R : in REAL) return COMPLEX_POLAR;

end package MATH_COMPLEX;

use std.standard.all;
package MATH_REAL is

-- $Date: 2008-04-10 17:16:09 +0930 (Thu, 10 Apr 2008) $

                                        -- Value of e
  constant MATH_E             : REAL := 2.71828_18284_59045_23536;
                                        -- Value of 1/e
  constant MATH_1_OVER_E      : REAL := 0.36787_94411_71442_32160;
                                        -- Value of pi
  constant MATH_PI            : REAL := 3.14159_26535_89793_23846;
                                        -- Value of 2*pi
  constant MATH_2_PI          : REAL := 6.28318_53071_79586_47693;
                                        -- Value of 1/pi
  constant MATH_1_OVER_PI     : REAL := 0.31830_98861_83790_67154;
                                        -- Value of pi/2
  constant MATH_PI_OVER_2     : REAL := 1.57079_63267_94896_61923;
                                        -- Value of pi/3
  constant MATH_PI_OVER_3     : REAL := 1.04719_75511_96597_74615;
                                        -- Value of pi/4
  constant MATH_PI_OVER_4     : REAL := 0.78539_81633_97448_30962;
                                        -- Value 3*pi/2
  constant MATH_3_PI_OVER_2   : REAL := 4.71238_89803_84689_85769;
                                        -- Natural log of 2
  constant MATH_LOG_OF_2      : REAL := 0.69314_71805_59945_30942;
                                        -- Natural log of 10
  constant MATH_LOG_OF_10     : REAL := 2.30258_50929_94045_68402;
                                        -- Log base 2 of e
  constant MATH_LOG2_OF_E     : REAL := 1.44269_50408_88963_4074;
                                        -- Log base 10 of e
  constant MATH_LOG10_OF_E    : REAL := 0.43429_44819_03251_82765;
                                        -- square root of 2
  constant MATH_SQRT_2        : REAL := 1.41421_35623_73095_04880;
                                        -- square root of 1/2
  constant MATH_1_OVER_SQRT_2 : REAL := 0.70710_67811_86547_52440;
                                        -- square root of pi
  constant MATH_SQRT_PI       : REAL := 1.77245_38509_05516_02730;
  -- Conversion factor from degree to radian
  constant MATH_DEG_TO_RAD    : REAL := 0.01745_32925_19943_29577;
  -- Conversion factor from radian to degree
  constant MATH_RAD_TO_DEG    : REAL := 57.29577_95130_82320_87680;

  -- Purpose:
  --         Returns 1.0 if X > 0.0; 0.0 if X = 0.0; -1.0 if X < 0.0
  -- Special values:
  --         None
  -- Domain:
  --         X in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         ABS(SIGN(X)) <= 1.0
  function SIGN (X : in REAL) return REAL;

  -- Purpose:
  --         Returns smallest INTEGER value (as REAL) not less than X
  -- Special values:
  --         None
  -- Domain:
  --         X in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         CEIL(X) is mathematically unbounded
  -- Notes:
  --         a) Implementations have to support at least the domain
  --                ABS(X) < REAL(INTEGER'HIGH)
  function CEIL (X : in REAL) return REAL;

  -- Purpose:
  --         Returns largest INTEGER value (as REAL) not greater than X
  -- Special values:
  --         FLOOR(0.0) = 0.0
  -- Domain:
  --         X in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         FLOOR(X) is mathematically unbounded
  -- Notes:
  --         a) Implementations have to support at least the domain
  --                ABS(X) < REAL(INTEGER'HIGH)
  function FLOOR (X : in REAL) return REAL;

  -- Purpose:
  --         Rounds X to the nearest integer value (as real). If X is
  --         halfway between two integers, rounding is away from 0.0
  -- Special values:
  --         ROUND(0.0) = 0.0
  -- Domain:
  --         X in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         ROUND(X) is mathematically unbounded
  -- Notes:
  --         a) Implementations have to support at least the domain
  --                ABS(X) < REAL(INTEGER'HIGH)
  function ROUND (X : in REAL) return REAL;

  -- Purpose:
  --         Truncates X towards 0.0 and returns truncated value
  -- Special values:
  --         TRUNC(0.0) = 0.0
  -- Domain:
  --         X in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         TRUNC(X) is mathematically unbounded
  -- Notes:
  --         a) Implementations have to support at least the domain
  --                ABS(X) < REAL(INTEGER'HIGH)
  function TRUNC (X : in REAL) return REAL;


  -- Purpose:
  --         Returns the algebraically larger of X and Y
  -- Special values:
  --         REALMAX(X,Y) = X when X = Y
  -- Domain:
  --         X in REAL; Y in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         REALMAX(X,Y) is mathematically unbounded
  function REALMAX (X, Y : in REAL) return REAL;

  -- Purpose:
  --         Returns the algebraically smaller of X and Y
  -- Special values:
  --         REALMIN(X,Y) = X when X = Y
  -- Domain:
  --         X in REAL; Y in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         REALMIN(X,Y) is mathematically unbounded
  function REALMIN (X, Y : in REAL) return REAL;

  -- Purpose:
  --         Returns, in X, a pseudo-random number with uniform
  --         distribution in the open interval (0.0, 1.0).
  -- Special values:
  --         None
  -- Domain:
  --         1 <= SEED1 <= 2147483562; 1 <= SEED2 <= 2147483398
  -- Error conditions:
  --         Error if SEED1 or SEED2 outside of valid domain
  -- Range:
  --         0.0 < X < 1.0
  -- Notes:
  --         a) The semantics for this function are described by the
  --            algorithm published by Pierre L'Ecuyer in "Communications
  --            of the ACM," vol. 31, no. 6, June 1988, pp. 742-774.
  --            The algorithm is based on the combination of two
  --            multiplicative linear congruential generators for 32-bit
  --            platforms.
  --
  --         b) Before the first call to UNIFORM, the seed values
  --            (SEED1, SEED2) have to be initialized to values in the range
  --            [1, 2147483562] and [1, 2147483398] respectively.  The
  --            seed values are modified after each call to UNIFORM.
  --
  --         c) This random number generator is portable for 32-bit
  --            computers, and it has a period of ~2.30584*(10**18) for each
  --            set of seed values.
  --
  --         d) For information on spectral tests for the algorithm, refer
  --            to the L'Ecuyer article.
  procedure UNIFORM(variable SEED1, SEED2 : inout POSITIVE; variable X : out REAL);

  -- Purpose:
  --         Returns square root of X
  -- Special values:
  --         SQRT(0.0) = 0.0
  --         SQRT(1.0) = 1.0
  -- Domain:
  --         X >= 0.0
  -- Error conditions:
  --         Error if X < 0.0
  -- Range:
  --         SQRT(X) >= 0.0
  -- Notes:
  --         a) The upper bound of the reachable range of SQRT is
  --            approximately given by:
  --                SQRT(X) <= SQRT(REAL'HIGH)
  function SQRT (X : in REAL) return REAL;

  -- Purpose:
  --         Returns cube root of X
  -- Special values:
  --         CBRT(0.0) = 0.0
  --         CBRT(1.0) = 1.0
  --         CBRT(-1.0) = -1.0
  -- Domain:
  --         X in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         CBRT(X) is mathematically unbounded
  -- Notes:
  --         a) The reachable range of CBRT is approximately given by:
  --                ABS(CBRT(X)) <= CBRT(REAL'HIGH)
  function CBRT (X : in REAL) return REAL;

  -- Purpose:
  --         Returns Y power of X ==>  X**Y
  -- Special values:
  --         X**0.0 = 1.0; X /= 0
  --         0**Y = 0.0; Y > 0.0
  --         X**1.0 = REAL(X); X >= 0
  --         1**Y = 1.0
  -- Domain:
  --         X > 0
  --         X = 0 for Y > 0.0
  --         X < 0 for Y = 0.0
  -- Error conditions:
  --         Error if X < 0 and Y /= 0.0
  --         Error if X = 0 and Y <= 0.0
  -- Range:
  --         X**Y >= 0.0
  -- Notes:
  --         a) The upper bound of the reachable range for "**" is
  --            approximately given by:
  --                X**Y <= REAL'HIGH
  function "**" (X : in INTEGER; Y : in REAL) return REAL;

  -- Purpose:
  --         Returns Y power of X ==>  X**Y
  -- Special values:
  --         X**0.0 = 1.0; X /= 0.0
  --         0.0**Y = 0.0; Y > 0.0
  --         X**1.0 = X; X >= 0.0
  --         1.0**Y = 1.0
  -- Domain:
  --         X > 0.0
  --         X = 0.0 for Y > 0.0
  --         X < 0.0 for Y = 0.0
  -- Error conditions:
  --         Error if X < 0.0 and Y /= 0.0
  --         Error if X = 0.0 and Y <= 0.0
  -- Range:
  --         X**Y >= 0.0
  -- Notes:
  --         a) The upper bound of the reachable range for "**" is
  --            approximately given by:
  --                X**Y <= REAL'HIGH
  function "**" (X : in REAL; Y : in REAL) return REAL;

  -- Purpose:
  --         Returns e**X; where e = MATH_E
  -- Special values:
  --         EXP(0.0) = 1.0
  --         EXP(1.0) = MATH_E
  --         EXP(-1.0) = MATH_1_OVER_E
  --         EXP(X) = 0.0 for X <= -LOG(REAL'HIGH)
  -- Domain:
  --         X in REAL such that EXP(X) <= REAL'HIGH
  -- Error conditions:
  --         Error if X > LOG(REAL'HIGH)
  -- Range:
  --         EXP(X) >= 0.0
  -- Notes:
  --         a) The usable domain of EXP is approximately given by:
  --                X <= LOG(REAL'HIGH)
  function EXP (X : in REAL) return REAL;

  -- Purpose:
  --         Returns natural logarithm of X
  -- Special values:
  --         LOG(1.0) = 0.0
  --         LOG(MATH_E) = 1.0
  -- Domain:
  --         X > 0.0
  -- Error conditions:
  --         Error if X <= 0.0
  -- Range:
  --         LOG(X) is mathematically unbounded
  -- Notes:
  --         a) The reachable range of LOG is approximately given by:
  --                LOG(0+) <= LOG(X) <= LOG(REAL'HIGH)
  function LOG (X : in REAL) return REAL;

  -- Purpose:
  --         Returns logarithm base 2 of X
  -- Special values:
  --         LOG2(1.0) = 0.0
  --         LOG2(2.0) = 1.0
  -- Domain:
  --         X > 0.0
  -- Error conditions:
  --         Error if X <= 0.0
  -- Range:
  --         LOG2(X) is mathematically unbounded
  -- Notes:
  --         a) The reachable range of LOG2 is approximately given by:
  --                LOG2(0+) <= LOG2(X) <= LOG2(REAL'HIGH)
  function LOG2 (X : in REAL) return REAL;

  -- Purpose:
  --         Returns logarithm base 10 of X
  -- Special values:
  --         LOG10(1.0) = 0.0
  --         LOG10(10.0) = 1.0
  -- Domain:
  --         X > 0.0
  -- Error conditions:
  --         Error if X <= 0.0
  -- Range:
  --         LOG10(X) is mathematically unbounded
  -- Notes:
  --         a) The reachable range of LOG10 is approximately given by:
  --                LOG10(0+) <= LOG10(X) <= LOG10(REAL'HIGH)
  function LOG10 (X : in REAL) return REAL;

  -- Purpose:
  --         Returns logarithm base BASE of X
  -- Special values:
  --         LOG(1.0, BASE) = 0.0
  --         LOG(BASE, BASE) = 1.0
  -- Domain:
  --         X > 0.0
  --         BASE > 0.0
  --         BASE /= 1.0
  -- Error conditions:
  --         Error if X <= 0.0
  --         Error if BASE <= 0.0
  --         Error if BASE = 1.0
  -- Range:
  --         LOG(X, BASE) is mathematically unbounded
  -- Notes:
  --         a) When BASE > 1.0, the reachable range of LOG is
  --            approximately given by:
  --                LOG(0+, BASE) <= LOG(X, BASE) <= LOG(REAL'HIGH, BASE)
  --         b) When 0.0 < BASE < 1.0, the reachable range of LOG is
  --            approximately given by:
  --                LOG(REAL'HIGH, BASE) <= LOG(X, BASE) <= LOG(0+, BASE)
  function LOG (X : in REAL; BASE : in REAL) return REAL;

  -- Purpose:
  --         Returns sine of X; X in radians
  -- Special values:
  --         SIN(X) = 0.0 for X = k*MATH_PI, where k is an INTEGER
  --         SIN(X) = 1.0 for X = (4*k+1)*MATH_PI_OVER_2, where k is an
  --                                                           INTEGER
  --         SIN(X) = -1.0 for X = (4*k+3)*MATH_PI_OVER_2, where k is an
  --                                                           INTEGER
  -- Domain:
  --         X in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         ABS(SIN(X)) <= 1.0
  -- Notes:
  --         a) For larger values of ABS(X), degraded accuracy is allowed.
  function SIN (X : in REAL) return REAL;

  -- Purpose:
  --         Returns cosine of X; X in radians
  -- Special values:
  --         COS(X) = 0.0 for X = (2*k+1)*MATH_PI_OVER_2, where k is an
  --                                                            INTEGER
  --         COS(X) = 1.0 for X = (2*k)*MATH_PI, where k is an INTEGER
  --         COS(X) = -1.0 for X = (2*k+1)*MATH_PI, where k is an INTEGER
  -- Domain:
  --         X in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         ABS(COS(X)) <= 1.0
  -- Notes:
  --         a) For larger values of ABS(X), degraded accuracy is allowed.
  function COS (X : in REAL) return REAL;

  -- Purpose:
  --         Returns tangent of X; X in radians
  -- Special values:
  --         TAN(X) = 0.0 for X = k*MATH_PI, where k is an INTEGER
  -- Domain:
  --         X in REAL and
  --         X /= (2*k+1)*MATH_PI_OVER_2, where k is an INTEGER
  -- Error conditions:
  --         Error if X = ((2*k+1) * MATH_PI_OVER_2), where k is an
  --                                                           INTEGER
  -- Range:
  --         TAN(X) is mathematically unbounded
  -- Notes:
  --         a) For larger values of ABS(X), degraded accuracy is allowed.
  function TAN (X : in REAL) return REAL;

  -- Purpose:
  --         Returns inverse sine of X
  -- Special values:
  --         ARCSIN(0.0) = 0.0
  --         ARCSIN(1.0) = MATH_PI_OVER_2
  --         ARCSIN(-1.0) = -MATH_PI_OVER_2
  -- Domain:
  --         ABS(X) <= 1.0
  -- Error conditions:
  --         Error if ABS(X) > 1.0
  -- Range:
  --         ABS(ARCSIN(X) <= MATH_PI_OVER_2
  function ARCSIN (X : in REAL) return REAL;

  -- Purpose:
  --         Returns inverse cosine of X
  -- Special values:
  --         ARCCOS(1.0) = 0.0
  --         ARCCOS(0.0) = MATH_PI_OVER_2
  --         ARCCOS(-1.0) = MATH_PI
  -- Domain:
  --         ABS(X) <= 1.0
  -- Error conditions:
  --         Error if ABS(X) > 1.0
  -- Range:
  --         0.0 <= ARCCOS(X) <= MATH_PI
  function ARCCOS (X : in REAL) return REAL;

  -- Purpose:
  --         Returns the value of the angle in radians of the point
  --        (1.0, Y), which is in rectangular coordinates
  -- Special values:
  --         ARCTAN(0.0) = 0.0
  -- Domain:
  --         Y in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         ABS(ARCTAN(Y)) <= MATH_PI_OVER_2
  function ARCTAN (Y : in REAL) return REAL;

  -- Purpose:
  --         Returns the principal value of the angle in radians of
  --         the point (X, Y), which is in rectangular coordinates
  -- Special values:
  --         ARCTAN(0.0, X) = 0.0 if X > 0.0
  --         ARCTAN(0.0, X) = MATH_PI if X < 0.0
  --         ARCTAN(Y, 0.0) = MATH_PI_OVER_2 if Y > 0.0
  --         ARCTAN(Y, 0.0) = -MATH_PI_OVER_2 if Y < 0.0
  -- Domain:
  --         Y in REAL
  --         X in REAL, X /= 0.0 when Y = 0.0
  -- Error conditions:
  --         Error if X = 0.0 and Y = 0.0
  -- Range:
  --         -MATH_PI < ARCTAN(Y,X) <= MATH_PI
  function ARCTAN (Y : in REAL; X : in REAL) return REAL;

  -- Purpose:
  --         Returns hyperbolic sine of X
  -- Special values:
  --         SINH(0.0) = 0.0
  -- Domain:
  --         X in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         SINH(X) is mathematically unbounded
  -- Notes:
  --         a) The usable domain of SINH is approximately given by:
  --                ABS(X) <= LOG(REAL'HIGH)
  function SINH (X : in REAL) return REAL;

  -- Purpose:
  --         Returns hyperbolic cosine of X
  -- Special values:
  --         COSH(0.0) = 1.0
  -- Domain:
  --         X in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         COSH(X) >= 1.0
  -- Notes:
  --         a) The usable domain of COSH is approximately given by:
  --                ABS(X) <= LOG(REAL'HIGH)
  function COSH (X : in REAL) return REAL;

  -- Purpose:
  --         Returns hyperbolic tangent of X
  -- Special values:
  --         TANH(0.0) = 0.0
  -- Domain:
  --         X in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         ABS(TANH(X)) <= 1.0
  function TANH (X : in REAL) return REAL;

  -- Purpose:
  --         Returns inverse hyperbolic sine of X
  -- Special values:
  --         ARCSINH(0.0) = 0.0
  -- Domain:
  --         X in REAL
  -- Error conditions:
  --         None
  -- Range:
  --         ARCSINH(X) is mathematically unbounded
  -- Notes:
  --         a) The reachable range of ARCSINH is approximately given by:
  --                ABS(ARCSINH(X)) <= LOG(REAL'HIGH)
  function ARCSINH (X : in REAL) return REAL;

  -- Purpose:
  --         Returns inverse hyperbolic cosine of X
  -- Special values:
  --         ARCCOSH(1.0) = 0.0
  -- Domain:
  --         X >= 1.0
  -- Error conditions:
  --         Error if X < 1.0
  -- Range:
  --         ARCCOSH(X) >= 0.0
  -- Notes:
  --         a) The upper bound of the reachable range of ARCCOSH is
  --            approximately given by:   ARCCOSH(X) <= LOG(REAL'HIGH)
  function ARCCOSH (X : in REAL) return REAL;

  -- Purpose:
  --         Returns inverse hyperbolic tangent of X
  -- Special values:
  --         ARCTANH(0.0) = 0.0
  -- Domain:
  --         ABS(X) < 1.0
  -- Error conditions:
  --         Error if ABS(X) >= 1.0
  -- Range:
  --         ARCTANH(X) is mathematically unbounded
  -- Notes:
  --         a) The reachable range of ARCTANH is approximately given by:
  --                ABS(ARCTANH(X)) < LOG(REAL'HIGH)
  function ARCTANH (X : in REAL) return REAL;

end package MATH_REAL;

use STD.TEXTIO.all;
use std.standard.all;
package NUMERIC_BIT is

-- $Date: 2008-04-10 17:16:09 +0930 (Thu, 10 Apr 2008) $

  --============================================================================
  -- Numeric Array Type Definitions
  --============================================================================

  type UNSIGNED is array (NATURAL range <>) of BIT;
  type SIGNED is array (NATURAL range <>) of BIT;

  --============================================================================
  -- Arithmetic Operators:
  --============================================================================

  -- Id: A.1
  function "abs" (ARG : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Returns the absolute value of a SIGNED vector ARG.

  -- Id: A.2
  function "-" (ARG : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Returns the value of the unary minus operation on a
  --         SIGNED vector ARG.

  --============================================================================

  -- Id: A.3
  function "+" (L, R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(MAXIMUM(L'LENGTH, R'LENGTH)-1 downto 0)
  -- Result: Adds two UNSIGNED vectors that may be of different lengths.

  -- Id: A.3R
  function "+"(L : UNSIGNED; R : BIT) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Similar to A.3 where R is a one bit UNSIGNED

  -- Id: A.3L
  function "+"(L : BIT; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Similar to A.3 where L is a one bit UNSIGNED

  -- Id: A.4
  function "+" (L, R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(MAXIMUM(L'LENGTH, R'LENGTH)-1 downto 0)
  -- Result: Adds two SIGNED vectors that may be of different lengths.

  -- Id: A.4R
  function "+"(L : SIGNED; R : BIT) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Similar to A.4 where R is bit 0 of a non-negative.

  -- Id: A.4L
  function "+"(L : BIT; R : SIGNED) return SIGNED;
  -- Result subtype: UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Similar to A.4 where L is bit 0 of a non-negative.

  -- Id: A.5
  function "+" (L : UNSIGNED; R : NATURAL) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Adds an UNSIGNED vector, L, with a nonnegative INTEGER, R.

  -- Id: A.6
  function "+" (L : NATURAL; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Adds a nonnegative INTEGER, L, with an UNSIGNED vector, R.

  -- Id: A.7
  function "+" (L : INTEGER; R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Adds an INTEGER, L(may be positive or negative), to a SIGNED
  -- vector, R.

  -- Id: A.8
  function "+" (L : SIGNED; R : INTEGER) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Adds a SIGNED vector, L, to an INTEGER, R.

  --============================================================================

  -- Id: A.9
  function "-" (L, R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(MAXIMUM(L'LENGTH, R'LENGTH)-1 downto 0)
  -- Result: Subtracts two UNSIGNED vectors that may be of different lengths.

  -- Id: A.9R
  function "-"(L : UNSIGNED; R : BIT) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Similar to A.9 where R is a one bit UNSIGNED

  -- Id: A.9L
  function "-"(L : BIT; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Similar to A.9 where L is a one bit UNSIGNED

  -- Id: A.10
  function "-" (L, R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(MAXIMUM(L'LENGTH, R'LENGTH)-1 downto 0)
  -- Result: Subtracts a SIGNED vector, R, from another SIGNED vector, L,
  --         that may possibly be of different lengths.

  -- Id: A.10R
  function "-"(L : SIGNED; R : BIT) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Similar to A.10 where R is bit 0 of a non-negative.

  -- Id: A.10L
  function "-"(L : BIT; R : SIGNED) return SIGNED;
  -- Result subtype: UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Similar to A.10 where R is bit 0 of a non-negative.

  -- Id: A.11
  function "-" (L : UNSIGNED; R : NATURAL) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Subtracts a nonnegative INTEGER, R, from an UNSIGNED vector, L.

  -- Id: A.12
  function "-" (L : NATURAL; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Subtracts an UNSIGNED vector, R, from a nonnegative INTEGER, L.

  -- Id: A.13
  function "-" (L : SIGNED; R : INTEGER) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Subtracts an INTEGER, R, from a SIGNED vector, L.

  -- Id: A.14
  function "-" (L : INTEGER; R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Subtracts a SIGNED vector, R, from an INTEGER, L.

  --============================================================================

  -- Id: A.15
  function "*" (L, R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED((L'LENGTH+R'LENGTH-1) downto 0)
  -- Result: Performs the multiplication operation on two UNSIGNED vectors
  --         that may possibly be of different lengths.

  -- Id: A.16
  function "*" (L, R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED((L'LENGTH+R'LENGTH-1) downto 0)
  -- Result: Multiplies two SIGNED vectors that may possibly be of
  --         different lengths.

  -- Id: A.17
  function "*" (L : UNSIGNED; R : NATURAL) return UNSIGNED;
  -- Result subtype: UNSIGNED((L'LENGTH+L'LENGTH-1) downto 0)
  -- Result: Multiplies an UNSIGNED vector, L, with a nonnegative
  --         INTEGER, R. R is converted to an UNSIGNED vector of
  --         size L'LENGTH before multiplication.

  -- Id: A.18
  function "*" (L : NATURAL; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED((R'LENGTH+R'LENGTH-1) downto 0)
  -- Result: Multiplies an UNSIGNED vector, R, with a nonnegative
  --         INTEGER, L. L is converted to an UNSIGNED vector of
  --         size R'LENGTH before multiplication.

  -- Id: A.19
  function "*" (L : SIGNED; R : INTEGER) return SIGNED;
  -- Result subtype: SIGNED((L'LENGTH+L'LENGTH-1) downto 0)
  -- Result: Multiplies a SIGNED vector, L, with an INTEGER, R. R is
  --         converted to a SIGNED vector of size L'LENGTH before
  --         multiplication.

  -- Id: A.20
  function "*" (L : INTEGER; R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED((R'LENGTH+R'LENGTH-1) downto 0)
  -- Result: Multiplies a SIGNED vector, R, with an INTEGER, L. L is
  --         converted to a SIGNED vector of size R'LENGTH before
  --         multiplication.

  --============================================================================
  --
  -- NOTE: If second argument is zero for "/" operator, a severity level
  --       of ERROR is issued.

  -- Id: A.21
  function "/" (L, R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Divides an UNSIGNED vector, L, by another UNSIGNED vector, R.

  -- Id: A.22
  function "/" (L, R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Divides an SIGNED vector, L, by another SIGNED vector, R.

  -- Id: A.23
  function "/" (L : UNSIGNED; R : NATURAL) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Divides an UNSIGNED vector, L, by a nonnegative INTEGER, R.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.24
  function "/" (L : NATURAL; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Divides a nonnegative INTEGER, L, by an UNSIGNED vector, R.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  -- Id: A.25
  function "/" (L : SIGNED; R : INTEGER) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Divides a SIGNED vector, L, by an INTEGER, R.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.26
  function "/" (L : INTEGER; R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Divides an INTEGER, L, by a SIGNED vector, R.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  --============================================================================
  --
  -- NOTE: If second argument is zero for "rem" operator, a severity level
  --       of ERROR is issued.

  -- Id: A.27
  function "rem" (L, R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where L and R are UNSIGNED vectors.

  -- Id: A.28
  function "rem" (L, R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where L and R are SIGNED vectors.

  -- Id: A.29
  function "rem" (L : UNSIGNED; R : NATURAL) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where L is an UNSIGNED vector and R is a
  --         nonnegative INTEGER.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.30
  function "rem" (L : NATURAL; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where R is an UNSIGNED vector and L is a
  --         nonnegative INTEGER.
  -- If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  -- Id: A.31
  function "rem" (L : SIGNED; R : INTEGER) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where L is SIGNED vector and R is an INTEGER.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.32
  function "rem" (L : INTEGER; R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where R is SIGNED vector and L is an INTEGER.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  --============================================================================
  --
  -- NOTE: If second argument is zero for "mod" operator, a severity level
  --       of ERROR is issued.

  -- Id: A.33
  function "mod" (L, R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L and R are UNSIGNED vectors.

  -- Id: A.34
  function "mod" (L, R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L and R are SIGNED vectors.

  -- Id: A.35
  function "mod" (L : UNSIGNED; R : NATURAL) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L is an UNSIGNED vector and R
  --         is a nonnegative INTEGER.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.36
  function "mod" (L : NATURAL; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where R is an UNSIGNED vector and L
  --         is a nonnegative INTEGER.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  -- Id: A.37
  function "mod" (L : SIGNED; R : INTEGER) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L is a SIGNED vector and
  --         R is an INTEGER.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.38
  function "mod" (L : INTEGER; R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L is an INTEGER and
  --         R is a SIGNED vector.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  --============================================================================
  -- Id: A.39
  function find_leftmost (ARG : UNSIGNED; Y : BIT) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Finds the leftmost occurrence of the value of Y in ARG.
  --         Returns the index of the occurrence if it exists, or -1 otherwise.

  -- Id: A.40
  function find_leftmost (ARG : SIGNED; Y : BIT) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Finds the leftmost occurrence of the value of Y in ARG.
  --         Returns the index of the occurrence if it exists, or -1 otherwise.

  -- Id: A.41
  function find_rightmost (ARG : UNSIGNED; Y : BIT) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Finds the leftmost occurrence of the value of Y in ARG.
  --         Returns the index of the occurrence if it exists, or -1 otherwise.

  -- Id: A.42
  function find_rightmost (ARG : SIGNED; Y : BIT) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Finds the leftmost occurrence of the value of Y in ARG.
  --         Returns the index of the occurrence if it exists, or -1 otherwise.

  --============================================================================
  -- Comparison Operators
  --============================================================================

  -- Id: C.1
  function ">" (L, R : UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.2
  function ">" (L, R : SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L and R are SIGNED vectors possibly
  --         of different lengths.

  -- Id: C.3
  function ">" (L : NATURAL; R : UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L is a nonnegative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.4
  function ">" (L : INTEGER; R : SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L is a INTEGER and
  --         R is a SIGNED vector.

  -- Id: C.5
  function ">" (L : UNSIGNED; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L is an UNSIGNED vector and
  --         R is a nonnegative INTEGER.

  -- Id: C.6
  function ">" (L : SIGNED; R : INTEGER) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L is a SIGNED vector and
  --         R is a INTEGER.

  --============================================================================

  -- Id: C.7
  function "<" (L, R : UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.8
  function "<" (L, R : SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L and R are SIGNED vectors possibly
  --         of different lengths.

  -- Id: C.9
  function "<" (L : NATURAL; R : UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L is a nonnegative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.10
  function "<" (L : INTEGER; R : SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L is an INTEGER and
  --         R is a SIGNED vector.

  -- Id: C.11
  function "<" (L : UNSIGNED; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L is an UNSIGNED vector and
  --         R is a nonnegative INTEGER.

  -- Id: C.12
  function "<" (L : SIGNED; R : INTEGER) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L is a SIGNED vector and
  --         R is an INTEGER.

  --============================================================================

  -- Id: C.13
  function "<=" (L, R : UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.14
  function "<=" (L, R : SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L and R are SIGNED vectors possibly
  --         of different lengths.

  -- Id: C.15
  function "<=" (L : NATURAL; R : UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L is a nonnegative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.16
  function "<=" (L : INTEGER; R : SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L is an INTEGER and
  --         R is a SIGNED vector.

  -- Id: C.17
  function "<=" (L : UNSIGNED; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L is an UNSIGNED vector and
  --         R is a nonnegative INTEGER.

  -- Id: C.18
  function "<=" (L : SIGNED; R : INTEGER) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L is a SIGNED vector and
  --         R is an INTEGER.

  --============================================================================

  -- Id: C.19
  function ">=" (L, R : UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.20
  function ">=" (L, R : SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L and R are SIGNED vectors possibly
  --         of different lengths.

  -- Id: C.21
  function ">=" (L : NATURAL; R : UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L is a nonnegative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.22
  function ">=" (L : INTEGER; R : SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L is an INTEGER and
  --         R is a SIGNED vector.

  -- Id: C.23
  function ">=" (L : UNSIGNED; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L is an UNSIGNED vector and
  --         R is a nonnegative INTEGER.

  -- Id: C.24
  function ">=" (L : SIGNED; R : INTEGER) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L is a SIGNED vector and
  --         R is an INTEGER.

  --============================================================================

  -- Id: C.25
  function "=" (L, R : UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.26
  function "=" (L, R : SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L and R are SIGNED vectors possibly
  --         of different lengths.

  -- Id: C.27
  function "=" (L : NATURAL; R : UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L is a nonnegative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.28
  function "=" (L : INTEGER; R : SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L is an INTEGER and
  --         R is a SIGNED vector.

  -- Id: C.29
  function "=" (L : UNSIGNED; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L is an UNSIGNED vector and
  --         R is a nonnegative INTEGER.

  -- Id: C.30
  function "=" (L : SIGNED; R : INTEGER) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L is a SIGNED vector and
  --         R is an INTEGER.

  --============================================================================

  -- Id: C.31
  function "/=" (L, R : UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.32
  function "/=" (L, R : SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L and R are SIGNED vectors possibly
  --         of different lengths.

  -- Id: C.33
  function "/=" (L : NATURAL; R : UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L is a nonnegative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.34
  function "/=" (L : INTEGER; R : SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L is an INTEGER and
  --         R is a SIGNED vector.

  -- Id: C.35
  function "/=" (L : UNSIGNED; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L is an UNSIGNED vector and
  --         R is a nonnegative INTEGER.

  -- Id: C.36
  function "/=" (L : SIGNED; R : INTEGER) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L is a SIGNED vector and
  --         R is an INTEGER.

  --============================================================================
  -- Shift and Rotate Functions
  --============================================================================

  -- Id: S.1
  function SHIFT_LEFT (ARG : UNSIGNED; COUNT : NATURAL) return UNSIGNED;
  -- Result subtype: UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a shift-left on an UNSIGNED vector COUNT times.
  --         The vacated positions are filled with Bit '0'.
  --         The COUNT leftmost bits are lost.

  -- Id: S.2
  function SHIFT_RIGHT (ARG : UNSIGNED; COUNT : NATURAL) return UNSIGNED;
  -- Result subtype: UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a shift-right on an UNSIGNED vector COUNT times.
  --         The vacated positions are filled with Bit '0'.
  --         The COUNT rightmost bits are lost.

  -- Id: S.3
  function SHIFT_LEFT (ARG : SIGNED; COUNT : NATURAL) return SIGNED;
  -- Result subtype: SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a shift-left on a SIGNED vector COUNT times.
  --         The vacated positions are filled with Bit '0'.
  --         The COUNT leftmost bits are lost.

  -- Id: S.4
  function SHIFT_RIGHT (ARG : SIGNED; COUNT : NATURAL) return SIGNED;
  -- Result subtype: SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a shift-right on a SIGNED vector COUNT times.
  --         The vacated positions are filled with the leftmost bit, ARG'LEFT.
  --         The COUNT rightmost bits are lost.

  --============================================================================

  -- Id: S.5
  function ROTATE_LEFT (ARG : UNSIGNED; COUNT : NATURAL) return UNSIGNED;
  -- Result subtype: UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a rotate-left of an UNSIGNED vector COUNT times.

  -- Id: S.6
  function ROTATE_RIGHT (ARG : UNSIGNED; COUNT : NATURAL) return UNSIGNED;
  -- Result subtype: UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a rotate-right of an UNSIGNED vector COUNT times.

  -- Id: S.7
  function ROTATE_LEFT (ARG : SIGNED; COUNT : NATURAL) return SIGNED;
  -- Result subtype: SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a logical rotate-left of a SIGNED vector COUNT times.

  -- Id: S.8
  function ROTATE_RIGHT (ARG : SIGNED; COUNT : NATURAL) return SIGNED;
  -- Result subtype: SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a logical rotate-right of a SIGNED vector COUNT times.

  --============================================================================

  ------------------------------------------------------------------------------
  -- Note: Function S.9 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.9
  function "sll" (ARG : UNSIGNED; COUNT : INTEGER) return UNSIGNED;
  -- Result subtype: UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.10 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.10
  function "sll" (ARG : SIGNED; COUNT : INTEGER) return SIGNED;
  -- Result subtype: SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.11 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.11
  function "srl" (ARG : UNSIGNED; COUNT : INTEGER) return UNSIGNED;
  -- Result subtype: UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_RIGHT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.12 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.12
  function "srl" (ARG : SIGNED; COUNT : INTEGER) return SIGNED;
  -- Result subtype: SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SIGNED(SHIFT_RIGHT(UNSIGNED(ARG), COUNT))

  ------------------------------------------------------------------------------
  -- Note: Function S.13 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.13
  function "rol" (ARG : UNSIGNED; COUNT : INTEGER) return UNSIGNED;
  -- Result subtype: UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: ROTATE_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.14 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.14
  function "rol" (ARG : SIGNED; COUNT : INTEGER) return SIGNED;
  -- Result subtype: SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: ROTATE_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.15 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.15
  function "ror" (ARG : UNSIGNED; COUNT : INTEGER) return UNSIGNED;
  -- Result subtype: UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: ROTATE_RIGHT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.16 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.16
  function "ror" (ARG : SIGNED; COUNT : INTEGER) return SIGNED;
  -- Result subtype: SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: ROTATE_RIGHT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.17 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.17
  function "sla" (ARG : UNSIGNED; COUNT : INTEGER) return UNSIGNED;
  -- Result subtype: UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.18 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.18
  function "sla" (ARG : SIGNED; COUNT : INTEGER) return SIGNED;
  -- Result subtype: SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.19 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.19
  function "sra" (ARG : UNSIGNED; COUNT : INTEGER) return UNSIGNED;
  -- Result subtype: UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_RIGHT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.20 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.20
  function "sra" (ARG : SIGNED; COUNT : INTEGER) return SIGNED;
  -- Result subtype: SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_RIGHT(ARG, COUNT)

  --============================================================================
  -- RESIZE Functions
  --============================================================================

  -- Id: R.1
  function RESIZE (ARG : SIGNED; NEW_SIZE : NATURAL) return SIGNED;
  -- Result subtype: SIGNED(NEW_SIZE-1 downto 0)
  -- Result: Resizes the SIGNED vector ARG to the specified size.
  --         To create a larger vector, the new [leftmost] bit positions
  --         are filled with the sign bit (ARG'LEFT). When truncating,
  --         the sign bit is retained along with the rightmost part.

  -- Id: R.2
  function RESIZE (ARG : UNSIGNED; NEW_SIZE : NATURAL) return UNSIGNED;
  -- Result subtype: UNSIGNED(NEW_SIZE-1 downto 0)
  -- Result: Resizes the UNSIGNED vector ARG to the specified size.
  --         To create a larger vector, the new [leftmost] bit positions
  --         are filled with '0'. When truncating, the leftmost bits
  --         are dropped.

  function RESIZE (ARG, SIZE_RES : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED (SIZE_RES'length-1 downto 0)

  function RESIZE (ARG, SIZE_RES : SIGNED) return SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED (SIZE_RES'length-1 downto 0)

  --============================================================================
  -- Conversion Functions
  --============================================================================

  -- Id: D.1
  function TO_INTEGER (ARG : UNSIGNED) return NATURAL;
  -- Result subtype: NATURAL. Value cannot be negative since parameter is an
  --         UNSIGNED vector.
  -- Result: Converts the UNSIGNED vector to an INTEGER.

  -- Id: D.2
  function TO_INTEGER (ARG : SIGNED) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Converts a SIGNED vector to an INTEGER.

  -- Id: D.3
  function TO_UNSIGNED (ARG, SIZE : NATURAL) return UNSIGNED;
  -- Result subtype: UNSIGNED(SIZE-1 downto 0)
  -- Result: Converts a nonnegative INTEGER to an UNSIGNED vector with
  --         the specified size.

  -- Id: D.4
  function TO_SIGNED (ARG : INTEGER; SIZE : NATURAL) return SIGNED;
  -- Result subtype: SIGNED(SIZE-1 downto 0)
  -- Result: Converts an INTEGER to a SIGNED vector of the specified size.

  function TO_UNSIGNED (ARG : NATURAL; SIZE_RES : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(SIZE_RES'length-1 downto 0)

  function TO_SIGNED (ARG : INTEGER; SIZE_RES : SIGNED) return SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(SIZE_RES'length-1 downto 0)

  --============================================================================
  -- Logical Operators
  --============================================================================

  -- Id: L.1
  function "not" (L : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Termwise inversion

  -- Id: L.2
  function "and" (L, R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector AND operation

  -- Id: L.3
  function "or" (L, R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector OR operation

  -- Id: L.4
  function "nand" (L, R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector NAND operation

  -- Id: L.5
  function "nor" (L, R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector NOR operation

  -- Id: L.6
  function "xor" (L, R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector XOR operation

  ------------------------------------------------------------------------------
  -- Note: Function L.7 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: L.7
  function "xnor" (L, R : UNSIGNED) return UNSIGNED;
  -- Result subtype: UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector XNOR operation

  -- Id: L.8
  function "not" (L : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Termwise inversion

  -- Id: L.9
  function "and" (L, R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector AND operation

  -- Id: L.10
  function "or" (L, R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector OR operation

  -- Id: L.11
  function "nand" (L, R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector NAND operation

  -- Id: L.12
  function "nor" (L, R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector NOR operation

  -- Id: L.13
  function "xor" (L, R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector XOR operation

  ------------------------------------------------------------------------------
  -- Note: Function L.14 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: L.14
  function "xnor" (L, R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector XNOR operation

  -- Id: L.15
  function "and" (L : BIT; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector AND operation

  -- Id: L.16
  function "and" (L : UNSIGNED; R : BIT) return UNSIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar AND operation

  -- Id: L.17
  function "or" (L : BIT; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector OR operation

  -- Id: L.18
  function "or" (L : UNSIGNED; R : BIT) return UNSIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar OR operation

  -- Id: L.19
  function "nand" (L : BIT; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector NAND operation

  -- Id: L.20
  function "nand" (L : UNSIGNED; R : BIT) return UNSIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar NAND operation

  -- Id: L.21
  function "nor" (L : BIT; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector NOR operation

  -- Id: L.22
  function "nor" (L : UNSIGNED; R : BIT) return UNSIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar NOR operation

  -- Id: L.23
  function "xor" (L : BIT; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector XOR operation

  -- Id: L.24
  function "xor" (L : UNSIGNED; R : BIT) return UNSIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar XOR operation

  ------------------------------------------------------------------------------
  -- Note: Function L.25 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: L.25
  function "xnor" (L : BIT; R : UNSIGNED) return UNSIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector XNOR operation

  ------------------------------------------------------------------------------
  -- Note: Function L.26 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: L.26
  function "xnor" (L : UNSIGNED; R : BIT) return UNSIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar XNOR operation

  -- Id: L.27
  function "and" (L : BIT; R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector AND operation

  -- Id: L.28
  function "and" (L : SIGNED; R : BIT) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar AND operation

  -- Id: L.29
  function "or" (L : BIT; R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector OR operation

  -- Id: L.30
  function "or" (L : SIGNED; R : BIT) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar OR operation

  -- Id: L.31
  function "nand" (L : BIT; R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector NAND operation

  -- Id: L.32
  function "nand" (L : SIGNED; R : BIT) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar NAND operation

  -- Id: L.33
  function "nor" (L : BIT; R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector NOR operation

  -- Id: L.34
  function "nor" (L : SIGNED; R : BIT) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar NOR operation

  -- Id: L.35
  function "xor" (L : BIT; R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector XOR operation

  -- Id: L.36
  function "xor" (L : SIGNED; R : BIT) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar XOR operation

  ------------------------------------------------------------------------------
  -- Note: Function L.37 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: L.37
  function "xnor" (L : BIT; R : SIGNED) return SIGNED;
  -- Result subtype: SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector XNOR operation

  ------------------------------------------------------------------------------
  -- Note: Function L.38 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: L.38
  function "xnor" (L : SIGNED; R : BIT) return SIGNED;
  -- Result subtype: SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar XNOR operation


  --============================================================================
  -- string conversion and write operations
  --============================================================================
  -- the following operations are predefined

  -- explicitly defined operations

  procedure READ(L : inout LINE; VALUE : out UNSIGNED; GOOD : out BOOLEAN);

  procedure READ(L : inout LINE; VALUE : out UNSIGNED);

  procedure READ(L : inout LINE; VALUE : out SIGNED; GOOD : out BOOLEAN);

  procedure READ(L : inout LINE; VALUE : out SIGNED);

  procedure WRITE (L         : inout LINE; VALUE : in UNSIGNED;
                   JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  procedure WRITE (L         : inout LINE; VALUE : in SIGNED;
                   JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  alias BREAD is READ [LINE, UNSIGNED, BOOLEAN];
  alias BREAD is READ [LINE, SIGNED, BOOLEAN];

  alias BREAD is READ [LINE, UNSIGNED];
  alias BREAD is READ [LINE, SIGNED];

  alias BINARY_READ is READ [LINE, UNSIGNED, BOOLEAN];
  alias BINARY_READ is READ [LINE, SIGNED, BOOLEAN];

  alias BINARY_READ is READ [LINE, UNSIGNED];
  alias BINARY_READ is READ [LINE, SIGNED];

  procedure OREAD (L : inout LINE; VALUE : out UNSIGNED; GOOD : out BOOLEAN);
  procedure OREAD (L : inout LINE; VALUE : out SIGNED; GOOD : out BOOLEAN);

  procedure OREAD (L : inout LINE; VALUE : out UNSIGNED);
  procedure OREAD (L : inout LINE; VALUE : out SIGNED);

  alias OCTAL_READ is OREAD [LINE, UNSIGNED, BOOLEAN];
  alias OCTAL_READ is OREAD [LINE, SIGNED, BOOLEAN];

  alias OCTAL_READ is OREAD [LINE, UNSIGNED];
  alias OCTAL_READ is OREAD [LINE, SIGNED];

  procedure HREAD (L : inout LINE; VALUE : out UNSIGNED; GOOD : out BOOLEAN);
  procedure HREAD (L : inout LINE; VALUE : out SIGNED; GOOD : out BOOLEAN);

  procedure HREAD (L : inout LINE; VALUE : out UNSIGNED);
  procedure HREAD (L : inout LINE; VALUE : out SIGNED);

  alias HEX_READ is HREAD [LINE, UNSIGNED, BOOLEAN];
  alias HEX_READ is HREAD [LINE, SIGNED, BOOLEAN];

  alias HEX_READ is HREAD [LINE, UNSIGNED];
  alias HEX_READ is HREAD [LINE, SIGNED];

  alias BWRITE is WRITE [LINE, UNSIGNED, SIDE, WIDTH];
  alias BWRITE is WRITE [LINE, SIGNED, SIDE, WIDTH];

  alias BINARY_WRITE is WRITE [LINE, UNSIGNED, SIDE, WIDTH];
  alias BINARY_WRITE is WRITE [LINE, SIGNED, SIDE, WIDTH];

  procedure OWRITE (L         : inout LINE; VALUE : in UNSIGNED;
                    JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  procedure OWRITE (L         : inout LINE; VALUE : in SIGNED;
                    JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  alias OCTAL_WRITE is OWRITE [LINE, UNSIGNED, SIDE, WIDTH];
  alias OCTAL_WRITE is OWRITE [LINE, SIGNED, SIDE, WIDTH];

  procedure HWRITE (L         : inout LINE; VALUE : in UNSIGNED;
                    JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  procedure HWRITE (L         : inout LINE; VALUE : in SIGNED;
                    JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  alias HEX_WRITE is HWRITE [LINE, UNSIGNED, SIDE, WIDTH];
  alias HEX_WRITE is HWRITE [LINE, SIGNED, SIDE, WIDTH];

end package NUMERIC_BIT;

-- $Date: 2008-04-30 10:04:53 +0930 (Wed, 30 Apr 2008) $
use STD.TEXTIO.all;
use std.standard.all;
package std_logic_1164 is

  -------------------------------------------------------------------
  -- logic state system  (unresolved)
  -------------------------------------------------------------------
  type STD_ULOGIC is ( 'U',             -- Uninitialized
                       'X',             -- Forcing  Unknown
                       '0',             -- Forcing  0
                       '1',             -- Forcing  1
                       'Z',             -- High Impedance
                       'W',             -- Weak     Unknown
                       'L',             -- Weak     0
                       'H',             -- Weak     1
                       '-'              -- Don't care
                       );
  -------------------------------------------------------------------
  -- unconstrained array of std_ulogic for use with the resolution function
  -- and for use in declaring signal arrays of unresolved elements
  -------------------------------------------------------------------
  type STD_ULOGIC_VECTOR is array (NATURAL range <>) of STD_ULOGIC;

  -------------------------------------------------------------------
  -- resolution function
  -------------------------------------------------------------------
  function resolved (s : STD_ULOGIC_VECTOR) return STD_ULOGIC;


  -------------------------------------------------------------------
  -- logic state system  (resolved)
  -------------------------------------------------------------------
  subtype STD_LOGIC is resolved STD_ULOGIC;

  -------------------------------------------------------------------
  -- unconstrained array of resolved std_ulogic for use in declaring
  -- signal arrays of resolved elements
  -------------------------------------------------------------------
  subtype STD_LOGIC_VECTOR is (resolved) STD_ULOGIC_VECTOR;

  -------------------------------------------------------------------
  -- common subtypes
  -------------------------------------------------------------------
  subtype X01 is resolved STD_ULOGIC range 'X' to '1';    -- ('X','0','1')
  subtype X01Z is resolved STD_ULOGIC range 'X' to 'Z';   -- ('X','0','1','Z')
  subtype UX01 is resolved STD_ULOGIC range 'U' to '1';   -- ('U','X','0','1')
  subtype UX01Z is resolved STD_ULOGIC range 'U' to 'Z';  -- ('U','X','0','1','Z')

  -------------------------------------------------------------------
  -- overloaded logical operators
  -------------------------------------------------------------------

  function "and"  (l : STD_ULOGIC; r : STD_ULOGIC) return UX01;
  function "nand" (l : STD_ULOGIC; r : STD_ULOGIC) return UX01;
  function "or"   (l : STD_ULOGIC; r : STD_ULOGIC) return UX01;
  function "nor"  (l : STD_ULOGIC; r : STD_ULOGIC) return UX01;
  function "xor"  (l : STD_ULOGIC; r : STD_ULOGIC) return UX01;
  function "xnor" (l : STD_ULOGIC; r : STD_ULOGIC) return ux01;
  function "not"  (l : STD_ULOGIC) return UX01;

  -------------------------------------------------------------------
  -- vectorized overloaded logical operators
  -------------------------------------------------------------------
  function "and"  (l, r : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  function "nand" (l, r : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  function "or"   (l, r : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  function "nor"  (l, r : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  function "xor"  (l, r : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  function "xnor" (l, r : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  function "not"  (l    : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;

  function "and"  (l : STD_ULOGIC_VECTOR; r : STD_ULOGIC) return STD_ULOGIC_VECTOR;
  function "and"  (l : STD_ULOGIC; r : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;

  function "nand" (l : STD_ULOGIC_VECTOR; r : STD_ULOGIC) return STD_ULOGIC_VECTOR;
  function "nand" (l : STD_ULOGIC; r : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;

  function "or"   (l : STD_ULOGIC_VECTOR; r : STD_ULOGIC) return STD_ULOGIC_VECTOR;
  function "or"   (l : STD_ULOGIC; r : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;

  function "nor"  (l : STD_ULOGIC_VECTOR; r : STD_ULOGIC) return STD_ULOGIC_VECTOR;
  function "nor"  (l : STD_ULOGIC; r : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;

  function "xor"  (l : STD_ULOGIC_VECTOR; r : STD_ULOGIC) return STD_ULOGIC_VECTOR;
  function "xor"  (l : STD_ULOGIC; r : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;

  function "xnor" (l : STD_ULOGIC_VECTOR; r : STD_ULOGIC) return STD_ULOGIC_VECTOR;
  function "xnor" (l : STD_ULOGIC; r : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;

  -------------------------------------------------------------------
  -- shift operators
  -------------------------------------------------------------------

--  function "sll" (l : STD_ULOGIC_VECTOR; r : INTEGER) return STD_ULOGIC_VECTOR;
--  function "srl" (l : STD_ULOGIC_VECTOR; r : INTEGER) return STD_ULOGIC_VECTOR;
--  function "rol" (l : STD_ULOGIC_VECTOR; r : INTEGER) return STD_ULOGIC_VECTOR;
--  function "ror" (l : STD_ULOGIC_VECTOR; r : INTEGER) return STD_ULOGIC_VECTOR;

  -------------------------------------------------------------------
  -- conversion functions
  -------------------------------------------------------------------
  function To_bit       (s : STD_ULOGIC; xmap : BIT        := '0') return BIT;
  function To_bitvector (s : STD_ULOGIC_VECTOR; xmap : BIT := '0') return BIT_VECTOR;

  function To_StdULogic       (b : BIT) return STD_ULOGIC;
  function To_StdLogicVector  (b : BIT_VECTOR) return STD_LOGIC_VECTOR;
  function To_StdLogicVector  (s : STD_ULOGIC_VECTOR) return STD_LOGIC_VECTOR;
  function To_StdULogicVector (b : BIT_VECTOR) return STD_ULOGIC_VECTOR;
  function To_StdULogicVector (s : STD_LOGIC_VECTOR) return STD_ULOGIC_VECTOR;

  alias To_Bit_Vector is
    To_bitvector[STD_ULOGIC_VECTOR, BIT return BIT_VECTOR];
  alias To_BV is
    To_bitvector[STD_ULOGIC_VECTOR, BIT return BIT_VECTOR];

  alias To_Std_Logic_Vector is
    To_StdLogicVector[BIT_VECTOR return STD_LOGIC_VECTOR];
  alias To_SLV is
    To_StdLogicVector[BIT_VECTOR return STD_LOGIC_VECTOR];

  alias To_Std_Logic_Vector is
    To_StdLogicVector[STD_ULOGIC_VECTOR return STD_LOGIC_VECTOR];
  alias To_SLV is
    To_StdLogicVector[STD_ULOGIC_VECTOR return STD_LOGIC_VECTOR];

  alias To_Std_ULogic_Vector is
    To_StdULogicVector[BIT_VECTOR return STD_ULOGIC_VECTOR];
  alias To_SULV is
    To_StdULogicVector[BIT_VECTOR return STD_ULOGIC_VECTOR];

  alias To_Std_ULogic_Vector is
    To_StdULogicVector[STD_LOGIC_VECTOR return STD_ULOGIC_VECTOR];
  alias To_SULV is
    To_StdULogicVector[STD_LOGIC_VECTOR return STD_ULOGIC_VECTOR];

  -------------------------------------------------------------------
  -- strength strippers and type convertors
  -------------------------------------------------------------------

  function TO_01 (s : STD_ULOGIC_VECTOR; xmap : STD_ULOGIC := '0')
    return STD_ULOGIC_VECTOR;
  function TO_01 (s : STD_ULOGIC; xmap : STD_ULOGIC := '0')
    return STD_ULOGIC;
  function TO_01 (s : BIT_VECTOR; xmap : STD_ULOGIC := '0')
    return STD_ULOGIC_VECTOR;
  function TO_01 (s : BIT; xmap : STD_ULOGIC := '0')
    return STD_ULOGIC;

  function To_X01 (s : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  function To_X01 (s : STD_ULOGIC) return X01;
  function To_X01 (b : BIT_VECTOR) return STD_ULOGIC_VECTOR;
  function To_X01 (b : BIT) return X01;

  function To_X01Z (s : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  function To_X01Z (s : STD_ULOGIC) return X01Z;
  function To_X01Z (b : BIT_VECTOR) return STD_ULOGIC_VECTOR;
  function To_X01Z (b : BIT) return X01Z;

  function To_UX01 (s : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  function To_UX01 (s : STD_ULOGIC) return UX01;
  function To_UX01 (b : BIT_VECTOR) return STD_ULOGIC_VECTOR;
  function To_UX01 (b : BIT) return UX01;

  -------------------------------------------------------------------
  -- edge detection
  -------------------------------------------------------------------
  function rising_edge  (signal s : STD_ULOGIC) return BOOLEAN;
  function falling_edge (signal s : STD_ULOGIC) return BOOLEAN;

  -------------------------------------------------------------------
  -- object contains an unknown
  -------------------------------------------------------------------
  function Is_X (s : STD_ULOGIC_VECTOR) return BOOLEAN;
  function Is_X (s : STD_ULOGIC) return BOOLEAN;

  -------------------------------------------------------------------
  -- string conversion and write operations
  -------------------------------------------------------------------
  -- the following operations are predefined

  -- explicitly defined operations

  procedure READ (L : inout LINE; VALUE : out STD_ULOGIC; GOOD : out BOOLEAN);
  procedure READ (L : inout LINE; VALUE : out STD_ULOGIC);

  procedure READ (L : inout LINE; VALUE : out STD_ULOGIC_VECTOR; GOOD : out BOOLEAN);
  procedure READ (L : inout LINE; VALUE : out STD_ULOGIC_VECTOR);

  procedure WRITE (L         : inout LINE; VALUE : in STD_ULOGIC;
                   JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  procedure WRITE (L         : inout LINE; VALUE : in STD_ULOGIC_VECTOR;
                   JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  alias BREAD is READ [LINE, STD_ULOGIC_VECTOR, BOOLEAN];
  alias BREAD is READ [LINE, STD_ULOGIC_VECTOR];
  alias BINARY_READ is READ [LINE, STD_ULOGIC_VECTOR, BOOLEAN];
  alias BINARY_READ is READ [LINE, STD_ULOGIC_VECTOR];

  procedure OREAD (L : inout LINE; VALUE : out STD_ULOGIC_VECTOR; GOOD : out BOOLEAN);
  procedure OREAD (L : inout LINE; VALUE : out STD_ULOGIC_VECTOR);
  alias OCTAL_READ is OREAD [LINE, STD_ULOGIC_VECTOR, BOOLEAN];
  alias OCTAL_READ is OREAD [LINE, STD_ULOGIC_VECTOR];

  procedure HREAD (L : inout LINE; VALUE : out STD_ULOGIC_VECTOR; GOOD : out BOOLEAN);
  procedure HREAD (L : inout LINE; VALUE : out STD_ULOGIC_VECTOR);
  alias HEX_READ is HREAD [LINE, STD_ULOGIC_VECTOR, BOOLEAN];
  alias HEX_READ is HREAD [LINE, STD_ULOGIC_VECTOR];

  alias BWRITE is WRITE [LINE, STD_ULOGIC_VECTOR, SIDE, WIDTH];
  alias BINARY_WRITE is WRITE [LINE, STD_ULOGIC_VECTOR, SIDE, WIDTH];

  procedure OWRITE (L         : inout LINE; VALUE : in STD_ULOGIC_VECTOR;
                    JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);
  alias OCTAL_WRITE is OWRITE [LINE, STD_ULOGIC_VECTOR, SIDE, WIDTH];

  procedure HWRITE (L         : inout LINE; VALUE : in STD_ULOGIC_VECTOR;
                    JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);
  alias HEX_WRITE is HWRITE [LINE, STD_ULOGIC_VECTOR, SIDE, WIDTH];

end package std_logic_1164;

library ieee;
use ieee.std_logic_1164.all;


use std.standard.all;
use STD.TEXTIO.all;
library IEEE;
use IEEE.STD_LOGIC_1164.all;

package NUMERIC_STD is
  constant CopyRightNotice : STRING
    := "Copyright � 2008 IEEE. All rights reserved.";


  --============================================================================
  -- Numeric Array Type Definitions
  --============================================================================

  type UNRESOLVED_UNSIGNED is array (NATURAL range <>) of STD_ULOGIC;
  type UNRESOLVED_SIGNED is array (NATURAL range <>) of STD_ULOGIC;

  --  FIXME: was alias
  subtype U_UNSIGNED is UNRESOLVED_UNSIGNED;
  subtype U_SIGNED is UNRESOLVED_SIGNED;

  subtype UNSIGNED is (resolved) UNRESOLVED_UNSIGNED;
  subtype SIGNED is (resolved) UNRESOLVED_SIGNED;

  --============================================================================
  -- Arithmetic Operators:
  --===========================================================================

  -- Id: A.1
  function "abs" (ARG : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Returns the absolute value of an UNRESOLVED_SIGNED vector ARG.

  -- Id: A.2
  function "-" (ARG : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Returns the value of the unary minus operation on a
  --         UNRESOLVED_SIGNED vector ARG.

  --============================================================================

  -- Id: A.3
  function "+" (L, R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(MAXIMUM(L'LENGTH, R'LENGTH)-1 downto 0)
  -- Result: Adds two UNRESOLVED_UNSIGNED vectors that may be of different lengths.

  -- Id: A.3R
  function "+"(L : UNRESOLVED_UNSIGNED; R : STD_ULOGIC) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Similar to A.3 where R is a one bit UNRESOLVED_UNSIGNED

  -- Id: A.3L
  function "+"(L : STD_ULOGIC; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Similar to A.3 where L is a one bit UNRESOLVED_UNSIGNED

  -- Id: A.4
  function "+" (L, R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(MAXIMUM(L'LENGTH, R'LENGTH)-1 downto 0)
  -- Result: Adds two UNRESOLVED_SIGNED vectors that may be of different lengths.

  -- Id: A.4R
  function "+"(L : UNRESOLVED_SIGNED; R : STD_ULOGIC) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Similar to A.4 where R is bit 0 of a non-negative.

  -- Id: A.4L
  function "+"(L : STD_ULOGIC; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Similar to A.4 where L is bit 0 of a non-negative.

  -- Id: A.5
  function "+" (L : UNRESOLVED_UNSIGNED; R : NATURAL) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Adds an UNRESOLVED_UNSIGNED vector, L, with a nonnegative INTEGER, R.

  -- Id: A.6
  function "+" (L : NATURAL; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Adds a nonnegative INTEGER, L, with an UNRESOLVED_UNSIGNED vector, R.

  -- Id: A.7
  function "+" (L : INTEGER; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Adds an INTEGER, L(may be positive or negative), to an UNRESOLVED_SIGNED
  --         vector, R.

  -- Id: A.8
  function "+" (L : UNRESOLVED_SIGNED; R : INTEGER) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Adds an UNRESOLVED_SIGNED vector, L, to an INTEGER, R.

  --============================================================================

  -- Id: A.9
  function "-" (L, R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(MAXIMUM(L'LENGTH, R'LENGTH)-1 downto 0)
  -- Result: Subtracts two UNRESOLVED_UNSIGNED vectors that may be of different lengths.

  -- Id: A.9R
  function "-"(L : UNRESOLVED_UNSIGNED; R : STD_ULOGIC) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Similar to A.9 where R is a one bit UNRESOLVED_UNSIGNED

  -- Id: A.9L
  function "-"(L : STD_ULOGIC; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Similar to A.9 where L is a one bit UNRESOLVED_UNSIGNED

  -- Id: A.10
  function "-" (L, R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(MAXIMUM(L'LENGTH, R'LENGTH)-1 downto 0)
  -- Result: Subtracts an UNRESOLVED_SIGNED vector, R, from another UNRESOLVED_SIGNED vector, L,
  --         that may possibly be of different lengths.

  -- Id: A.10R
  function "-"(L : UNRESOLVED_SIGNED; R : STD_ULOGIC) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Similar to A.10 where R is bit 0 of a non-negative.

  -- Id: A.10L
  function "-"(L : STD_ULOGIC; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Similar to A.10 where R is bit 0 of a non-negative.

  -- Id: A.11
  function "-" (L : UNRESOLVED_UNSIGNED; R : NATURAL) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Subtracts a nonnegative INTEGER, R, from an UNRESOLVED_UNSIGNED vector, L.

  -- Id: A.12
  function "-" (L : NATURAL; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Subtracts an UNRESOLVED_UNSIGNED vector, R, from a nonnegative INTEGER, L.

  -- Id: A.13
  function "-" (L : UNRESOLVED_SIGNED; R : INTEGER) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Subtracts an INTEGER, R, from an UNRESOLVED_SIGNED vector, L.

  -- Id: A.14
  function "-" (L : INTEGER; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Subtracts an UNRESOLVED_SIGNED vector, R, from an INTEGER, L.

  --============================================================================

  -- Id: A.15
  function "*" (L, R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED((L'LENGTH+R'LENGTH-1) downto 0)
  -- Result: Performs the multiplication operation on two UNRESOLVED_UNSIGNED vectors
  --         that may possibly be of different lengths.

  -- Id: A.16
  function "*" (L, R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED((L'LENGTH+R'LENGTH-1) downto 0)
  -- Result: Multiplies two UNRESOLVED_SIGNED vectors that may possibly be of
  --         different lengths.

  -- Id: A.17
  function "*" (L : UNRESOLVED_UNSIGNED; R : NATURAL) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED((L'LENGTH+L'LENGTH-1) downto 0)
  -- Result: Multiplies an UNRESOLVED_UNSIGNED vector, L, with a nonnegative
  --         INTEGER, R. R is converted to an UNRESOLVED_UNSIGNED vector of
  --         SIZE L'LENGTH before multiplication.

  -- Id: A.18
  function "*" (L : NATURAL; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED((R'LENGTH+R'LENGTH-1) downto 0)
  -- Result: Multiplies an UNRESOLVED_UNSIGNED vector, R, with a nonnegative
  --         INTEGER, L. L is converted to an UNRESOLVED_UNSIGNED vector of
  --         SIZE R'LENGTH before multiplication.

  -- Id: A.19
  function "*" (L : UNRESOLVED_SIGNED; R : INTEGER) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED((L'LENGTH+L'LENGTH-1) downto 0)
  -- Result: Multiplies an UNRESOLVED_SIGNED vector, L, with an INTEGER, R. R is
  --         converted to an UNRESOLVED_SIGNED vector of SIZE L'LENGTH before
  --         multiplication.

  -- Id: A.20
  function "*" (L : INTEGER; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED((R'LENGTH+R'LENGTH-1) downto 0)
  -- Result: Multiplies an UNRESOLVED_SIGNED vector, R, with an INTEGER, L. L is
  --         converted to an UNRESOLVED_SIGNED vector of SIZE R'LENGTH before
  --         multiplication.

  --============================================================================
  --
  -- NOTE: If second argument is zero for "/" operator, a severity level
  --       of ERROR is issued.

  -- Id: A.21
  function "/" (L, R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Divides an UNRESOLVED_UNSIGNED vector, L, by another UNRESOLVED_UNSIGNED vector, R.

  -- Id: A.22
  function "/" (L, R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Divides an UNRESOLVED_SIGNED vector, L, by another UNRESOLVED_SIGNED vector, R.

  -- Id: A.23
  function "/" (L : UNRESOLVED_UNSIGNED; R : NATURAL) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Divides an UNRESOLVED_UNSIGNED vector, L, by a nonnegative INTEGER, R.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.24
  function "/" (L : NATURAL; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Divides a nonnegative INTEGER, L, by an UNRESOLVED_UNSIGNED vector, R.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  -- Id: A.25
  function "/" (L : UNRESOLVED_SIGNED; R : INTEGER) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Divides an UNRESOLVED_SIGNED vector, L, by an INTEGER, R.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.26
  function "/" (L : INTEGER; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Divides an INTEGER, L, by an UNRESOLVED_SIGNED vector, R.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  --============================================================================
  --
  -- NOTE: If second argument is zero for "rem" operator, a severity level
  --       of ERROR is issued.

  -- Id: A.27
  function "rem" (L, R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where L and R are UNRESOLVED_UNSIGNED vectors.

  -- Id: A.28
  function "rem" (L, R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where L and R are UNRESOLVED_SIGNED vectors.

  -- Id: A.29
  function "rem" (L : UNRESOLVED_UNSIGNED; R : NATURAL) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where L is an UNRESOLVED_UNSIGNED vector and R is a
  --         nonnegative INTEGER.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.30
  function "rem" (L : NATURAL; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where R is an UNRESOLVED_UNSIGNED vector and L is a
  --         nonnegative INTEGER.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  -- Id: A.31
  function "rem" (L : UNRESOLVED_SIGNED; R : INTEGER) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where L is UNRESOLVED_SIGNED vector and R is an INTEGER.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.32
  function "rem" (L : INTEGER; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where R is UNRESOLVED_SIGNED vector and L is an INTEGER.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  --============================================================================
  --
  -- NOTE: If second argument is zero for "mod" operator, a severity level
  --       of ERROR is issued.

  -- Id: A.33
  function "mod" (L, R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L and R are UNRESOLVED_UNSIGNED vectors.

  -- Id: A.34
  function "mod" (L, R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L and R are UNRESOLVED_SIGNED vectors.

  -- Id: A.35
  function "mod" (L : UNRESOLVED_UNSIGNED; R : NATURAL) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L is an UNRESOLVED_UNSIGNED vector and R
  --         is a nonnegative INTEGER.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.36
  function "mod" (L : NATURAL; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where R is an UNRESOLVED_UNSIGNED vector and L
  --         is a nonnegative INTEGER.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  -- Id: A.37
  function "mod" (L : UNRESOLVED_SIGNED; R : INTEGER) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L is an UNRESOLVED_SIGNED vector and
  --         R is an INTEGER.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.38
  function "mod" (L : INTEGER; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L is an INTEGER and
  --         R is an UNRESOLVED_SIGNED vector.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  --============================================================================
  -- Id: A.39
  function find_leftmost (ARG : UNRESOLVED_UNSIGNED; Y : STD_ULOGIC) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Finds the leftmost occurrence of the value of Y in ARG.
  --         Returns the index of the occurrence if it exists, or -1 otherwise.

  -- Id: A.40
  function find_leftmost (ARG : UNRESOLVED_SIGNED; Y : STD_ULOGIC) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Finds the leftmost occurrence of the value of Y in ARG.
  --         Returns the index of the occurrence if it exists, or -1 otherwise.

  -- Id: A.41
  function find_rightmost (ARG : UNRESOLVED_UNSIGNED; Y : STD_ULOGIC) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Finds the leftmost occurrence of the value of Y in ARG.
  --         Returns the index of the occurrence if it exists, or -1 otherwise.

  -- Id: A.42
  function find_rightmost (ARG : UNRESOLVED_SIGNED; Y : STD_ULOGIC) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Finds the leftmost occurrence of the value of Y in ARG.
  --         Returns the index of the occurrence if it exists, or -1 otherwise.

  --============================================================================
  -- Comparison Operators
  --============================================================================

  -- Id: C.1
  function ">" (L, R : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L and R are UNRESOLVED_UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.2
  function ">" (L, R : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L and R are UNRESOLVED_SIGNED vectors possibly
  --         of different lengths.

  -- Id: C.3
  function ">" (L : NATURAL; R : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L is a nonnegative INTEGER and
  --         R is an UNRESOLVED_UNSIGNED vector.

  -- Id: C.4
  function ">" (L : INTEGER; R : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L is a INTEGER and
  --         R is an UNRESOLVED_SIGNED vector.

  -- Id: C.5
  function ">" (L : UNRESOLVED_UNSIGNED; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L is an UNRESOLVED_UNSIGNED vector and
  --         R is a nonnegative INTEGER.

  -- Id: C.6
  function ">" (L : UNRESOLVED_SIGNED; R : INTEGER) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L is an UNRESOLVED_SIGNED vector and
  --         R is a INTEGER.

  --============================================================================

  -- Id: C.7
  function "<" (L, R : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L and R are UNRESOLVED_UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.8
  function "<" (L, R : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L and R are UNRESOLVED_SIGNED vectors possibly
  --         of different lengths.

  -- Id: C.9
  function "<" (L : NATURAL; R : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L is a nonnegative INTEGER and
  --         R is an UNRESOLVED_UNSIGNED vector.

  -- Id: C.10
  function "<" (L : INTEGER; R : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L is an INTEGER and
  --         R is an UNRESOLVED_SIGNED vector.

  -- Id: C.11
  function "<" (L : UNRESOLVED_UNSIGNED; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L is an UNRESOLVED_UNSIGNED vector and
  --         R is a nonnegative INTEGER.

  -- Id: C.12
  function "<" (L : UNRESOLVED_SIGNED; R : INTEGER) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L is an UNRESOLVED_SIGNED vector and
  --         R is an INTEGER.

  --============================================================================

  -- Id: C.13
  function "<=" (L, R : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L and R are UNRESOLVED_UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.14
  function "<=" (L, R : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L and R are UNRESOLVED_SIGNED vectors possibly
  --         of different lengths.

  -- Id: C.15
  function "<=" (L : NATURAL; R : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L is a nonnegative INTEGER and
  --         R is an UNRESOLVED_UNSIGNED vector.

  -- Id: C.16
  function "<=" (L : INTEGER; R : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L is an INTEGER and
  --         R is an UNRESOLVED_SIGNED vector.

  -- Id: C.17
  function "<=" (L : UNRESOLVED_UNSIGNED; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L is an UNRESOLVED_UNSIGNED vector and
  --         R is a nonnegative INTEGER.

  -- Id: C.18
  function "<=" (L : UNRESOLVED_SIGNED; R : INTEGER) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L is an UNRESOLVED_SIGNED vector and
  --         R is an INTEGER.

  --============================================================================

  -- Id: C.19
  function ">=" (L, R : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L and R are UNRESOLVED_UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.20
  function ">=" (L, R : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L and R are UNRESOLVED_SIGNED vectors possibly
  --         of different lengths.

  -- Id: C.21
  function ">=" (L : NATURAL; R : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L is a nonnegative INTEGER and
  --         R is an UNRESOLVED_UNSIGNED vector.

  -- Id: C.22
  function ">=" (L : INTEGER; R : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L is an INTEGER and
  --         R is an UNRESOLVED_SIGNED vector.

  -- Id: C.23
  function ">=" (L : UNRESOLVED_UNSIGNED; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L is an UNRESOLVED_UNSIGNED vector and
  --         R is a nonnegative INTEGER.

  -- Id: C.24
  function ">=" (L : UNRESOLVED_SIGNED; R : INTEGER) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L is an UNRESOLVED_SIGNED vector and
  --         R is an INTEGER.

  --============================================================================

  -- Id: C.25
  function "=" (L, R : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L and R are UNRESOLVED_UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.26
  function "=" (L, R : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L and R are UNRESOLVED_SIGNED vectors possibly
  --         of different lengths.

  -- Id: C.27
  function "=" (L : NATURAL; R : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L is a nonnegative INTEGER and
  --         R is an UNRESOLVED_UNSIGNED vector.

  -- Id: C.28
  function "=" (L : INTEGER; R : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L is an INTEGER and
  --         R is an UNRESOLVED_SIGNED vector.

  -- Id: C.29
  function "=" (L : UNRESOLVED_UNSIGNED; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L is an UNRESOLVED_UNSIGNED vector and
  --         R is a nonnegative INTEGER.

  -- Id: C.30
  function "=" (L : UNRESOLVED_SIGNED; R : INTEGER) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L is an UNRESOLVED_SIGNED vector and
  --         R is an INTEGER.

  --============================================================================

  -- Id: C.31
  function "/=" (L, R : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L and R are UNRESOLVED_UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.32
  function "/=" (L, R : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L and R are UNRESOLVED_SIGNED vectors possibly
  --         of different lengths.

  -- Id: C.33
  function "/=" (L : NATURAL; R : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L is a nonnegative INTEGER and
  --         R is an UNRESOLVED_UNSIGNED vector.

  -- Id: C.34
  function "/=" (L : INTEGER; R : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L is an INTEGER and
  --         R is an UNRESOLVED_SIGNED vector.

  -- Id: C.35
  function "/=" (L : UNRESOLVED_UNSIGNED; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L is an UNRESOLVED_UNSIGNED vector and
  --         R is a nonnegative INTEGER.

  -- Id: C.36
  function "/=" (L : UNRESOLVED_SIGNED; R : INTEGER) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L is an UNRESOLVED_SIGNED vector and
  --         R is an INTEGER.

  --============================================================================
  -- Shift and Rotate Functions
  --============================================================================

  -- Id: S.1
  function SHIFT_LEFT (ARG : UNRESOLVED_UNSIGNED; COUNT : NATURAL) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a shift-left on an UNRESOLVED_UNSIGNED vector COUNT times.
  --         The vacated positions are filled with '0'.
  --         The COUNT leftmost elements are lost.

  -- Id: S.2
  function SHIFT_RIGHT (ARG : UNRESOLVED_UNSIGNED; COUNT : NATURAL) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a shift-right on an UNRESOLVED_UNSIGNED vector COUNT times.
  --         The vacated positions are filled with '0'.
  --         The COUNT rightmost elements are lost.

  -- Id: S.3
  function SHIFT_LEFT (ARG : UNRESOLVED_SIGNED; COUNT : NATURAL) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a shift-left on an UNRESOLVED_SIGNED vector COUNT times.
  --         The vacated positions are filled with '0'.
  --         The COUNT leftmost elements are lost.

  -- Id: S.4
  function SHIFT_RIGHT (ARG : UNRESOLVED_SIGNED; COUNT : NATURAL) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a shift-right on an UNRESOLVED_SIGNED vector COUNT times.
  --         The vacated positions are filled with the leftmost
  --         element, ARG'LEFT. The COUNT rightmost elements are lost.

  --============================================================================

  -- Id: S.5
  function ROTATE_LEFT (ARG : UNRESOLVED_UNSIGNED; COUNT : NATURAL) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a rotate-left of an UNRESOLVED_UNSIGNED vector COUNT times.

  -- Id: S.6
  function ROTATE_RIGHT (ARG : UNRESOLVED_UNSIGNED; COUNT : NATURAL) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a rotate-right of an UNRESOLVED_UNSIGNED vector COUNT times.

  -- Id: S.7
  function ROTATE_LEFT (ARG : UNRESOLVED_SIGNED; COUNT : NATURAL) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a logical rotate-left of an UNRESOLVED_SIGNED
  --         vector COUNT times.

  -- Id: S.8
  function ROTATE_RIGHT (ARG : UNRESOLVED_SIGNED; COUNT : NATURAL) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a logical rotate-right of an UNRESOLVED_SIGNED
  --         vector COUNT times.

  --============================================================================

  --============================================================================

  ------------------------------------------------------------------------------
  --   Note: Function S.9 is not compatible with IEEE Std 1076-1987. Comment
  --   out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.9
  function "sll" (ARG : UNRESOLVED_UNSIGNED; COUNT : INTEGER) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.10 is not compatible with IEEE Std 1076-1987. Comment
  --   out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.10
  function "sll" (ARG : UNRESOLVED_SIGNED; COUNT : INTEGER) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  --   Note: Function S.11 is not compatible with IEEE Std 1076-1987. Comment
  --   out the function (declaration and body) for IEEE StdL 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.11
  function "srl" (ARG : UNRESOLVED_UNSIGNED; COUNT : INTEGER) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_RIGHT(ARG, COUNT)

  ------------------------------------------------------------------------------
  --   Note: Function S.12 is not compatible with IEEE Std 1076-1987. Comment
  --   out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.12
  function "srl" (ARG : UNRESOLVED_SIGNED; COUNT : INTEGER) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: UNRESOLVED_SIGNED(SHIFT_RIGHT(UNRESOLVED_UNSIGNED(ARG), COUNT))

  ------------------------------------------------------------------------------
  --   Note: Function S.13 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.13
  function "rol" (ARG : UNRESOLVED_UNSIGNED; COUNT : INTEGER) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: ROTATE_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  --   Note: Function S.14 is not compatible with IEEE Std 1076-1987. Comment
  --   out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.14
  function "rol" (ARG : UNRESOLVED_SIGNED; COUNT : INTEGER) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: ROTATE_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.15 is not compatible with IEEE Std 1076-1987. Comment
  --   out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.15
  function "ror" (ARG : UNRESOLVED_UNSIGNED; COUNT : INTEGER) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: ROTATE_RIGHT(ARG, COUNT)

  ------------------------------------------------------------------------------
  --   Note: Function S.16 is not compatible with IEEE Std 1076-1987. Comment
  --   out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.16
  function "ror" (ARG : UNRESOLVED_SIGNED; COUNT : INTEGER) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: ROTATE_RIGHT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.17 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.17
  function "sla" (ARG : UNRESOLVED_UNSIGNED; COUNT : INTEGER) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.18 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.18
  function "sla" (ARG : UNRESOLVED_SIGNED; COUNT : INTEGER) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.19 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.19
  function "sra" (ARG : UNRESOLVED_UNSIGNED; COUNT : INTEGER) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_RIGHT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.20 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.20
  function "sra" (ARG : UNRESOLVED_SIGNED; COUNT : INTEGER) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_RIGHT(ARG, COUNT)

  --============================================================================
  --   RESIZE Functions
  --============================================================================

  -- Id: R.1
  function RESIZE (ARG : UNRESOLVED_SIGNED; NEW_SIZE : NATURAL) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(NEW_SIZE-1 downto 0)
  -- Result: Resizes the UNRESOLVED_SIGNED vector ARG to the specified size.
  --         To create a larger vector, the new [leftmost] bit positions
  --         are filled with the sign bit (ARG'LEFT). When truncating,
  --         the sign bit is retained along with the rightmost part.

  -- Id: R.2
  function RESIZE (ARG : UNRESOLVED_UNSIGNED; NEW_SIZE : NATURAL) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(NEW_SIZE-1 downto 0)
  -- Result: Resizes the UNRESOLVED_SIGNED vector ARG to the specified size.
  --         To create a larger vector, the new [leftmost] bit positions
  --         are filled with '0'. When truncating, the leftmost bits
  --         are dropped.

  function RESIZE (ARG, SIZE_RES : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED (SIZE_RES'length-1 downto 0)

  function RESIZE (ARG, SIZE_RES : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED (SIZE_RES'length-1 downto 0)

  --============================================================================
  -- Conversion Functions
  --============================================================================

  -- Id: D.1
  function TO_INTEGER (ARG : UNRESOLVED_UNSIGNED) return NATURAL;
  -- Result subtype: NATURAL. Value cannot be negative since parameter is an
  --             UNRESOLVED_UNSIGNED vector.
  -- Result: Converts the UNRESOLVED_UNSIGNED vector to an INTEGER.

  -- Id: D.2
  function TO_INTEGER (ARG : UNRESOLVED_SIGNED) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Converts an UNRESOLVED_SIGNED vector to an INTEGER.

  -- Result subtype: UNRESOLVED_UNSIGNED(SIZE-1 downto 0)
  -- Result: Converts a nonnegative INTEGER to an UNRESOLVED_UNSIGNED vector with
  --         the specified SIZE.
  function TO_UNSIGNED (ARG, SIZE : NATURAL) return UNRESOLVED_UNSIGNED;

  -- Id: D.4
  function TO_SIGNED (ARG : INTEGER; SIZE : NATURAL) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(SIZE-1 downto 0)
  -- Result: Converts an INTEGER to a UNRESOLVED_SIGNED vector of the specified SIZE.

  -- Result subtype: UNRESOLVED_UNSIGNED(SIZE_RES'length-1 downto 0)
  function TO_UNSIGNED (ARG : NATURAL; SIZE_RES : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;

  function TO_SIGNED (ARG : INTEGER; SIZE_RES : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(SIZE_RES'length-1 downto 0)

  --============================================================================
  -- Logical Operators
  --============================================================================

  -- Id: L.1
  function "not" (L : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Termwise inversion

  -- Id: L.2
  function "and" (L, R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector AND operation

  -- Id: L.3
  function "or" (L, R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector OR operation

  -- Id: L.4
  function "nand" (L, R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector NAND operation

  -- Id: L.5
  function "nor" (L, R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector NOR operation

  -- Id: L.6
  function "xor" (L, R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector XOR operation

  -- ---------------------------------------------------------------------------
  -- Note: Function L.7 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  -- ---------------------------------------------------------------------------
  -- Id: L.7
  function "xnor" (L, R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector XNOR operation

  -- Id: L.8
  function "not" (L : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Termwise inversion

  -- Id: L.9
  function "and" (L, R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector AND operation

  -- Id: L.10
  function "or" (L, R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector OR operation

  -- Id: L.11
  function "nand" (L, R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector NAND operation

  -- Id: L.12
  function "nor" (L, R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector NOR operation

  -- Id: L.13
  function "xor" (L, R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector XOR operation

  -- ---------------------------------------------------------------------------
  -- Note: Function L.14 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  -- ---------------------------------------------------------------------------
  -- Id: L.14
  function "xnor" (L, R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector XNOR operation

  -- Id: L.15
  function "and" (L : STD_ULOGIC; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector AND operation

  -- Id: L.16
  function "and" (L : UNRESOLVED_UNSIGNED; R : STD_ULOGIC) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar AND operation

  -- Id: L.17
  function "or" (L : STD_ULOGIC; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector OR operation

  -- Id: L.18
  function "or" (L : UNRESOLVED_UNSIGNED; R : STD_ULOGIC) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar OR operation

  -- Id: L.19
  function "nand" (L : STD_ULOGIC; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector NAND operation

  -- Id: L.20
  function "nand" (L : UNRESOLVED_UNSIGNED; R : STD_ULOGIC) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar NAND operation

  -- Id: L.21
  function "nor" (L : STD_ULOGIC; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector NOR operation

  -- Id: L.22
  function "nor" (L : UNRESOLVED_UNSIGNED; R : STD_ULOGIC) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar NOR operation

  -- Id: L.23
  function "xor" (L : STD_ULOGIC; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector XOR operation

  -- Id: L.24
  function "xor" (L : UNRESOLVED_UNSIGNED; R : STD_ULOGIC) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar XOR operation

  ------------------------------------------------------------------------------
  -- Note: Function L.25 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: L.25
  function "xnor" (L : STD_ULOGIC; R : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector XNOR operation

  ------------------------------------------------------------------------------
  -- Note: Function L.26 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: L.26
  function "xnor" (L : UNRESOLVED_UNSIGNED; R : STD_ULOGIC) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar XNOR operation

  -- Id: L.27
  function "and" (L : STD_ULOGIC; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector AND operation

  -- Id: L.28
  function "and" (L : UNRESOLVED_SIGNED; R : STD_ULOGIC) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar AND operation

  -- Id: L.29
  function "or" (L : STD_ULOGIC; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector OR operation

  -- Id: L.30
  function "or" (L : UNRESOLVED_SIGNED; R : STD_ULOGIC) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar OR operation

  -- Id: L.31
  function "nand" (L : STD_ULOGIC; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector NAND operation

  -- Id: L.32
  function "nand" (L : UNRESOLVED_SIGNED; R : STD_ULOGIC) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar NAND operation

  -- Id: L.33
  function "nor" (L : STD_ULOGIC; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector NOR operation

  -- Id: L.34
  function "nor" (L : UNRESOLVED_SIGNED; R : STD_ULOGIC) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar NOR operation

  -- Id: L.35
  function "xor" (L : STD_ULOGIC; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector XOR operation

  -- Id: L.36
  function "xor" (L : UNRESOLVED_SIGNED; R : STD_ULOGIC) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar XOR operation

  ------------------------------------------------------------------------------
  -- Note: Function L.37 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: L.37
  function "xnor" (L : STD_ULOGIC; R : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(R'LENGTH-1 downto 0)
  -- Result: Scalar/Vector XNOR operation

  ------------------------------------------------------------------------------
  -- Note: Function L.38 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: L.38
  function "xnor" (L : UNRESOLVED_SIGNED; R : STD_ULOGIC) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(L'LENGTH-1 downto 0)
  -- Result: Vector/Scalar XNOR operation

  --============================================================================
  -- Match Functions
  --============================================================================

  -- Id: M.1
  function STD_MATCH (L, R : STD_ULOGIC) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: terms compared per STD_LOGIC_1164 intent

  -- Id: M.2
  function STD_MATCH (L, R : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: terms compared per STD_LOGIC_1164 intent

  -- Id: M.3
  function STD_MATCH (L, R : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: terms compared per STD_LOGIC_1164 intent

  -- Id: M.5
  function STD_MATCH (L, R : STD_ULOGIC_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: terms compared per STD_LOGIC_1164 intent

  --============================================================================
  -- Translation Functions
  --============================================================================

  -- Id: T.1
  function TO_01 (S : UNRESOLVED_UNSIGNED; XMAP : STD_ULOGIC := '0') return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(S'RANGE)
  -- Result: Termwise, 'H' is translated to '1', and 'L' is translated
  --         to '0'. If a value other than '0'|'1'|'H'|'L' is found,
  --         the array is set to (others => XMAP), and a warning is
  --         issued.

  -- Id: T.2
  function TO_01 (S : UNRESOLVED_SIGNED; XMAP : STD_ULOGIC := '0') return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(S'RANGE)
  -- Result: Termwise, 'H' is translated to '1', and 'L' is translated
  --         to '0'. If a value other than '0'|'1'|'H'|'L' is found,
  --         the array is set to (others => XMAP), and a warning is
  --         issued.

  -- Id: T.3
  function TO_X01 (S : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(S'RANGE)
  -- Result: Termwise, 'H' is translated to '1', 'L' is translated to '0',
  --         and values other than '0'|'1'|'H'|'L' are translated to 'X'.

  -- Id: T.4
  function TO_X01 (S : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(S'RANGE)
  -- Result: Termwise, 'H' is translated to '1', 'L' is translated to '0',
  --         and values other than '0'|'1'|'H'|'L' are translated to 'X'.

  -- Id: T.5
  function TO_X01Z (S : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(S'RANGE)
  -- Result: Termwise, 'H' is translated to '1', 'L' is translated to '0',
  --         and values other than '0'|'1'|'H'|'L'|'Z' are translated to 'X'.

  -- Id: T.6
  function TO_X01Z (S : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(S'RANGE)
  -- Result: Termwise, 'H' is translated to '1', 'L' is translated to '0',
  --         and values other than '0'|'1'|'H'|'L'|'Z' are translated to 'X'.

  -- Id: T.7
  function TO_UX01 (S : UNRESOLVED_UNSIGNED) return UNRESOLVED_UNSIGNED;
  -- Result subtype: UNRESOLVED_UNSIGNED(S'RANGE)
  -- Result: Termwise, 'H' is translated to '1', 'L' is translated to '0',
  --         and values other than 'U'|'0'|'1'|'H'|'L' are translated to 'X'.

  -- Id: T.8
  function TO_UX01 (S : UNRESOLVED_SIGNED) return UNRESOLVED_SIGNED;
  -- Result subtype: UNRESOLVED_SIGNED(S'RANGE)
  -- Result: Termwise, 'H' is translated to '1', 'L' is translated to '0',
  --         and values other than 'U'|'0'|'1'|'H'|'L' are translated to 'X'.

  -- Id: T.9
  function IS_X (S : UNRESOLVED_UNSIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: TRUE if S contains a 'U'|'X'|'Z'|'W'|'-' value, FALSE otherwise.

  -- Id: T.10
  function IS_X (S : UNRESOLVED_SIGNED) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: TRUE if S contains a 'U'|'X'|'Z'|'W'|'-' value, FALSE otherwise.

  --============================================================================
  -- string conversion and write operations
  --============================================================================
  -- the following operations are predefined

  -- explicitly defined operations

  procedure READ(L : inout LINE; VALUE : out UNRESOLVED_UNSIGNED; GOOD : out BOOLEAN);

  procedure READ(L : inout LINE; VALUE : out UNRESOLVED_UNSIGNED);

  procedure READ(L : inout LINE; VALUE : out UNRESOLVED_SIGNED; GOOD : out BOOLEAN);

  procedure READ(L : inout LINE; VALUE : out UNRESOLVED_SIGNED);

  procedure WRITE (L         : inout LINE; VALUE : in UNRESOLVED_UNSIGNED;
                   JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  procedure WRITE (L         : inout LINE; VALUE : in UNRESOLVED_SIGNED;
                   JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  alias BREAD is READ [LINE, UNRESOLVED_UNSIGNED, BOOLEAN];
  alias BREAD is READ [LINE, UNRESOLVED_SIGNED, BOOLEAN];

  alias BREAD is READ [LINE, UNRESOLVED_UNSIGNED];
  alias BREAD is READ [LINE, UNRESOLVED_SIGNED];

  alias BINARY_READ is READ [LINE, UNRESOLVED_UNSIGNED, BOOLEAN];
  alias BINARY_READ is READ [LINE, UNRESOLVED_SIGNED, BOOLEAN];

  alias BINARY_READ is READ [LINE, UNRESOLVED_UNSIGNED];
  alias BINARY_READ is READ [LINE, UNRESOLVED_SIGNED];

  procedure OREAD (L : inout LINE; VALUE : out UNRESOLVED_UNSIGNED; GOOD : out BOOLEAN);
  procedure OREAD (L : inout LINE; VALUE : out UNRESOLVED_SIGNED; GOOD : out BOOLEAN);

  procedure OREAD (L : inout LINE; VALUE : out UNRESOLVED_UNSIGNED);
  procedure OREAD (L : inout LINE; VALUE : out UNRESOLVED_SIGNED);

  alias OCTAL_READ is OREAD [LINE, UNRESOLVED_UNSIGNED, BOOLEAN];
  alias OCTAL_READ is OREAD [LINE, UNRESOLVED_SIGNED, BOOLEAN];

  alias OCTAL_READ is OREAD [LINE, UNRESOLVED_UNSIGNED];
  alias OCTAL_READ is OREAD [LINE, UNRESOLVED_SIGNED];

  procedure HREAD (L : inout LINE; VALUE : out UNRESOLVED_UNSIGNED; GOOD : out BOOLEAN);
  procedure HREAD (L : inout LINE; VALUE : out UNRESOLVED_SIGNED; GOOD : out BOOLEAN);

  procedure HREAD (L : inout LINE; VALUE : out UNRESOLVED_UNSIGNED);
  procedure HREAD (L : inout LINE; VALUE : out UNRESOLVED_SIGNED);

  alias HEX_READ is HREAD [LINE, UNRESOLVED_UNSIGNED, BOOLEAN];
  alias HEX_READ is HREAD [LINE, UNRESOLVED_SIGNED, BOOLEAN];

  alias HEX_READ is HREAD [LINE, UNRESOLVED_UNSIGNED];
  alias HEX_READ is HREAD [LINE, UNRESOLVED_SIGNED];

  alias BWRITE is WRITE [LINE, UNRESOLVED_UNSIGNED, SIDE, WIDTH];
  alias BWRITE is WRITE [LINE, UNRESOLVED_SIGNED, SIDE, WIDTH];

  alias BINARY_WRITE is WRITE [LINE, UNRESOLVED_UNSIGNED, SIDE, WIDTH];
  alias BINARY_WRITE is WRITE [LINE, UNRESOLVED_SIGNED, SIDE, WIDTH];

  procedure OWRITE (L         : inout LINE; VALUE : in UNRESOLVED_UNSIGNED;
                    JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  procedure OWRITE (L         : inout LINE; VALUE : in UNRESOLVED_SIGNED;
                    JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  alias OCTAL_WRITE is OWRITE [LINE, UNRESOLVED_UNSIGNED, SIDE, WIDTH];
  alias OCTAL_WRITE is OWRITE [LINE, UNRESOLVED_SIGNED, SIDE, WIDTH];

  procedure HWRITE (L         : inout LINE; VALUE : in UNRESOLVED_UNSIGNED;
                    JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  procedure HWRITE (L         : inout LINE; VALUE : in UNRESOLVED_SIGNED;
                    JUSTIFIED : in    SIDE := right; FIELD : in WIDTH := 0);

  alias HEX_WRITE is HWRITE [LINE, UNRESOLVED_UNSIGNED, SIDE, WIDTH];
  alias HEX_WRITE is HWRITE [LINE, UNRESOLVED_SIGNED, SIDE, WIDTH];

end package NUMERIC_STD;


PACKAGE BODY NUMERIC_STD IS

    -- null range array constants
    constant NAU : UNRESOLVED_UNSIGNED (0 downto 1) := (others => '0');
    constant NAS : UNRESOLVED_SIGNED (0 downto 1) := (others => '0');

    FUNCTION TO_UNSIGNED(ARG,SIZE: NATURAL) RETURN UNRESOLVED_UNSIGNED IS
        VARIABLE RESULT: UNRESOLVED_UNSIGNED (SIZE-1 DOWNTO 0) ;
        VARIABLE i_val:natural := ARG;
    BEGIN
        IF (SIZE < 1) THEN RETURN NAU; END IF;
        FOR i in 0 TO RESULT'left LOOP
            IF (i_val MOD 2) = 0 THEN
                RESULT(i) := '0';
            ELSE
                RESULT(i) := '1' ;
            END IF;
            i_val := i_val/2 ;
        END LOOP;
        IF NOT(i_val=0) THEN
            report "numeric_std.TO_UNSIGNED : vector truncated"
            severity WARNING ;
        END IF;
        RETURN RESULT;
    END TO_UNSIGNED;
    --------------------------------------------------------------------
    FUNCTION TO_SIGNED(ARG: INTEGER; SIZE: NATURAL) RETURN UNRESOLVED_SIGNED IS
    VARIABLE RESULT: UNRESOLVED_SIGNED (SIZE-1 DOWNTO 0) ;
    VARIABLE b_val : STD_LOGIC:= '0' ;
    VARIABLE i_val : INTEGER:= ARG ;
    BEGIN
    IF (SIZE < 1) THEN RETURN NAS; END IF;
    IF (ARG<0) THEN
        b_val := '1' ;
        i_val := -(ARG+1) ;
    END IF ;
    for i IN 0 TO RESULT'left LOOP
        IF (i_val MOD 2) = 0 THEN
            RESULT(i) := b_val;
        else 
            RESULT(i) := NOT b_val ;
        END IF;
        i_val := i_val/2 ;
    END LOOP;
    IF ((i_val/=0) or (b_val/=RESULT(RESULT'left))) THEN
        report "numeric_std.TO_SIGNED : vector truncated"
        severity WARNING ;
    END IF;
    RETURN RESULT;
    END TO_SIGNED;


END NUMERIC_STD;


library ieee;
use ieee.std_logic_1164.all;
use std.standard.all;
package std_logic_arith is

    type UNSIGNED is array (NATURAL range <>) of STD_LOGIC;
    type SIGNED is array (NATURAL range <>) of STD_LOGIC;
    subtype SMALL_INT is INTEGER range 0 to 1;

    function "+"(L: UNSIGNED; R: UNSIGNED) return UNSIGNED;
    function "+"(L: SIGNED; R: SIGNED) return SIGNED;
    function "+"(L: UNSIGNED; R: SIGNED) return SIGNED;
    function "+"(L: SIGNED; R: UNSIGNED) return SIGNED;
    function "+"(L: UNSIGNED; R: INTEGER) return UNSIGNED;
    function "+"(L: INTEGER; R: UNSIGNED) return UNSIGNED;
    function "+"(L: SIGNED; R: INTEGER) return SIGNED;
    function "+"(L: INTEGER; R: SIGNED) return SIGNED;
    function "+"(L: UNSIGNED; R: STD_ULOGIC) return UNSIGNED;
    function "+"(L: STD_ULOGIC; R: UNSIGNED) return UNSIGNED;
    function "+"(L: SIGNED; R: STD_ULOGIC) return SIGNED;
    function "+"(L: STD_ULOGIC; R: SIGNED) return SIGNED;

    function "+"(L: UNSIGNED; R: UNSIGNED) return STD_LOGIC_VECTOR;
    function "+"(L: SIGNED; R: SIGNED) return STD_LOGIC_VECTOR;
    function "+"(L: UNSIGNED; R: SIGNED) return STD_LOGIC_VECTOR;
    function "+"(L: SIGNED; R: UNSIGNED) return STD_LOGIC_VECTOR;
    function "+"(L: UNSIGNED; R: INTEGER) return STD_LOGIC_VECTOR;
    function "+"(L: INTEGER; R: UNSIGNED) return STD_LOGIC_VECTOR;
    function "+"(L: SIGNED; R: INTEGER) return STD_LOGIC_VECTOR;
    function "+"(L: INTEGER; R: SIGNED) return STD_LOGIC_VECTOR;
    function "+"(L: UNSIGNED; R: STD_ULOGIC) return STD_LOGIC_VECTOR;
    function "+"(L: STD_ULOGIC; R: UNSIGNED) return STD_LOGIC_VECTOR;
    function "+"(L: SIGNED; R: STD_ULOGIC) return STD_LOGIC_VECTOR;
    function "+"(L: STD_ULOGIC; R: SIGNED) return STD_LOGIC_VECTOR;

    function "-"(L: UNSIGNED; R: UNSIGNED) return UNSIGNED;
    function "-"(L: SIGNED; R: SIGNED) return SIGNED;
    function "-"(L: UNSIGNED; R: SIGNED) return SIGNED;
    function "-"(L: SIGNED; R: UNSIGNED) return SIGNED;
    function "-"(L: UNSIGNED; R: INTEGER) return UNSIGNED;
    function "-"(L: INTEGER; R: UNSIGNED) return UNSIGNED;
    function "-"(L: SIGNED; R: INTEGER) return SIGNED;
    function "-"(L: INTEGER; R: SIGNED) return SIGNED;
    function "-"(L: UNSIGNED; R: STD_ULOGIC) return UNSIGNED;
    function "-"(L: STD_ULOGIC; R: UNSIGNED) return UNSIGNED;
    function "-"(L: SIGNED; R: STD_ULOGIC) return SIGNED;
    function "-"(L: STD_ULOGIC; R: SIGNED) return SIGNED;

    function "-"(L: UNSIGNED; R: UNSIGNED) return STD_LOGIC_VECTOR;
    function "-"(L: SIGNED; R: SIGNED) return STD_LOGIC_VECTOR;
    function "-"(L: UNSIGNED; R: SIGNED) return STD_LOGIC_VECTOR;
    function "-"(L: SIGNED; R: UNSIGNED) return STD_LOGIC_VECTOR;
    function "-"(L: UNSIGNED; R: INTEGER) return STD_LOGIC_VECTOR;
    function "-"(L: INTEGER; R: UNSIGNED) return STD_LOGIC_VECTOR;
    function "-"(L: SIGNED; R: INTEGER) return STD_LOGIC_VECTOR;
    function "-"(L: INTEGER; R: SIGNED) return STD_LOGIC_VECTOR;
    function "-"(L: UNSIGNED; R: STD_ULOGIC) return STD_LOGIC_VECTOR;
    function "-"(L: STD_ULOGIC; R: UNSIGNED) return STD_LOGIC_VECTOR;
    function "-"(L: SIGNED; R: STD_ULOGIC) return STD_LOGIC_VECTOR;
    function "-"(L: STD_ULOGIC; R: SIGNED) return STD_LOGIC_VECTOR;

    function "+"(L: UNSIGNED) return UNSIGNED;
    function "+"(L: SIGNED) return SIGNED;
    function "-"(L: SIGNED) return SIGNED;
    function "ABS"(L: SIGNED) return SIGNED;

    function "+"(L: UNSIGNED) return STD_LOGIC_VECTOR;
    function "+"(L: SIGNED) return STD_LOGIC_VECTOR;
    function "-"(L: SIGNED) return STD_LOGIC_VECTOR;
    function "ABS"(L: SIGNED) return STD_LOGIC_VECTOR;

    function "*"(L: UNSIGNED; R: UNSIGNED) return UNSIGNED;
    function "*"(L: SIGNED; R: SIGNED) return SIGNED;
    function "*"(L: SIGNED; R: UNSIGNED) return SIGNED;
    function "*"(L: UNSIGNED; R: SIGNED) return SIGNED;

    function "*"(L: UNSIGNED; R: UNSIGNED) return STD_LOGIC_VECTOR;
    function "*"(L: SIGNED; R: SIGNED) return STD_LOGIC_VECTOR;
    function "*"(L: SIGNED; R: UNSIGNED) return STD_LOGIC_VECTOR;
    function "*"(L: UNSIGNED; R: SIGNED) return STD_LOGIC_VECTOR;

    function "<"(L: UNSIGNED; R: UNSIGNED) return BOOLEAN;
    function "<"(L: SIGNED; R: SIGNED) return BOOLEAN;
    function "<"(L: UNSIGNED; R: SIGNED) return BOOLEAN;
    function "<"(L: SIGNED; R: UNSIGNED) return BOOLEAN;
    function "<"(L: UNSIGNED; R: INTEGER) return BOOLEAN;
    function "<"(L: INTEGER; R: UNSIGNED) return BOOLEAN;
    function "<"(L: SIGNED; R: INTEGER) return BOOLEAN;
    function "<"(L: INTEGER; R: SIGNED) return BOOLEAN;

    function "<="(L: UNSIGNED; R: UNSIGNED) return BOOLEAN;
    function "<="(L: SIGNED; R: SIGNED) return BOOLEAN;
    function "<="(L: UNSIGNED; R: SIGNED) return BOOLEAN;
    function "<="(L: SIGNED; R: UNSIGNED) return BOOLEAN;
    function "<="(L: UNSIGNED; R: INTEGER) return BOOLEAN;
    function "<="(L: INTEGER; R: UNSIGNED) return BOOLEAN;
    function "<="(L: SIGNED; R: INTEGER) return BOOLEAN;
    function "<="(L: INTEGER; R: SIGNED) return BOOLEAN;

    function ">"(L: UNSIGNED; R: UNSIGNED) return BOOLEAN;
    function ">"(L: SIGNED; R: SIGNED) return BOOLEAN;
    function ">"(L: UNSIGNED; R: SIGNED) return BOOLEAN;

    function ">"(L: SIGNED; R: UNSIGNED) return BOOLEAN;
    function ">"(L: UNSIGNED; R: INTEGER) return BOOLEAN;
    function ">"(L: INTEGER; R: UNSIGNED) return BOOLEAN;
    function ">"(L: SIGNED; R: INTEGER) return BOOLEAN;
    function ">"(L: INTEGER; R: SIGNED) return BOOLEAN;

    function ">="(L: UNSIGNED; R: UNSIGNED) return BOOLEAN;
    function ">="(L: SIGNED; R: SIGNED) return BOOLEAN;
    function ">="(L: UNSIGNED; R: SIGNED) return BOOLEAN;
    function ">="(L: SIGNED; R: UNSIGNED) return BOOLEAN;
    function ">="(L: UNSIGNED; R: INTEGER) return BOOLEAN;
    function ">="(L: INTEGER; R: UNSIGNED) return BOOLEAN;
    function ">="(L: SIGNED; R: INTEGER) return BOOLEAN;
    function ">="(L: INTEGER; R: SIGNED) return BOOLEAN;

    function "="(L: UNSIGNED; R: UNSIGNED) return BOOLEAN;
    function "="(L: SIGNED; R: SIGNED) return BOOLEAN;
    function "="(L: UNSIGNED; R: SIGNED) return BOOLEAN;
    function "="(L: SIGNED; R: UNSIGNED) return BOOLEAN;
    function "="(L: UNSIGNED; R: INTEGER) return BOOLEAN;
    function "="(L: INTEGER; R: UNSIGNED) return BOOLEAN;
    function "="(L: SIGNED; R: INTEGER) return BOOLEAN;
    function "="(L: INTEGER; R: SIGNED) return BOOLEAN;

    function "/="(L: UNSIGNED; R: UNSIGNED) return BOOLEAN;
    function "/="(L: SIGNED; R: SIGNED) return BOOLEAN;
    function "/="(L: UNSIGNED; R: SIGNED) return BOOLEAN;
    function "/="(L: SIGNED; R: UNSIGNED) return BOOLEAN;
    function "/="(L: UNSIGNED; R: INTEGER) return BOOLEAN;
    function "/="(L: INTEGER; R: UNSIGNED) return BOOLEAN;
    function "/="(L: SIGNED; R: INTEGER) return BOOLEAN;
    function "/="(L: INTEGER; R: SIGNED) return BOOLEAN;

    function SHL(ARG: UNSIGNED; COUNT: UNSIGNED) return UNSIGNED;
    function SHL(ARG: SIGNED; COUNT: UNSIGNED) return SIGNED;
    function SHR(ARG: UNSIGNED; COUNT: UNSIGNED) return UNSIGNED;
    function SHR(ARG: SIGNED; COUNT: UNSIGNED) return SIGNED;

    function CONV_INTEGER(ARG: INTEGER) return INTEGER;
    function CONV_INTEGER(ARG: UNSIGNED) return INTEGER;
    function CONV_INTEGER(ARG: SIGNED) return INTEGER;
    function CONV_INTEGER(ARG: STD_ULOGIC) return SMALL_INT;

    function CONV_UNSIGNED(ARG: INTEGER; SIZE: INTEGER) return UNSIGNED;
    function CONV_UNSIGNED(ARG: UNSIGNED; SIZE: INTEGER) return UNSIGNED;
    function CONV_UNSIGNED(ARG: SIGNED; SIZE: INTEGER) return UNSIGNED;
    function CONV_UNSIGNED(ARG: STD_ULOGIC; SIZE: INTEGER) return UNSIGNED;

    function CONV_SIGNED(ARG: INTEGER; SIZE: INTEGER) return SIGNED;
    function CONV_SIGNED(ARG: UNSIGNED; SIZE: INTEGER) return SIGNED;
    function CONV_SIGNED(ARG: SIGNED; SIZE: INTEGER) return SIGNED;
    function CONV_SIGNED(ARG: STD_ULOGIC; SIZE: INTEGER) return SIGNED;

    function CONV_STD_LOGIC_VECTOR(ARG: INTEGER; SIZE: INTEGER)
						       return STD_LOGIC_VECTOR;
    function CONV_STD_LOGIC_VECTOR(ARG: UNSIGNED; SIZE: INTEGER)
						       return STD_LOGIC_VECTOR;
    function CONV_STD_LOGIC_VECTOR(ARG: SIGNED; SIZE: INTEGER)
						       return STD_LOGIC_VECTOR;
    function CONV_STD_LOGIC_VECTOR(ARG: STD_ULOGIC; SIZE: INTEGER)
						       return STD_LOGIC_VECTOR;
    -- zero extend STD_LOGIC_VECTOR (ARG) to SIZE,
    -- SIZE < 0 is same as SIZE = 0
    -- returns STD_LOGIC_VECTOR(SIZE-1 downto 0)
    function EXT(ARG: STD_LOGIC_VECTOR; SIZE: INTEGER) return STD_LOGIC_VECTOR;

    -- sign extend STD_LOGIC_VECTOR (ARG) to SIZE,
    -- SIZE < 0 is same as SIZE = 0
    -- return STD_LOGIC_VECTOR(SIZE-1 downto 0)
    function SXT(ARG: STD_LOGIC_VECTOR; SIZE: INTEGER) return STD_LOGIC_VECTOR;

end std_logic_arith;

library ieee;
use ieee.std_logic_1164.all;
use std.standard.all;
package std_logic_signed is

    function "+"(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;
    function "+"(L: STD_LOGIC_VECTOR; R: INTEGER) return STD_LOGIC_VECTOR;
    function "+"(L: INTEGER; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;
    function "+"(L: STD_LOGIC_VECTOR; R: STD_LOGIC) return STD_LOGIC_VECTOR;
    function "+"(L: STD_LOGIC; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;

    function "-"(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;
    function "-"(L: STD_LOGIC_VECTOR; R: INTEGER) return STD_LOGIC_VECTOR;
    function "-"(L: INTEGER; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;
    function "-"(L: STD_LOGIC_VECTOR; R: STD_LOGIC) return STD_LOGIC_VECTOR;
    function "-"(L: STD_LOGIC; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;

    function "+"(L: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;
    function "-"(L: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;
    function "ABS"(L: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;

    function "*"(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;

    function "<"(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return BOOLEAN;
    function "<"(L: STD_LOGIC_VECTOR; R: INTEGER) return BOOLEAN;
    function "<"(L: INTEGER; R: STD_LOGIC_VECTOR) return BOOLEAN;

    function "<="(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return BOOLEAN;
    function "<="(L: STD_LOGIC_VECTOR; R: INTEGER) return BOOLEAN;
    function "<="(L: INTEGER; R: STD_LOGIC_VECTOR) return BOOLEAN;

    function ">"(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return BOOLEAN;
    function ">"(L: STD_LOGIC_VECTOR; R: INTEGER) return BOOLEAN;
    function ">"(L: INTEGER; R: STD_LOGIC_VECTOR) return BOOLEAN;

    function ">="(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return BOOLEAN;
    function ">="(L: STD_LOGIC_VECTOR; R: INTEGER) return BOOLEAN;
    function ">="(L: INTEGER; R: STD_LOGIC_VECTOR) return BOOLEAN;

    function "="(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return BOOLEAN;
    function "="(L: STD_LOGIC_VECTOR; R: INTEGER) return BOOLEAN;
    function "="(L: INTEGER; R: STD_LOGIC_VECTOR) return BOOLEAN;

    function "/="(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return BOOLEAN;
    function "/="(L: STD_LOGIC_VECTOR; R: INTEGER) return BOOLEAN;
    function "/="(L: INTEGER; R: STD_LOGIC_VECTOR) return BOOLEAN;

    function SHL(ARG:STD_LOGIC_VECTOR;COUNT: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;
    function SHR(ARG:STD_LOGIC_VECTOR;COUNT: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;

    function CONV_INTEGER(ARG: STD_LOGIC_VECTOR) return INTEGER;

end std_logic_signed;

library ieee;
use ieee.std_logic_1164.all;
use std.standard.all;
package std_logic_unsigned is

    function "+"(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;
    function "+"(L: STD_LOGIC_VECTOR; R: INTEGER) return STD_LOGIC_VECTOR;
    function "+"(L: INTEGER; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;
    function "+"(L: STD_LOGIC_VECTOR; R: STD_LOGIC) return STD_LOGIC_VECTOR;
    function "+"(L: STD_LOGIC; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;

    function "-"(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;
    function "-"(L: STD_LOGIC_VECTOR; R: INTEGER) return STD_LOGIC_VECTOR;
    function "-"(L: INTEGER; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;
    function "-"(L: STD_LOGIC_VECTOR; R: STD_LOGIC) return STD_LOGIC_VECTOR;
    function "-"(L: STD_LOGIC; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;

    function "+"(L: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;

    function "*"(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;

    function "<"(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return BOOLEAN;
    function "<"(L: STD_LOGIC_VECTOR; R: INTEGER) return BOOLEAN;
    function "<"(L: INTEGER; R: STD_LOGIC_VECTOR) return BOOLEAN;

    function "<="(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return BOOLEAN;
    function "<="(L: STD_LOGIC_VECTOR; R: INTEGER) return BOOLEAN;
    function "<="(L: INTEGER; R: STD_LOGIC_VECTOR) return BOOLEAN;

    function ">"(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return BOOLEAN;
    function ">"(L: STD_LOGIC_VECTOR; R: INTEGER) return BOOLEAN;
    function ">"(L: INTEGER; R: STD_LOGIC_VECTOR) return BOOLEAN;

    function ">="(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return BOOLEAN;
    function ">="(L: STD_LOGIC_VECTOR; R: INTEGER) return BOOLEAN;
    function ">="(L: INTEGER; R: STD_LOGIC_VECTOR) return BOOLEAN;

    function "="(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return BOOLEAN;
    function "="(L: STD_LOGIC_VECTOR; R: INTEGER) return BOOLEAN;
    function "="(L: INTEGER; R: STD_LOGIC_VECTOR) return BOOLEAN;

    function "/="(L: STD_LOGIC_VECTOR; R: STD_LOGIC_VECTOR) return BOOLEAN;
    function "/="(L: STD_LOGIC_VECTOR; R: INTEGER) return BOOLEAN;
    function "/="(L: INTEGER; R: STD_LOGIC_VECTOR) return BOOLEAN;
    function SHL(ARG:STD_LOGIC_VECTOR;COUNT: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;
    function SHR(ARG:STD_LOGIC_VECTOR;COUNT: STD_LOGIC_VECTOR) return STD_LOGIC_VECTOR;

    function CONV_INTEGER(ARG: STD_LOGIC_VECTOR) return INTEGER;

end std_logic_unsigned;

use std.standard.all;
use STD.TEXTIO.all;
use IEEE.std_logic_1164.all;
PACKAGE std_logic_textio IS
-- $Date: 2008-06-27 14:28:49 +0930 (Fri, 27 Jun 2008) $

  alias READ  is IEEE.std_logic_1164.READ [LINE, STD_ULOGIC];
  alias READ  is IEEE.std_logic_1164.READ [LINE, STD_ULOGIC, BOOLEAN];
  alias READ  is IEEE.std_logic_1164.READ [LINE, STD_ULOGIC_VECTOR];
  alias READ  is IEEE.std_logic_1164.READ [LINE, STD_ULOGIC_VECTOR, BOOLEAN];
  alias WRITE is IEEE.std_logic_1164.WRITE [LINE, STD_ULOGIC, SIDE, WIDTH];
  alias WRITE is IEEE.std_logic_1164.WRITE [LINE, STD_ULOGIC_VECTOR, SIDE, WIDTH];

  alias HREAD  is IEEE.std_logic_1164.HREAD [LINE, STD_ULOGIC_VECTOR];
  alias HREAD  is IEEE.std_logic_1164.HREAD [LINE, STD_ULOGIC_VECTOR, BOOLEAN];
  alias HWRITE is IEEE.std_logic_1164.HWRITE [LINE, STD_ULOGIC_VECTOR, SIDE, WIDTH];

  alias OREAD  is IEEE.std_logic_1164.OREAD [LINE, STD_ULOGIC_VECTOR];
  alias OREAD  is IEEE.std_logic_1164.OREAD [LINE, STD_ULOGIC_VECTOR, BOOLEAN];
  alias OWRITE is IEEE.std_logic_1164.OWRITE [LINE, STD_ULOGIC_VECTOR, SIDE, WIDTH];

END PACKAGE std_logic_textio;

use std.standard.all;
library ieee;
use ieee.std_logic_1164.all;
package std_logic_misc is

    type STRENGTH is (strn_X01, strn_X0H, strn_XL1, strn_X0Z, strn_XZ1, strn_WLH, strn_WLZ, strn_WZH, strn_W0H, strn_WL1);

    type MINOMAX is array (1 to 3) of TIME;

    function strength_map(input: STD_ULOGIC; strn: STRENGTH) return STD_LOGIC;
    function strength_map_z(input:STD_ULOGIC; strn:STRENGTH) return STD_LOGIC;
    function Drive (V: STD_ULOGIC_VECTOR) return STD_LOGIC_VECTOR;
    function Drive (V: STD_LOGIC_VECTOR) return STD_ULOGIC_VECTOR;
    function Sense (V: STD_ULOGIC; vZ, vU, vDC: STD_ULOGIC) return STD_LOGIC;
    function Sense (V: STD_ULOGIC_VECTOR; vZ, vU, vDC: STD_ULOGIC) return STD_ULOGIC_VECTOR;
    function Sense (V: STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
    function Sense (V: STD_LOGIC_VECTOR; vZ, vU, vDC: STD_ULOGIC) return STD_LOGIC_VECTOR;
    function Sense (V: STD_LOGIC_VECTOR) return STD_ULOGIC_VECTOR;

    function STD_LOGIC_VECTORtoBIT_VECTOR (V: STD_LOGIC_VECTOR; vX, vZ, vU, vDC: BIT := '0'; Xflag, Zflag, Uflag, DCflag: BOOLEAN := FALSE) return BIT_VECTOR;
    function STD_ULOGIC_VECTORtoBIT_VECTOR (V: STD_ULOGIC_VECTOR; vX, vZ, vU, vDC: BIT := '0'; Xflag, Zflag, Uflag, DCflag: BOOLEAN := FALSE) return BIT_VECTOR;
    function STD_ULOGICtoBIT (V: STD_ULOGIC; vX, vZ, vU, vDC: BIT := '0'; Xflag, Zflag, Uflag, DCflag: BOOLEAN := FALSE) return BIT;

    function AND_REDUCE(ARG: STD_LOGIC_VECTOR) return UX01;
    function NAND_REDUCE(ARG: STD_LOGIC_VECTOR) return UX01;
    function OR_REDUCE(ARG: STD_LOGIC_VECTOR) return UX01;
    function NOR_REDUCE(ARG: STD_LOGIC_VECTOR) return UX01;
    function XOR_REDUCE(ARG: STD_LOGIC_VECTOR) return UX01;
    function XNOR_REDUCE(ARG: STD_LOGIC_VECTOR) return UX01;

    function AND_REDUCE(ARG: STD_ULOGIC_VECTOR) return UX01;
    function NAND_REDUCE(ARG: STD_ULOGIC_VECTOR) return UX01;
    function OR_REDUCE(ARG: STD_ULOGIC_VECTOR) return UX01;
    function NOR_REDUCE(ARG: STD_ULOGIC_VECTOR) return UX01;
    function XOR_REDUCE(ARG: STD_ULOGIC_VECTOR) return UX01;
    function XNOR_REDUCE(ARG: STD_ULOGIC_VECTOR) return UX01;

    function fun_BUF3S(Input, Enable: UX01; Strn: STRENGTH) return STD_LOGIC;
    function fun_BUF3SL(Input, Enable: UX01; Strn: STRENGTH) return STD_LOGIC;
    function fun_MUX2x1(Input0, Input1, Sel: UX01) return UX01;

    function fun_MAJ23(Input0, Input1, Input2: UX01) return UX01;
    function fun_WiredX(Input0, Input1: std_ulogic) return STD_LOGIC;

end;

use std.standard.all;
library ieee;
use ieee.std_logic_1164.all;
PACKAGE VITAL_Timing IS
    TYPE VitalTransitionType IS ( tr01, tr10, tr0z, trz1, tr1z, trz0,
                                  tr0X, trx1, tr1x, trx0, trxz, trzx);

    SUBTYPE VitalDelayType     IS TIME;
    TYPE VitalDelayType01   IS ARRAY (VitalTransitionType   RANGE tr01 to tr10)
         OF TIME;
    TYPE VitalDelayType01Z  IS ARRAY (VitalTransitionType   RANGE tr01 to trz0)
         OF TIME;
    TYPE VitalDelayType01ZX IS ARRAY (VitalTransitionType   RANGE tr01 to trzx)
         OF TIME;

    TYPE VitalDelayArrayType     IS ARRAY (NATURAL RANGE <>) OF VitalDelayType;
    TYPE VitalDelayArrayType01   IS ARRAY (NATURAL RANGE <>) OF VitalDelayType01;
    TYPE VitalDelayArrayType01Z  IS ARRAY (NATURAL RANGE <>) OF VitalDelayType01Z;
    TYPE VitalDelayArrayType01ZX IS ARRAY (NATURAL RANGE <>) OF VitalDelayType01ZX;
    -- ----------------------------------------------------------------------
    -- **********************************************************************
    -- ----------------------------------------------------------------------

    CONSTANT VitalZeroDelay     : VitalDelayType     :=   0 ns;
    CONSTANT VitalZeroDelay01   : VitalDelayType01   := ( 0 ns, 0 ns );
    CONSTANT VitalZeroDelay01Z  : VitalDelayType01Z  := ( OTHERS => 0 ns );
    CONSTANT VitalZeroDelay01ZX : VitalDelayType01ZX := ( OTHERS => 0 ns );

    ---------------------------------------------------------------------------
    -- examples of usage:
    ---------------------------------------------------------------------------
    -- tpd_CLK_Q : VitalDelayType  := 5 ns;
    -- tpd_CLK_Q : VitalDelayType01  := (tr01 => 2 ns, tr10 => 3 ns);
    -- tpd_CLK_Q : VitalDelayType01Z := ( 1 ns, 2 ns, 3 ns, 4 ns, 5 ns, 6 ns );
    -- tpd_CLK_Q : VitalDelayArrayType(0 to 1)
    --                          := (0 => 5 ns, 1 => 6 ns);
    -- tpd_CLK_Q : VitalDelayArrayType01(0 to 1)
    --                          := (0 => (tr01 => 2 ns, tr10 => 3 ns),
    --                              1 => (tr01 => 2 ns, tr10 => 3 ns));
    -- tpd_CLK_Q : VitalDelayArrayType01Z(0 to 1)
    --                        := (0 => ( 1 ns, 2 ns, 3 ns, 4 ns, 5 ns, 6 ns ),
    --                            1 => ( 1 ns, 2 ns, 3 ns, 4 ns, 5 ns, 6 ns ));
    ---------------------------------------------------------------------------

    -- TRUE if the model is LEVEL0 | LEVEL1 compliant
    ATTRIBUTE VITAL_Level0 : BOOLEAN;
    ATTRIBUTE VITAL_Level1 : BOOLEAN;

    SUBTYPE std_logic_vector2 IS std_logic_vector(1 DOWNTO 0);
    SUBTYPE std_logic_vector3 IS std_logic_vector(2 DOWNTO 0);
    SUBTYPE std_logic_vector4 IS std_logic_vector(3 DOWNTO 0);
    SUBTYPE std_logic_vector8 IS std_logic_vector(7 DOWNTO 0);

    -- Types for strength mapping of outputs
    TYPE VitalOutputMapType  IS ARRAY ( std_ulogic ) OF std_ulogic;
    TYPE VitalResultMapType  IS ARRAY ( UX01       ) OF std_ulogic;
    TYPE VitalResultZMapType IS ARRAY ( UX01Z      ) OF std_ulogic;
    CONSTANT VitalDefaultOutputMap  : VitalOutputMapType
                                    := "UX01ZWLH-";
    CONSTANT VitalDefaultResultMap  : VitalResultMapType
                                    := ( 'U', 'X', '0', '1' );
    CONSTANT VitalDefaultResultZMap : VitalResultZMapType
                                    := ( 'U', 'X', '0', '1', 'Z' );

    -- Types for fields of VitalTimingDataType
    TYPE VitalTimeArrayT  IS ARRAY (INTEGER RANGE <>) OF TIME;
    TYPE VitalTimeArrayPT IS ACCESS VitalTimeArrayT;
    TYPE VitalBoolArrayT  IS ARRAY (INTEGER RANGE <>) OF BOOLEAN;
    TYPE VitalBoolArrayPT IS ACCESS VitalBoolArrayT;
    TYPE VitalLogicArrayPT IS ACCESS std_logic_vector;

    TYPE VitalTimingDataType IS RECORD
        NotFirstFlag : BOOLEAN;
        RefLast   : X01;
        RefTime   : TIME;
        HoldEn    : BOOLEAN;
        TestLast  : std_ulogic;
        TestTime  : TIME;
        SetupEn   : BOOLEAN;
        TestLastA : VitalLogicArrayPT;
        TestTimeA : VitalTimeArrayPT;
        HoldEnA   : VitalBoolArrayPT;
        SetupEnA  : VitalBoolArrayPT;
    END RECORD;

    FUNCTION VitalTimingDataInit RETURN VitalTimingDataType;

    -- type for internal data of VitalPeriodPulseCheck
    TYPE VitalPeriodDataType IS RECORD
        Last : X01;
        Rise : TIME;
        Fall : TIME;
        NotFirstFlag : BOOLEAN;
    END RECORD;
    CONSTANT VitalPeriodDataInit : VitalPeriodDataType
                                 := ('X', 0 ns, 0 ns, FALSE );

    -- Type for specifying the kind of Glitch handling to use
    TYPE VitalGlitchKindType IS (OnEvent,
                                 OnDetect,
                                 VitalInertial,
                                 VitalTransport);

    TYPE VitalGlitchDataType IS
      RECORD
        SchedTime    : TIME;
        GlitchTime   : TIME;
        SchedValue   : std_ulogic;
        LastValue    : std_ulogic;
      END RECORD;
    TYPE VitalGlitchDataArrayType IS ARRAY (NATURAL RANGE <>)
         OF VitalGlitchDataType;

    -- PathTypes: for handling simple PathDelay info
    TYPE VitalPathType IS RECORD
        InputChangeTime : TIME;             -- timestamp for path input signal
        PathDelay       : VitalDelayType;   -- delay for this path
        PathCondition   : BOOLEAN;          -- path sensitize condition
    END RECORD;
    TYPE VitalPath01Type IS RECORD
        InputChangeTime : TIME;             -- timestamp for path input signal
        PathDelay       : VitalDelayType01; -- delay for this path
        PathCondition   : BOOLEAN;          -- path sensitize condition
    END RECORD;
    TYPE VitalPath01ZType IS RECORD
        InputChangeTime : TIME;             -- timestamp for path input signal
        PathDelay       : VitalDelayType01Z;-- delay for this path
        PathCondition   : BOOLEAN;          -- path sensitize condition
    END RECORD;

    -- For representing multiple paths to an output
    TYPE VitalPathArrayType    IS ARRAY (NATURAL RANGE <> ) OF VitalPathType;
    TYPE VitalPathArray01Type  IS ARRAY (NATURAL RANGE <> ) OF VitalPath01Type;
    TYPE VitalPathArray01ZType IS ARRAY (NATURAL RANGE <> ) OF VitalPath01ZType;

    TYPE VitalTableSymbolType IS (
       '/',      -- 0 -> 1
       '\',      -- 1 -> 0
       'P',      -- Union of '/' and '^' (any edge to 1)
       'N',      -- Union of '\' and 'v' (any edge to 0)
       'r',      -- 0 -> X
       'f',      -- 1 -> X
       'p',      -- Union of '/' and 'r' (any edge from 0)
       'n',      -- Union of '\' and 'f' (any edge from 1)
       'R',      -- Union of '^' and 'p' (any possible rising edge)
       'F',      -- Union of 'v' and 'n' (any possible falling edge)
       '^',      -- X -> 1
       'v',      -- X -> 0
       'E',      -- Union of 'v' and '^' (any edge from X)
       'A',      -- Union of 'r' and '^' (rising edge to or from 'X')
       'D',      -- Union of 'f' and 'v' (falling edge to or from 'X')
       '*',      -- Union of 'R' and 'F' (any edge)
       'X',      -- Unknown level
       '0',      -- low level
       '1',      -- high level
       '-',      -- don't care
       'B',      -- 0 or 1
       'Z',      -- High Impedance
       'S'       -- steady value
    );

    SUBTYPE VitalEdgeSymbolType IS VitalTableSymbolType RANGE '/' TO '*';




 -- Addition of Vital Skew Type Information
 -- March 14, 1998

    ---------------------------------------------------------------------------
    -- Procedures and Type Definitions for Defining Skews
    ---------------------------------------------------------------------------

    TYPE VitalSkewExpectedType IS (none, s1r, s1f, s2r, s2f);

    TYPE VitalSkewDataType IS RECORD
        ExpectedType : VitalSkewExpectedType;
        Signal1Old1 : TIME;
        Signal2Old1 : TIME;
        Signal1Old2 : TIME;
        Signal2Old2 : TIME;
    END RECORD;

    CONSTANT VitalSkewDataInit : VitalSkewDataType := ( none, 0 ns, 0 ns, 0 ns, 0 ns );


    -- ------------------------------------------------------------------------
    --
    -- Function Name:   VitalExtendToFillDelay
    --
    -- Description:     A six element array of delay values of type
    --                  VitalDelayType01Z is returned when a 1, 2 or 6
    --                  element array is given.  This function will convert
    --                  VitalDelayType and VitalDelayType01 delay values into
    --                  a VitalDelayType01Z type following these rules:
    --
    --                  When a VitalDelayType is passed, all six transition
    --                  values are assigned the input value.  When a
    --                  VitalDelayType01 is passed, the 01 transitions are
    --                  assigned to the 01, 0Z and Z1 transitions and the 10
    --                  transitions are assigned to 10, 1Z and Z0 transition
    --                  values.  When a VitalDelayType01Z is passed, the values
    --                  are kept as is.
    --
    --                  The function is overloaded based on input type.
    --
    --                  There is no function to fill a 12 value delay
    --                  type.
    --
    -- Arguments:
    --
    --  IN                 Type           Description
    --                     Delay          A one, two or six delay value Vital-
    --                                    DelayType is passed and a six delay,
    --                                    VitalDelayType01Z, item is returned.
    --
    --  INOUT
    --   none
    --
    --  OUT
    --   none
    --
    --  Returns
    --   VitalDelayType01Z
    --
    -- -------------------------------------------------------------------------
    FUNCTION VitalExtendToFillDelay (
            CONSTANT Delay : IN VitalDelayType
          ) RETURN VitalDelayType01Z;
    FUNCTION VitalExtendToFillDelay (
            CONSTANT Delay : IN VitalDelayType01
          ) RETURN VitalDelayType01Z;
    FUNCTION VitalExtendToFillDelay (
            CONSTANT Delay : IN VitalDelayType01Z
          ) RETURN VitalDelayType01Z;

    -- ------------------------------------------------------------------------
    --
    -- Function Name:   VitalCalcDelay
    --
    -- Description:     This function accepts a 1, 2 or 6 value delay and
    --                  chooses the correct delay time to delay the NewVal
    --                  signal.  This function is overloaded based on the
    --                  delay type passed. The function returns a single value
    --                  of time.
    --
    --                  This function is provided for Level 0 models in order
    --                  to calculate the delay which should be applied
    --                  for the passed signal. The delay selection is performed
    --                  using the OldVal and the NewVal to determine the
    --                  transition to select.  The default value of OldVal is X.
    --
    --                  This function cannot be used in a Level 1 model since
    --                  the VitalPathDelay routines perform the delay path
    --                  selection and output driving function.
    --
    -- Arguments:
    --
    --  IN                 Type           Description
    --                      NewVal         New value of the signal to be
    --                                     assigned
    --                      OldVal         Previous value of the signal.
    --                                     Default value is 'X'
    --                      Delay          The delay structure from which to
    --                                     select the appropriate delay.  The
    --                                     function overload is based on the
    --                                     type of delay passed.  In the case of
    --                                     the single delay, VitalDelayType, no
    --                                     selection is performed, since there
    --                                     is only one value to choose from.
    --                                     For the other cases, the transition
    --                                     from the old value to the new value
    --                                     decide the value returned.
    --
    --  INOUT
    --   none
    --
    --  OUT
    --   none
    --
    --  Returns
    --   Time                              The time value selected from the
    --                                     Delay INPUT is returned.
    --
    -- -------------------------------------------------------------------------
    FUNCTION VitalCalcDelay (
            CONSTANT NewVal : IN std_ulogic   := 'X';
            CONSTANT OldVal : IN std_ulogic   := 'X';
            CONSTANT Delay  : IN VitalDelayType
          ) RETURN TIME;
    FUNCTION VitalCalcDelay (
            CONSTANT NewVal : IN std_ulogic   := 'X';
            CONSTANT OldVal : IN std_ulogic   := 'X';
            CONSTANT Delay  : IN VitalDelayType01
          ) RETURN TIME;
    FUNCTION VitalCalcDelay (
            CONSTANT NewVal : IN std_ulogic   := 'X';
            CONSTANT OldVal : IN std_ulogic   := 'X';
            CONSTANT Delay  : IN VitalDelayType01Z
          ) RETURN TIME;

   -- ------------------------------------------------------------------------
    --
    -- Function Name:   VitalPathDelay
    --
    -- Description:     VitalPathDelay is the Level 1 routine used to select
    --                  the propagation delay path and schedule a new output
    --                  value.
    --
    --                  For single and dual delay values, VitalDelayType and
    --                  VitalDelayType01 are used.  The output value is
    --                  scheduled with a calculated delay without strength
    --                  modification.
    --
    --                  For the six delay value, VitalDelayType01Z, the output
    --                  value is scheduled with a calculated delay.  The drive
    --                  strength can be modified to handle weak signal strengths
    --                  to model tri-state devices, pull-ups and pull-downs as
    --                  an example.
    --
    --                  The correspondence between the delay type and the
    --                  path delay function is as follows:
    --
    --                  Delay Type                Path Type
    --
    --                  VitalDelayType            VitalPathDelay
    --                  VitalDelayType01          VitalPathDelay01
    --                  VitalDelayType01Z         VitalPathDelay01Z
    --
    --                  For each of these routines, the following capabilities
    --                  is provided:
    --
    --                  o Transition dependent path delay selection
    --                  o User controlled glitch detection with the ability
    --                    to generate "X" on output and report the violation
    --                  o Control of the severity level for message generation
    --                  o Scheduling of the computed values on the specified
    --                    signal.
    --
    --                  Selection of the appropriate path delay begins with the
    --                  candidate paths.  The candidate paths are selected by
    --                  identifying the paths for which the PathCondition is
    --                  true.  If there is a single candidate path, then that
    --                  delay is selected.  If there is more than one candidate
    --                  path, then the shortest delay is selected using
    --                  transition dependent delay selection.  If there is no
    --                  candidate paths, then the delay specified by the
    --                  DefaultDelay parameter to the path delay is used.
    --
    --                  Once the delay is known, the output signal is then
    --                  scheduled with that delay.  In the case of
    --                  VitalPathDelay01Z, an additional result mapping of
    --                  the output value is performed before scheduling.  The
    --                  result mapping is performed after transition dependent
    --                  delay selection but before scheduling the final output.
    --
    --                  In order to perform glitch detection, the user is
    --                  obligated to provide a variable of VitalGlitchDataType
    --                  for the propagation delay functions to use.  The user
    --                  cannot modify or use this information.
    --
    -- Arguments:
    --
    --  IN             Type                   Description
    --   OutSignalName  string                 The name of the output signal
    --   OutTemp        std_logic              The new output value to be driven
    --   Paths          VitalPathArrayType     A list of paths of VitalPathArray
    --                  VitalPathArrayType01   type.  The VitalPathDelay routine
    --                  VitalPathArrayType01Z  is overloaded based on the type
    --                                         of constant passed in.  With
    --                                         VitalPathArrayType01Z, the
    --                                         resulting output strengths can be
    --                                         mapped.
    --   DefaultDelay   VitalDelayType         The default delay can be changed
    --                  VitalDelayType01       from zero-delay to another set
    --                  VitalDelayType01Z      of values.
    --
    --   IgnoreDefaultDelay BOOLEAN            If TRUE, the default delay will
    --                                         be used when no paths are
    --                                         selected.  If false, no event
    --                                         will be scheduled if no paths are
    --                                         selected.
    --
    --   Mode           VitalGlitchKindType    The value of this constant
    --                                         selects the type of glitch
    --                                         detection.
    --                       OnEvent           Glitch on transition event
    --                     | OnDetect          Glitch immediate on detection
    --                     | VitalInertial     No glitch, use INERTIAL
    --                                         assignment
    --                     | VitalTransport    No glitch, use TRANSPORT
    --                                         assignment
    --   XOn            BOOLEAN                Control for generation of 'X' on
    --                                         glitch.  When TRUE, 'X's are
    --                                         scheduled for glitches, otherwise
    --                                         no are generated.
    --   MsgOn          BOOLEAN                Control for message generation on
    --                                         glitch detect.  When TRUE,
    --                                         glitches are reported, otherwise
    --                                         they are not reported.
    --   MsgSeverity    SEVERITY_LEVEL         The level at which the message,
    --                                         or assertion, will be reported.
    --   IgnoreDefaultDelay BOOLEAN            Tells the VPD whether to use the
    --                                         default delay value in the absense
    --                                         of a valid delay for input conditions 3/14/98 MG
    --
    --   OutputMap      VitalOutputMapType     For VitalPathDelay01Z, the output
    --                                         can be mapped to alternate
    --                                         strengths to model tri-state
    --                                         devices, pull-ups and pull-downs.
    --
    --  INOUT
    --   GlitchData     VitalGlitchDataType    The internal data storage
    --                                         variable required to detect
    --                                         glitches.
    --
    --  OUT
    --   OutSignal      std_logic              The output signal to be driven
    --
    --  Returns
    --   none
    --
    -- -------------------------------------------------------------------------
    PROCEDURE VitalPathDelay (
        SIGNAL   OutSignal          : OUT   std_logic;
        VARIABLE GlitchData         : INOUT VitalGlitchDataType;
        CONSTANT OutSignalName      : IN    string;
        CONSTANT OutTemp            : IN    std_logic;
        CONSTANT Paths              : IN    VitalPathArrayType;
        CONSTANT DefaultDelay       : IN    VitalDelayType      := VitalZeroDelay;
        CONSTANT Mode               : IN    VitalGlitchKindType := OnEvent;
        CONSTANT XOn                : IN    BOOLEAN             := TRUE;
        CONSTANT MsgOn              : IN    BOOLEAN             := TRUE;
        CONSTANT MsgSeverity        : IN    SEVERITY_LEVEL      := WARNING;
        CONSTANT NegPreemptOn       : IN    BOOLEAN             := FALSE;  --IR225 3/14/98
        CONSTANT IgnoreDefaultDelay : IN    BOOLEAN             := FALSE   --IR248 3/14/98
    );
    PROCEDURE VitalPathDelay01 (
        SIGNAL   OutSignal          : OUT   std_logic;
        VARIABLE GlitchData         : INOUT VitalGlitchDataType;
        CONSTANT OutSignalName      : IN    string;
        CONSTANT OutTemp            : IN    std_logic;
        CONSTANT Paths              : IN    VitalPathArray01Type;
        CONSTANT DefaultDelay       : IN    VitalDelayType01    := VitalZeroDelay01;
        CONSTANT Mode               : IN    VitalGlitchKindType := OnEvent;
        CONSTANT XOn                : IN    BOOLEAN             := TRUE;
        CONSTANT MsgOn              : IN    BOOLEAN             := TRUE;
        CONSTANT MsgSeverity        : IN    SEVERITY_LEVEL      := WARNING;
        CONSTANT NegPreemptOn       : IN    BOOLEAN             := FALSE;  --IR225 3/14/98
        CONSTANT IgnoreDefaultDelay : IN    BOOLEAN             := FALSE;  --IR248 3/14/98
        CONSTANT RejectFastPath     : IN    BOOLEAN             := FALSE   --IR250
    );
    PROCEDURE VitalPathDelay01Z (
        SIGNAL   OutSignal          : OUT   std_logic;
        VARIABLE GlitchData         : INOUT VitalGlitchDataType;
        CONSTANT OutSignalName      : IN    string;
        CONSTANT OutTemp            : IN    std_logic;
        CONSTANT Paths              : IN    VitalPathArray01ZType;
        CONSTANT DefaultDelay       : IN    VitalDelayType01Z   := VitalZeroDelay01Z;
        CONSTANT Mode               : IN    VitalGlitchKindType := OnEvent;
        CONSTANT XOn                : IN    BOOLEAN             := TRUE;
        CONSTANT MsgOn              : IN    BOOLEAN             := TRUE;
        CONSTANT MsgSeverity        : IN    SEVERITY_LEVEL      := WARNING;
        CONSTANT OutputMap          : IN    VitalOutputMapType  := VitalDefaultOutputMap;
        CONSTANT NegPreemptOn       : IN    BOOLEAN             := FALSE;  --IR225 3/14/98
        CONSTANT IgnoreDefaultDelay : IN    BOOLEAN             := FALSE;  --IR248 3/14/98
        CONSTANT RejectFastPath     : IN    BOOLEAN             := FALSE   --IR250
    );

    -- ------------------------------------------------------------------------
    --
    -- Function Name:   VitalWireDelay
    --
    -- Description:     VitalWireDelay is used to delay an input signal.
    --                  The delay is selected from the input parameter passed.
    --                  The function is useful for back annotation of actual
    --                  net delays.
    --
    --                  The function is overloaded to permit passing a delay
    --                  value for twire for VitalDelayType, VitalDelayType01
    --                  and VitalDelayType01Z.  twire is a generic which can
    --                  be back annotated and must be constructed to follow
    --                  the SDF to generic mapping rules.
    --
    -- Arguments:
    --
    --  IN          Type                  Description
    --   InSig       std_ulogic            The input signal (port) to be
    --                                     delayed.
    --   twire       VitalDelayType        The delay value for which the input
    --               VitalDelayType01      signal should be delayed.  For Vital-
    --               VitalDelayType01Z     DelayType, the value is single value
    --                                     passed.  For VitalDelayType01 and
    --                                     VitalDelayType01Z, the appropriate
    --                                     delay value is selected by VitalCalc-
    --                                     Delay.
    --
    --  INOUT
    --   none
    --
    --  OUT
    --   OutSig      std_ulogic            The internal delayed signal
    --
    --  Returns
    --   none
    --
    -- -------------------------------------------------------------------------
    PROCEDURE VitalWireDelay (
            SIGNAL   OutSig     : OUT   std_ulogic;
            SIGNAL   InSig      : IN    std_ulogic;
            CONSTANT twire      : IN    VitalDelayType
        );

    PROCEDURE VitalWireDelay (
            SIGNAL   OutSig     : OUT   std_ulogic;
            SIGNAL   InSig      : IN    std_ulogic;
            CONSTANT twire      : IN    VitalDelayType01
        );

    PROCEDURE VitalWireDelay (
            SIGNAL   OutSig     : OUT   std_ulogic;
            SIGNAL   InSig      : IN    std_ulogic;
            CONSTANT twire      : IN    VitalDelayType01Z
        );

    -- ------------------------------------------------------------------------
    --
    -- Function Name:   VitalSignalDelay
    --
    -- Description:     The VitalSignalDelay procedure is called in a signal
    --                  delay block in the architecture to delay the
    --                  appropriate test or reference signal in order to
    --                  accommodate negative constraint checks.
    --
    --                  The amount of delay is of type TIME and is a constant.
    --
    -- Arguments:
    --
    --  IN                Type            Description
    --   InSig             std_ulogic      The signal to be delayed.
    --   dly               TIME            The amount of time the signal is
    --                                     delayed.
    --
    --  INOUT
    --   none
    --
    --  OUT
    --   OutSig            std_ulogic      The delayed signal
    --
    --  Returns
    --   none
    --
    -- -------------------------------------------------------------------------
    PROCEDURE VitalSignalDelay (
            SIGNAL   OutSig     : OUT   std_ulogic;
            SIGNAL   InSig      : IN    std_ulogic;
            CONSTANT dly        : IN   TIME
    );

    -- ------------------------------------------------------------------------
    --
    -- Function Name:  VitalSetupHoldCheck
    --
    -- Description:    The VitalSetupHoldCheck procedure detects a setup or a
    --                 hold violation on the input test signal with respect
    --                 to the corresponding input reference signal.  The timing
    --                 constraints are specified through parameters
    --                 representing the high and low values for the setup and
    --                 hold values for the setup and hold times.  This
    --                 procedure assumes non-negative values for setup and hold
    --                 timing constraints.
    --
    --                 It is assumed that negative timing constraints
    --                 are handled by internally delaying the test or
    --                 reference signals.  Negative setup times result in
    --                 a delayed reference signal.  Negative hold times
    --                 result in a delayed test signal.  Furthermore, the
    --                 delays and constraints associated with these and
    --                 other signals may need to be appropriately
    --                 adjusted so that all constraint intervals overlap
    --                 the delayed reference signals and all constraint
    --                 values (with respect to the delayed signals) are
    --                 non-negative.
    --
    --                 This function is overloaded based on the input
    --                 TestSignal.  A vector and scalar form are provided.
    --
    -- TestSignal XXXXXXXXXXXX____________________________XXXXXXXXXXXXXXXXXXXXXX
    --            :
    --            :        -->|       error region       |<--
    --            :
    --            _______________________________
    -- RefSignal                                 \______________________________
    --            :           |                  |       |
    --            :           |               -->|       |<-- thold
    --            :        -->|     tsetup       |<--
    --
    -- Arguments:
    --
    --  IN                 Type           Description
    --   TestSignal         std_ulogic     Value of test signal
    --                      std_logic_vector
    --   TestSignalName     STRING         Name of test signal
    --   TestDelay          TIME           Model's internal delay associated
    --                                     with TestSignal
    --   RefSignal          std_ulogic     Value of reference signal
    --   RefSignalName      STRING         Name of reference signal
    --   RefDelay           TIME           Model's internal delay associated
    --                                     with RefSignal
    --   SetupHigh          TIME           Absolute minimum time duration before
    --                                     the transition of RefSignal for which
    --                                     transitions of TestSignal are allowed
    --                                     to proceed to the "1" state without
    --                                     causing a setup violation.
    --   SetupLow           TIME           Absolute minimum time duration before
    --                                     the transition of RefSignal for which
    --                                     transitions of TestSignal are allowed
    --                                     to proceed to the "0" state without
    --                                     causing a setup violation.
    --   HoldHigh           TIME           Absolute minimum time duration after
    --                                     the transition of RefSignal for which
    --                                     transitions of TestSignal are allowed
    --                                     to proceed to the "1" state without
    --                                     causing a hold violation.
    --   HoldLow            TIME           Absolute minimum time duration after
    --                                     the transition of RefSignal for which
    --                                     transitions of TestSignal are allowed
    --                                     to proceed to the "0" state without
    --                                     causing a hold violation.
    --   CheckEnabled       BOOLEAN        Check performed if TRUE.
    --   RefTransition      VitalEdgeSymbolType
    --                                     Reference edge specified. Events on
    --                                     the RefSignal which match the edge
    --                                     spec. are used as reference edges.
    --   HeaderMsg          STRING         String that will accompany any
    --                                     assertion messages produced.
    --   XOn                BOOLEAN         If TRUE, Violation output parameter
    --                                     is set to "X". Otherwise, Violation
    --                                     is always set to "0".
    --   MsgOn              BOOLEAN        If TRUE, set and hold violation
    --                                     message will be generated.
    --                                     Otherwise, no messages are generated,
    --                                     even upon violations.
    --   MsgSeverity        SEVERITY_LEVEL Severity level for the assertion.
    --   EnableSetupOnTest  BOOLEAN        If FALSE at the time that the
    --                                     TestSignal signal changes,
    --                                     no setup check will be performed.
    --   EnableSetupOnRef   BOOLEAN        If FALSE at the time that the
    --                                     RefSignal signal changes,
    --                                     no setup check will be performed.
    --   EnableHoldOnRef    BOOLEAN        If FALSE at the time that the
    --                                     RefSignal signal changes,
    --                                     no hold check will be performed.
    --   EnableHoldOnTest   BOOLEAN        If FALSE at the time that the
    --                                     TestSignal signal changes,
    --                                     no hold check will be performed.
    --
    --  INOUT
    --   TimingData         VitalTimingDataType
    --                                     VitalSetupHoldCheck information
    --                                     storage area.  This is used
    --                                     internally to detect reference edges
    --                                     and record the time of the last edge.
    --
    --  OUT
    --   Violation   X01                   This is the violation flag returned.
    --
    --  Returns
    --   none
    --
    -- -------------------------------------------------------------------------
    PROCEDURE VitalSetupHoldCheck (
            VARIABLE Violation     : OUT    X01;
            VARIABLE TimingData    : INOUT  VitalTimingDataType;
            SIGNAL   TestSignal    : IN     std_ulogic;
            CONSTANT TestSignalName: IN     STRING := "";
            CONSTANT TestDelay     : IN     TIME := 0 ns;
            SIGNAL   RefSignal     : IN     std_ulogic;
            CONSTANT RefSignalName : IN     STRING := "";
            CONSTANT RefDelay      : IN     TIME := 0 ns;
            CONSTANT SetupHigh     : IN     TIME := 0 ns;
            CONSTANT SetupLow      : IN     TIME := 0 ns;
            CONSTANT HoldHigh      : IN     TIME := 0 ns;
            CONSTANT HoldLow       : IN     TIME := 0 ns;
            CONSTANT CheckEnabled  : IN     BOOLEAN := TRUE;
            CONSTANT RefTransition : IN     VitalEdgeSymbolType;
            CONSTANT HeaderMsg     : IN     STRING := " ";
            CONSTANT XOn           : IN     BOOLEAN := TRUE;
            CONSTANT MsgOn         : IN     BOOLEAN := TRUE;
            CONSTANT MsgSeverity   : IN     SEVERITY_LEVEL := WARNING;
            CONSTANT EnableSetupOnTest : IN BOOLEAN := TRUE;	--IR252 3/23/98
	        CONSTANT EnableSetupOnRef  : IN BOOLEAN := TRUE;	--IR252 3/23/98
    	    CONSTANT EnableHoldOnRef   : IN BOOLEAN := TRUE;    --IR252 3/23/98
	        CONSTANT EnableHoldOnTest  : IN BOOLEAN := TRUE	    --IR252 3/23/98
      );

    PROCEDURE VitalSetupHoldCheck (
            VARIABLE Violation     : OUT    X01;
            VARIABLE TimingData    : INOUT  VitalTimingDataType;
            SIGNAL   TestSignal    : IN     std_logic_vector;
            CONSTANT TestSignalName: IN     STRING := "";
            CONSTANT TestDelay     : IN     TIME := 0 ns;
            SIGNAL   RefSignal     : IN     std_ulogic;
            CONSTANT RefSignalName : IN     STRING := "";
            CONSTANT RefDelay      : IN     TIME := 0 ns;
            CONSTANT SetupHigh     : IN     TIME := 0 ns;
            CONSTANT SetupLow      : IN     TIME := 0 ns;
            CONSTANT HoldHigh      : IN     TIME := 0 ns;
            CONSTANT HoldLow       : IN     TIME := 0 ns;
            CONSTANT CheckEnabled  : IN     BOOLEAN := TRUE;
            CONSTANT RefTransition : IN     VitalEdgeSymbolType;
            CONSTANT HeaderMsg     : IN     STRING := " ";
            CONSTANT XOn           : IN     BOOLEAN := TRUE;
            CONSTANT MsgOn         : IN     BOOLEAN := TRUE;
            CONSTANT MsgSeverity   : IN     SEVERITY_LEVEL := WARNING;
            CONSTANT EnableSetupOnTest : IN BOOLEAN := TRUE;	--IR252 3/23/98
	        CONSTANT EnableSetupOnRef  : IN BOOLEAN := TRUE;	--IR252 3/23/98
    	    CONSTANT EnableHoldOnRef   : IN BOOLEAN := TRUE;    --IR252 3/23/98
	        CONSTANT EnableHoldOnTest  : IN BOOLEAN := TRUE	    --IR252 3/23/98
    );


    -- ------------------------------------------------------------------------
    --
    -- Function Name:   VitalRecoveryRemovalCheck
    --
    -- Description:     The VitalRecoveryRemovalCheck detects the presence of
    --                  a recovery or removal violation on the input test
    --                  signal with respect to the corresponding input reference
    --                  signal.  It assumes non-negative values of setup and
    --                  hold timing constraints.  The timing constraint is
    --                  specified through parameters representing the recovery
    --                  and removal times associated with a reference edge of
    --                  the reference signal.  A flag indicates whether a test
    --                  signal is asserted when it is high or when it is low.
    --
    --                  It is assumed that negative timing constraints
    --                  are handled by internally delaying the test or
    --                  reference signals.  Negative recovery times result in
    --                  a delayed reference signal.  Negative removal times
    --                  result in a delayed test signal.  Furthermore, the
    --                  delays and constraints associated with these and
    --                  other signals may need to be appropriately
    --                  adjusted so that all constraint intervals overlap
    --                  the delayed reference signals and all constraint
    --                  values (with respect to the delayed signals) are
    --                  non-negative.
    --
    -- Arguments:
    --
    --  IN               Type             Description
    --   TestSignal       std_ulogic       Value of TestSignal.  The routine is
    --   TestSignalName   STRING           Name of TestSignal
    --   TestDelay        TIME             Model internal delay associated with
    --                                     the TestSignal
    --   RefSignal        std_ulogic       Value of RefSignal
    --   RefSignalName    STRING           Name of RefSignal
    --   RefDelay         TIME             Model internal delay associated with
    --                                     the RefSignal
    --   Recovery         TIME             A change to an unasserted value on
    --                                     the asynchronous TestSignal must
    --                                     precede reference edge (on RefSignal)
    --                                     by at least this time.
    --   Removal          TIME             An asserted condition must be present
    --                                     on the asynchronous TestSignal for at
    --                                     least the removal time following a
    --                                     reference edge on RefSignal.
    --   ActiveLow        BOOLEAN          A flag which indicates if TestSignal
    --                                     is asserted when it is low - "0."
    --                                     FALSE indicate that TestSignal is
    --                                     asserted when it has a value "1."
    --   CheckEnabled     BOOLEAN          The check in enabled when the value
    --                                     is TRUE, otherwise the constraints
    --                                     are not checked.
    --   RefTransition    VitalEdgeSymbolType
    --                                     Reference edge specifier.  Events on
    --                                     RefSignal will match the edge
    --                                     specified.
    --   HeaderMsg         STRING          A header message that will accompany
    --                                     any assertion message.
    --   XOn               BOOLEAN         When TRUE, the output Violation is
    --                                     set to "X."  When FALSE, it is always
    --                                     "0."
    --   MsgOn             BOOLEAN         When TRUE, violation messages are
    --                                     output.  When FALSE, no messages are
    --                                     generated.
    --   MsgSeverity       SEVERITY_LEVEL  Severity level of the asserted
    --                                     message.
    --   EnableRecOnTest   BOOLEAN         If FALSE at the time that the
    --                                     TestSignal signal changes,
    --                                     no recovery check will be performed.
    --   EnableRecOnRef    BOOLEAN         If FALSE at the time that the
    --                                     RefSignal signal changes,
    --                                     no recovery check will be performed.
    --   EnableRemOnRef    BOOLEAN         If FALSE at the time that the
    --                                     RefSignal signal changes,
    --                                     no removal check will be performed.
    --   EnableRemOnTest   BOOLEAN         If FALSE at the time that the
    --                                     TestSignal signal changes,
    --                                     no removal check will be performed.
    --
    --   INOUT
    --    TimingData       VitalTimingDataType
    --                                     VitalRecoveryRemovalCheck information
    --                                     storage area.  This is used
    --                                     internally to detect reference edges
    --                                     and record the time of the last edge.
    --   OUT
    --    Violation        X01             This is the violation flag returned.
    --
    --   Returns
    --           none
    --
    -- -------------------------------------------------------------------------
    PROCEDURE VitalRecoveryRemovalCheck (
            VARIABLE Violation     : OUT    X01;
            VARIABLE TimingData    : INOUT  VitalTimingDataType;
            SIGNAL   TestSignal    : IN     std_ulogic;
            CONSTANT TestSignalName: IN     STRING := "";
            CONSTANT TestDelay     : IN     TIME := 0 ns;
            SIGNAL   RefSignal     : IN     std_ulogic;
            CONSTANT RefSignalName : IN     STRING := "";
            CONSTANT RefDelay      : IN     TIME := 0 ns;
            CONSTANT Recovery      : IN     TIME := 0 ns;
            CONSTANT Removal       : IN     TIME := 0 ns;
            CONSTANT ActiveLow     : IN     BOOLEAN := TRUE;
            CONSTANT CheckEnabled  : IN     BOOLEAN := TRUE;
            CONSTANT RefTransition : IN     VitalEdgeSymbolType;
            CONSTANT HeaderMsg     : IN     STRING := " ";
            CONSTANT XOn           : IN     BOOLEAN := TRUE;
            CONSTANT MsgOn         : IN     BOOLEAN := TRUE;
            CONSTANT MsgSeverity   : IN     SEVERITY_LEVEL := WARNING;
    	    CONSTANT EnableRecOnTest : IN   BOOLEAN := TRUE;	--IR252 3/23/98
	        CONSTANT EnableRecOnRef  : IN   BOOLEAN := TRUE;	--IR252 3/23/98
	        CONSTANT EnableRemOnRef  : IN   BOOLEAN := TRUE;	--IR252 3/23/98
	        CONSTANT EnableRemOnTest : IN   BOOLEAN := TRUE		--IR252 3/23/98
    );

    -- ------------------------------------------------------------------------
    --
    -- Function Name:  VitalPeriodPulseCheck
    --
    -- Description:    VitalPeriodPulseCheck checks for minimum and maximum
    --                 periodicity and pulse width for "1" and "0" values of
    --                 the input test signal.  The timing constraint is
    --                 specified through parameters representing the minimal
    --                 period between successive rising and falling edges of
    --                 the input test signal and the minimum pulse widths
    --                 associated with high and low values.
    --
    --                 VitalPeriodCheck's accepts rising and falling edges
    --                 from 1 and 0 as well as transitions to and from 'X.'
    --
    --                    _______________         __________
    --       ____________|               |_______|
    --
    --                   |<--- pw_hi --->|
    --                   |<-------- period ----->|
    --                                -->| pw_lo |<--
    --
    -- Arguments:
    --  IN                 Type           Description
    --    TestSignal        std_ulogic     Value of test signal
    --    TestSignalName    STRING         Name of the test signal
    --    TestDelay         TIME           Model's internal delay associated
    --                                     with TestSignal
    --    Period            TIME           Minimum period allowed between
    --                                     consecutive rising ('P') or
    --                                     falling ('F') transitions.
    --    PulseWidthHigh    TIME           Minimum time allowed for a high
    --                                     pulse ('1' or 'H')
    --    PulseWidthLow     TIME           Minimum time allowed for a low
    --                                     pulse ('0' or 'L')
    --    CheckEnabled      BOOLEAN        Check performed if TRUE.
    --    HeaderMsg         STRING         String that will accompany any
    --                                     assertion messages produced.
    --    XOn               BOOLEAN        If TRUE, Violation output parameter
    --                                     is set to "X". Otherwise, Violation
    --                                     is always set to "0".
    --                                     XOnChecks is a global that allows for
    --                                     only timing checks to be turned on.
    --    MsgOn             BOOLEAN        If TRUE, period/pulse violation
    --                                     message will be generated.
    --                                     Otherwise, no messages are generated,
    --                                     even though a violation is detected.
    --                                     MsgOnChecks allows for only timing
    --                                     check messages to be turned on.
    --    MsgSeverity       SEVERITY_LEVEL Severity level for the assertion.
    --
    --   INOUT
    --    PeriodData        VitalPeriodDataType
    --                                     VitalPeriodPulseCheck information
    --                                     storage area.  This is used
    --                                     internally to detect reference edges
    --                                     and record the pulse and period
    --                                     times.
    --   OUT
    --    Violation         X01            This is the violation flag returned.
    --
    --   Returns
    --    none
    --
    -- ------------------------------------------------------------------------
    PROCEDURE VitalPeriodPulseCheck  (
            VARIABLE Violation      : OUT    X01;
            VARIABLE PeriodData     : INOUT  VitalPeriodDataType;
            SIGNAL   TestSignal     : IN     std_ulogic;
            CONSTANT TestSignalName : IN     STRING := "";
            CONSTANT TestDelay      : IN     TIME := 0 ns;
            CONSTANT Period         : IN     TIME := 0 ns;
            CONSTANT PulseWidthHigh : IN     TIME := 0 ns;
            CONSTANT PulseWidthLow  : IN     TIME := 0 ns;
            CONSTANT CheckEnabled   : IN     BOOLEAN := TRUE;
            CONSTANT HeaderMsg      : IN     STRING := " ";
            CONSTANT XOn           : IN     BOOLEAN := TRUE;
            CONSTANT MsgOn         : IN     BOOLEAN := TRUE;
            CONSTANT MsgSeverity   : IN     SEVERITY_LEVEL := WARNING
        );

    -- ------------------------------------------------------------------------
    --
    -- Function Name:  VitalInPhaseSkewCheck
    --
    -- Description:    The VitalInPhaseSkewCheck procedure detects an in-phase
    --                 skew violation between input signals Signal1 and Signal2.
    --                 This is a timer based skew check in which a
    --                 violation is detected if Signal1 and Signal2 are in
    --                 different logic states longer than the specified skew
    --                 interval.
    --
    --                 The timing constraints are specified through parameters
    --                 representing the skew values for the different states
    --                 of Signal1 and Signal2.
    --
    --
    -- Signal2    XXXXXXXXXXXX___________________________XXXXXXXXXXXXXXXXXXXXXX
    --            :
    --            :        -->|                         |<--
    --            :      Signal2 should go low in this region
    --            :
    --
    --            ____________
    -- Signal1                \_________________________________________________
    --            :           |                         |
    --            :           |<-------- tskew -------->|
    --
    -- Arguments:
    --
    --  IN                 Type           Description
    --   Signal1            std_ulogic     Value of first signal
    --   Signal1Name        STRING         Name of first signal
    --   Signal1Delay       TIME           Model's internal delay associated
    --                                     with Signal1
    --   Signal2            std_ulogic     Value of second signal
    --   Signal2Name        STRING         Name of second signal
    --   Signal2Delay       TIME           Model's internal delay associated
    --                                     with Signal2
    --   SkewS1S2RiseRise   TIME           Absolute maximum time duration for
    --                                     which Signal2 can remain at "0"
    --                                     after Signal1 goes to the "1" state,
    --                                     without causing a skew violation.
    --   SkewS2S1RiseRise   TIME           Absolute maximum time duration for
    --                                     which Signal1 can remain at "0"
    --                                     after Signal2 goes to the "1" state,
    --                                     without causing a skew violation.
    --   SkewS1S2FallFall   TIME           Absolute maximum time duration for
    --                                     which Signal2 can remain at "1"
    --                                     after Signal1 goes to the "0" state,
    --                                     without causing a skew violation.
    --   SkewS2S1FallFall   TIME           Absolute maximum time duration for
    --                                     which Signal1 can remain at "1"
    --                                     after Signal2 goes to the "0" state,
    --                                     without causing a skew violation.
    --   CheckEnabled       BOOLEAN        Check performed if TRUE.
    --   HeaderMsg          STRING         String that will accompany any
    --                                     assertion messages produced.
    --   XOn                BOOLEAN        If TRUE, Violation output parameter
    --                                     is set to "X". Otherwise, Violation
    --                                     is always set to "0."
    --   MsgOn              BOOLEAN        If TRUE, skew timing violation
    --                                     messages will be generated.
    --                                     Otherwise, no messages are generated,
    --                                     even upon violations.
    --   MsgSeverity        SEVERITY_LEVEL Severity level for the assertion.
    --
    --  INOUT
    --   SkewData           VitalSkewDataType
    --                                     VitalInPhaseSkewCheck information
    --                                     storage area.  This is used
    --                                     internally to detect signal edges
    --                                     and record the time of the last edge.
    --
    --
    --   Trigger            std_ulogic     This signal is used to trigger the
    --                                     process in which the timing check
    --                                     occurs upon expiry of the skew
    --                                     interval.
    --
    --  OUT
    --   Violation   X01                   This is the violation flag returned.
    --
    --  Returns
    --   none
    --
    -- -------------------------------------------------------------------------

    PROCEDURE VitalInPhaseSkewCheck (
        VARIABLE Violation         : OUT    X01;
        VARIABLE SkewData          : INOUT  VitalSkewDataType;
        SIGNAL   Signal1           : IN     std_ulogic;
        CONSTANT Signal1Name       : IN     STRING := "";
        CONSTANT Signal1Delay      : IN     TIME := 0 ns;
        SIGNAL   Signal2           : IN     std_ulogic;
        CONSTANT Signal2Name       : IN     STRING := "";
        CONSTANT Signal2Delay      : IN     TIME := 0 ns;
        CONSTANT SkewS1S2RiseRise  : IN     TIME := TIME'HIGH;
        CONSTANT SkewS2S1RiseRise  : IN     TIME := TIME'HIGH;
        CONSTANT SkewS1S2FallFall  : IN     TIME := TIME'HIGH;
        CONSTANT SkewS2S1FallFall  : IN     TIME := TIME'HIGH;
        CONSTANT CheckEnabled      : IN     BOOLEAN := TRUE;
        CONSTANT XOn               : IN     BOOLEAN := TRUE;
        CONSTANT MsgOn             : IN     BOOLEAN := TRUE;
        CONSTANT MsgSeverity       : IN     SEVERITY_LEVEL := WARNING;
        CONSTANT HeaderMsg         : IN     STRING  := "";
        SIGNAL   Trigger           : INOUT  std_ulogic
    );


  -- ------------------------------------------------------------------------
    --
    -- Function Name:  VitalOutPhaseSkewCheck
    --
    -- Description:    The VitalOutPhaseSkewCheck procedure detects an
    --                 out-of-phase skew violation between input signals Signal1
    --                 and Signal2. This is a timer based skew check in
    --                 which a violation is detected if Signal1 and Signal2 are
    --                 in the same logic state longer than the specified skew
    --                 interval.
    --
    --                 The timing constraints are specified through parameters
    --                 representing the skew values for the different states
    --                 of Signal1 and Signal2.
    --
    --
    -- Signal2    XXXXXXXXXXXX___________________________XXXXXXXXXXXXXXXXXXXXXX
    --            :
    --            :        -->|                         |<--
    --            :     Signal2 should go high in this region
    --            :
    --
    --            ____________
    -- Signal1                \_________________________________________________
    --            :           |                         |
    --            :           |<-------- tskew -------->|
    --
    -- Arguments:
    --
    --  IN                 Type           Description
    --   Signal1            std_ulogic     Value of first signal
    --   Signal1Name        STRING         Name of first signal
    --   Signal1Delay       TIME           Model's internal delay associated
    --                                     with Signal1
    --   Signal2            std_ulogic     Value of second signal
    --   Signal2Name        STRING         Name of second signal
    --   Signal2Delay       TIME           Model's internal delay associated
    --                                     with Signal2
    --   SkewS1S2RiseFall   TIME           Absolute maximum time duration for
    --                                     which Signal2 can remain at "1"
    --                                     after Signal1 goes to the "1" state,
    --                                     without causing a skew violation.
    --   SkewS2S1RiseFall   TIME           Absolute maximum time duration for
    --                                     which Signal1 can remain at "1"
    --                                     after Signal2 goes to the "1" state,
    --                                     without causing a skew violation.
    --   SkewS1S2FallRise   TIME           Absolute maximum time duration for
    --                                     which Signal2 can remain at "0"
    --                                     after Signal1 goes to the "0" state,
    --                                     without causing a skew violation.
    --   SkewS2S1FallRise   TIME           Absolute maximum time duration for
    --                                     which Signal1 can remain at "0"
    --                                     after Signal2 goes to the "0" state,
    --                                     without causing a skew violation.
    --   CheckEnabled       BOOLEAN        Check performed if TRUE.
    --   HeaderMsg          STRING         String that will accompany any
    --                                     assertion messages produced.
    --   XOn                BOOLEAN        If TRUE, Violation output parameter
    --                                     is set to "X". Otherwise, Violation
    --                                     is always set to "0."
    --   MsgOn              BOOLEAN        If TRUE, skew timing violation
    --                                     messages will be generated.
    --                                     Otherwise, no messages are generated,
    --                                     even upon violations.
    --   MsgSeverity        SEVERITY_LEVEL Severity level for the assertion.
    --
    --  INOUT
    --   SkewData           VitalSkewDataType
    --                                     VitalInPhaseSkewCheck information
    --                                     storage area.  This is used
    --                                     internally to detect signal edges
    --                                     and record the time of the last edge.
    --
    --   Trigger            std_ulogic     This signal is used to trigger the
    --                                     process in which the timing check
    --                                     occurs upon expiry of the skew
    --                                     interval.
    --
    --  OUT
    --   Violation   X01                   This is the violation flag returned.
    --
    --  Returns
    --   none
    --
    -- -------------------------------------------------------------------------
    PROCEDURE VitalOutPhaseSkewCheck (
        VARIABLE Violation         : OUT    X01;
        VARIABLE SkewData          : INOUT  VitalSkewDataType;
        SIGNAL   Signal1           : IN     std_ulogic;
        CONSTANT Signal1Name       : IN     STRING := "";
        CONSTANT Signal1Delay      : IN     TIME := 0 ns;
        SIGNAL   Signal2           : IN     std_ulogic;
        CONSTANT Signal2Name       : IN     STRING := "";
        CONSTANT Signal2Delay      : IN     TIME := 0 ns;
        CONSTANT SkewS1S2RiseFall  : IN     TIME := TIME'HIGH;
        CONSTANT SkewS2S1RiseFall  : IN     TIME := TIME'HIGH;
        CONSTANT SkewS1S2FallRise  : IN     TIME := TIME'HIGH;
        CONSTANT SkewS2S1FallRise  : IN     TIME := TIME'HIGH;
        CONSTANT CheckEnabled      : IN     BOOLEAN := TRUE;
        CONSTANT XOn               : IN     BOOLEAN := TRUE;
        CONSTANT MsgOn             : IN     BOOLEAN := TRUE;
        CONSTANT MsgSeverity       : IN     SEVERITY_LEVEL := WARNING;
        CONSTANT HeaderMsg         : IN     STRING  := "";
        SIGNAL   Trigger           : INOUT  std_ulogic
    );

END VITAL_Timing;


use std.standard.all;
LIBRARY IEEE;
USE     IEEE.Std_Logic_1164.ALL;
USE     IEEE.VITAL_Timing.ALL;
PACKAGE VITAL_Primitives IS
    SUBTYPE VitalTruthSymbolType IS VitalTableSymbolType RANGE 'X' TO 'Z';
    SUBTYPE VitalStateSymbolType IS VitalTableSymbolType RANGE '/' TO 'S';

    TYPE VitalTruthTableType IS ARRAY ( NATURAL RANGE <>, NATURAL RANGE <> )
         OF VitalTruthSymbolType;
    TYPE VitalStateTableType IS ARRAY ( NATURAL RANGE <>, NATURAL RANGE <> )
         OF VitalStateSymbolType;

    CONSTANT VitalDefDelay01  : VitalDelayType01;
    CONSTANT VitalDefDelay01Z : VitalDelayType01Z;

    FUNCTION VitalAND    (
            CONSTANT      Data : IN std_logic_vector;
            CONSTANT ResultMap : IN VitalResultMapType := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalOR     (
            CONSTANT      Data : IN std_logic_vector;
            CONSTANT ResultMap : IN VitalResultMapType := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalXOR    (
            CONSTANT      Data : IN std_logic_vector;
            CONSTANT ResultMap : IN VitalResultMapType := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalNAND   (
            CONSTANT      Data : IN std_logic_vector;
            CONSTANT ResultMap : IN VitalResultMapType := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalNOR    (
            CONSTANT      Data : IN std_logic_vector;
            CONSTANT ResultMap : IN VitalResultMapType := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalXNOR   (
            CONSTANT      Data : IN std_logic_vector;
            CONSTANT ResultMap : IN VitalResultMapType := VitalDefaultResultMap
          ) RETURN std_ulogic;

    PROCEDURE VitalAND   (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         Data :  IN std_logic_vector;
            CONSTANT tpd_data_q :  IN VitalDelayArrayType01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalOR    (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         Data :  IN std_logic_vector;
            CONSTANT tpd_data_q :  IN VitalDelayArrayType01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalXOR   (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         Data :  IN std_logic_vector;
            CONSTANT tpd_data_q :  IN VitalDelayArrayType01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalNAND  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         Data :  IN std_logic_vector;
            CONSTANT tpd_data_q :  IN VitalDelayArrayType01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalNOR   (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         Data :  IN std_logic_vector;
            CONSTANT tpd_data_q :  IN VitalDelayArrayType01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalXNOR  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         Data :  IN std_logic_vector;
            CONSTANT tpd_data_q :  IN VitalDelayArrayType01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    FUNCTION VitalAND2   (
            CONSTANT       a, b :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalOR2    (
            CONSTANT       a, b :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalXOR2   (
            CONSTANT       a, b :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalNAND2  (
            CONSTANT       a, b :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalNOR2   (
            CONSTANT       a, b :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalXNOR2  (
            CONSTANT       a, b :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalAND3   (
            CONSTANT    a, b, c :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalOR3    (
            CONSTANT    a, b, c :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalXOR3   (
            CONSTANT    a, b, c :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalNAND3  (
            CONSTANT    a, b, c :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalNOR3   (
            CONSTANT    a, b, c :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalXNOR3  (
            CONSTANT    a, b, c :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalAND4   (
            CONSTANT a, b, c, d :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalOR4    (
            CONSTANT a, b, c, d :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalXOR4   (
            CONSTANT a, b, c, d :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalNAND4  (
            CONSTANT a, b, c, d :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalNOR4   (
            CONSTANT a, b, c, d :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalXNOR4  (
            CONSTANT a, b, c, d :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap
          ) RETURN std_ulogic;

    PROCEDURE VitalAND2  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         a, b :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalOR2   (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         a, b :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalXOR2  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         a, b :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalNAND2 (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         a, b :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalNOR2  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         a, b :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalXNOR2 (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         a, b :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalAND3  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL      a, b, c :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_c_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalOR3   (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL      a, b, c :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_c_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalXOR3  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL      a, b, c :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_c_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalNAND3 (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL      a, b, c :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_c_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalNOR3  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL      a, b, c :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_c_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalXNOR3 (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL      a, b, c :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_c_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalAND4  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL   a, b, c, d :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_c_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_d_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalOR4   (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL   a, b, c, d :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_c_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_d_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalXOR4  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL   a, b, c, d :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_c_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_d_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalNAND4 (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL   a, b, c, d :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_c_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_d_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalNOR4  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL   a, b, c, d :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_c_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_d_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    PROCEDURE VitalXNOR4 (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL   a, b, c, d :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_b_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_c_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    tpd_d_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                      := VitalDefaultResultMap );

    FUNCTION VitalBUF    (
            CONSTANT         Data :  IN std_ulogic;
            CONSTANT    ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap
          ) RETURN std_ulogic;
    FUNCTION VitalBUFIF0 (
            CONSTANT Data, Enable :  IN std_ulogic;
            CONSTANT    ResultMap :  IN VitalResultZMapType
                                        := VitalDefaultResultZMap
          ) RETURN std_ulogic;
    FUNCTION VitalBUFIF1 (
            CONSTANT Data, Enable :  IN std_ulogic;
            CONSTANT    ResultMap :  IN VitalResultZMapType
                                        := VitalDefaultResultZMap
          ) RETURN std_ulogic;
    FUNCTION VitalIDENT  (
            CONSTANT         Data :  IN std_ulogic;
            CONSTANT    ResultMap :  IN VitalResultZMapType
                                        := VitalDefaultResultZMap
          ) RETURN std_ulogic;

    PROCEDURE VitalBUF   (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL            a :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap );

    PROCEDURE VitalBUFIF0 (
            SIGNAL              q : OUT std_ulogic;
            SIGNAL           Data :  IN std_ulogic;
            SIGNAL         Enable :  IN std_ulogic;
            CONSTANT   tpd_data_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT tpd_enable_q :  IN VitalDelayType01Z   := VitalDefDelay01Z;
            CONSTANT    ResultMap :  IN VitalResultZMapType
                                        := VitalDefaultResultZMap);


    PROCEDURE VitalBUFIF1 (
            SIGNAL              q : OUT std_ulogic;
            SIGNAL           Data :  IN std_ulogic;
            SIGNAL         Enable :  IN std_ulogic;
            CONSTANT   tpd_data_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT tpd_enable_q :  IN VitalDelayType01Z   := VitalDefDelay01Z;
            CONSTANT    ResultMap :  IN VitalResultZMapType
                                        := VitalDefaultResultZMap);

    PROCEDURE VitalIDENT (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL            a :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01Z   := VitalDefDelay01Z;
            CONSTANT  ResultMap :  IN VitalResultZMapType
                                        := VitalDefaultResultZMap );

    FUNCTION VitalINV    (
            CONSTANT         Data :  IN std_ulogic;
            CONSTANT    ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalINVIF0 (
            CONSTANT Data, Enable :  IN std_ulogic;
            CONSTANT    ResultMap :  IN VitalResultZMapType
                                        := VitalDefaultResultZMap
          ) RETURN std_ulogic;

    FUNCTION VitalINVIF1 (
            CONSTANT Data, Enable :  IN std_ulogic;
            CONSTANT    ResultMap :  IN VitalResultZMapType
                                        := VitalDefaultResultZMap
          ) RETURN std_ulogic;

    PROCEDURE VitalINV   (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL            a :  IN std_ulogic;
            CONSTANT    tpd_a_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap );
    PROCEDURE VitalINVIF0 (
            SIGNAL              q : OUT std_ulogic;
            SIGNAL           Data :  IN std_ulogic;
            SIGNAL         Enable :  IN std_ulogic;
            CONSTANT   tpd_data_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT tpd_enable_q :  IN VitalDelayType01Z   := VitalDefDelay01Z;
            CONSTANT    ResultMap :  IN VitalResultZMapType
                                        := VitalDefaultResultZMap);

    PROCEDURE VitalINVIF1 (
            SIGNAL              q : OUT std_ulogic;
            SIGNAL           Data :  IN std_ulogic;
            SIGNAL         Enable :  IN std_ulogic;
            CONSTANT   tpd_data_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT tpd_enable_q :  IN VitalDelayType01Z   := VitalDefDelay01Z;
            CONSTANT    ResultMap :  IN VitalResultZMapType
                                        := VitalDefaultResultZMap);

    FUNCTION VitalMUX   (
            CONSTANT       Data :  IN std_logic_vector;
            CONSTANT    dSelect :  IN std_logic_vector;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalMUX2  (
            CONSTANT Data1, Data0 :  IN std_ulogic;
            CONSTANT      dSelect :  IN std_ulogic;
            CONSTANT    ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalMUX4  (
            CONSTANT       Data :  IN std_logic_vector4;
            CONSTANT    dSelect :  IN std_logic_vector2;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap
          ) RETURN std_ulogic;

    FUNCTION VitalMUX8  (
            CONSTANT       Data :  IN std_logic_vector8;
            CONSTANT    dSelect :  IN std_logic_vector3;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap
          ) RETURN std_ulogic;

    PROCEDURE VitalMUX   (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         Data :  IN std_logic_vector;
            SIGNAL         dSel :  IN std_logic_vector;
            CONSTANT tpd_data_q :  IN VitalDelayArrayType01;
            CONSTANT tpd_dsel_q :  IN VitalDelayArrayType01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap );

    PROCEDURE VitalMUX2  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL       d1, d0 :  IN std_ulogic;
            SIGNAL         dSel :  IN std_ulogic;
            CONSTANT   tpd_d1_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT   tpd_d0_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT tpd_dsel_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap );

    PROCEDURE VitalMUX4  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         Data :  IN std_logic_vector4;
            SIGNAL         dSel :  IN std_logic_vector2;
            CONSTANT tpd_data_q :  IN VitalDelayArrayType01;
            CONSTANT tpd_dsel_q :  IN VitalDelayArrayType01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap );

    PROCEDURE VitalMUX8  (
            SIGNAL            q : OUT std_ulogic;
            SIGNAL         Data :  IN std_logic_vector8;
            SIGNAL         dSel :  IN std_logic_vector3;
            CONSTANT tpd_data_q :  IN VitalDelayArrayType01;
            CONSTANT tpd_dsel_q :  IN VitalDelayArrayType01;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap );

    FUNCTION VitalDECODER   (
            CONSTANT       Data :  IN std_logic_vector;
            CONSTANT     Enable :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap
          ) RETURN std_logic_vector;

    FUNCTION VitalDECODER2  (
            CONSTANT       Data :  IN std_ulogic;
            CONSTANT     Enable :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap
          ) RETURN std_logic_vector2;

    FUNCTION VitalDECODER4  (
            CONSTANT       Data :  IN std_logic_vector2;
            CONSTANT     Enable :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap
          ) RETURN std_logic_vector4;

    FUNCTION VitalDECODER8  (
            CONSTANT       Data :  IN std_logic_vector3;
            CONSTANT     Enable :  IN std_ulogic;
            CONSTANT  ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap
          ) RETURN std_logic_vector8;

    PROCEDURE VitalDECODER   (
            SIGNAL              q : OUT std_logic_vector;
            SIGNAL           Data :  IN std_logic_vector;
            SIGNAL         Enable :  IN std_ulogic;
            CONSTANT   tpd_data_q :  IN VitalDelayArrayType01;
            CONSTANT tpd_enable_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap );
    PROCEDURE VitalDECODER2  (
            SIGNAL              q : OUT std_logic_vector2;
            SIGNAL           Data :  IN std_ulogic;
            SIGNAL         Enable :  IN std_ulogic;
            CONSTANT   tpd_data_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT tpd_enable_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap );

    PROCEDURE VitalDECODER4  (
            SIGNAL              q : OUT std_logic_vector4;
            SIGNAL           Data :  IN std_logic_vector2;
            SIGNAL         Enable :  IN std_ulogic;
            CONSTANT   tpd_data_q :  IN VitalDelayArrayType01;
            CONSTANT tpd_enable_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap );
    PROCEDURE VitalDECODER8  (
            SIGNAL              q : OUT std_logic_vector8;
            SIGNAL           Data :  IN std_logic_vector3;
            SIGNAL         Enable :  IN std_ulogic;
            CONSTANT   tpd_data_q :  IN VitalDelayArrayType01;
            CONSTANT tpd_enable_q :  IN VitalDelayType01    := VitalDefDelay01;
            CONSTANT    ResultMap :  IN VitalResultMapType
                                        := VitalDefaultResultMap );

    FUNCTION VitalTruthTable  (
            CONSTANT TruthTable   : IN VitalTruthTableType;
            CONSTANT DataIn       : IN std_logic_vector
          ) RETURN std_logic_vector;

    FUNCTION VitalTruthTable  (
            CONSTANT TruthTable   : IN VitalTruthTableType;
            CONSTANT DataIn       : IN std_logic_vector
          ) RETURN std_logic;

    PROCEDURE VitalTruthTable (
            SIGNAL   Result       : OUT std_logic_vector;
            CONSTANT TruthTable   : IN VitalTruthTableType;
            CONSTANT DataIn       : IN std_logic_vector
                              );
    PROCEDURE VitalTruthTable (
            SIGNAL   Result       : OUT std_logic;
            CONSTANT TruthTable   : IN VitalTruthTableType;
            CONSTANT DataIn       : IN std_logic_vector
                              );
   PROCEDURE VitalStateTable (
            VARIABLE Result         : INOUT std_logic_vector;
            VARIABLE PreviousDataIn : INOUT std_logic_vector;
            CONSTANT StateTable     : IN VitalStateTableType;
            CONSTANT DataIn         : IN  std_logic_vector;
            CONSTANT NumStates      : IN NATURAL
   );

   PROCEDURE VitalStateTable (
            VARIABLE Result         : INOUT std_logic;
            VARIABLE PreviousDataIn : INOUT std_logic_vector;
            CONSTANT StateTable     : IN VitalStateTableType;
            CONSTANT DataIn         : IN  std_logic_vector
   );

   PROCEDURE VitalStateTable (
            SIGNAL   Result     : INOUT std_logic_vector;
            CONSTANT StateTable : IN VitalStateTableType;
            SIGNAL   DataIn     : IN std_logic_vector;
            CONSTANT NumStates  : IN NATURAL
   );

   PROCEDURE VitalStateTable (
            SIGNAL   Result     : INOUT std_logic;
            CONSTANT StateTable : IN VitalStateTableType;
            SIGNAL   DataIn     : IN std_logic_vector
   );

    PROCEDURE VitalResolve (
            SIGNAL              q : OUT std_ulogic;
            CONSTANT         Data :  IN std_logic_vector);

END VITAL_Primitives;

use std.standard.all;
library IEEE;
use IEEE.STD_LOGIC_1164.all;
package NUMERIC_STD_UNSIGNED is

	-- Id: A.3
  function "+" (L, R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(MAXIMUM(L'LENGTH, R'LENGTH)-1 downto 0).
  -- Result: Adds two UNSIGNED vectors that may be of different lengths.

  -- Id: A.3R
  function "+"(L : STD_ULOGIC_VECTOR; R : STD_ULOGIC) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(L'LENGTH-1 downto 0)
  -- Result: Similar to A.3 where R is a one bit STD_ULOGIC_VECTOR

  -- Id: A.3L
  function "+"(L : STD_ULOGIC; R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(R'LENGTH-1 downto 0)
  -- Result: Similar to A.3 where L is a one bit UNSIGNED

  -- Id: A.5
  function "+" (L : STD_ULOGIC_VECTOR; R : NATURAL) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(L'LENGTH-1 downto 0).
  -- Result: Adds an UNSIGNED vector, L, with a non-negative INTEGER, R.

  -- Id: A.6
  function "+" (L : NATURAL; R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(R'LENGTH-1 downto 0).
  -- Result: Adds a non-negative INTEGER, L, with an UNSIGNED vector, R.

  --============================================================================

  -- Id: A.9
  function "-" (L, R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: UNSIGNED(MAXIMUM(L'LENGTH, R'LENGTH)-1 downto 0).
  -- Result: Subtracts two UNSIGNED vectors that may be of different lengths.

  -- Id: A.9R
  function "-"(L : STD_ULOGIC_VECTOR; R : STD_ULOGIC) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(L'LENGTH-1 downto 0)
  -- Result: Similar to A.9 where R is a one bit UNSIGNED

  -- Id: A.9L
  function "-"(L : STD_ULOGIC; R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(R'LENGTH-1 downto 0)
  -- Result: Similar to A.9 where L is a one bit UNSIGNED

  -- Id: A.11
  function "-" (L : STD_ULOGIC_VECTOR; R : NATURAL) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(L'LENGTH-1 downto 0).
  -- Result: Subtracts a non-negative INTEGER, R, from an UNSIGNED vector, L.

  -- Id: A.12
  function "-" (L : NATURAL; R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(R'LENGTH-1 downto 0).
  -- Result: Subtracts an UNSIGNED vector, R, from a non-negative INTEGER, L.

  --============================================================================

  -- Id: A.15
  function "*" (L, R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR((L'LENGTH+R'LENGTH-1) downto 0).
  -- Result: Performs the multiplication operation on two UNSIGNED vectors
  --         that may possibly be of different lengths.

  -- Id: A.17
  function "*" (L : STD_ULOGIC_VECTOR; R : NATURAL) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR((L'LENGTH+L'LENGTH-1) downto 0).
  -- Result: Multiplies an UNSIGNED vector, L, with a non-negative
  --         INTEGER, R. R is converted to an UNSIGNED vector of
  --         SIZE L'LENGTH before multiplication.

  -- Id: A.18
  function "*" (L : NATURAL; R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR((R'LENGTH+R'LENGTH-1) downto 0).
  -- Result: Multiplies an UNSIGNED vector, R, with a non-negative
  --         INTEGER, L. L is converted to an UNSIGNED vector of
  --         SIZE R'LENGTH before multiplication.

  --============================================================================
  --
  -- NOTE: If second argument is zero for "/" operator, a severity level
  --       of ERROR is issued.

  -- Id: A.21
  function "/" (L, R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(L'LENGTH-1 downto 0)
  -- Result: Divides an UNSIGNED vector, L, by another UNSIGNED vector, R.

  -- Id: A.23
  function "/" (L : STD_ULOGIC_VECTOR; R : NATURAL) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(L'LENGTH-1 downto 0)
  -- Result: Divides an UNSIGNED vector, L, by a non-negative INTEGER, R.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.24
  function "/" (L : NATURAL; R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(R'LENGTH-1 downto 0)
  -- Result: Divides a non-negative INTEGER, L, by an UNSIGNED vector, R.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  --============================================================================
  --
  -- NOTE: If second argument is zero for "rem" operator, a severity level
  --       of ERROR is issued.

  -- Id: A.27
  function "rem" (L, R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(R'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where L and R are UNSIGNED vectors.

  -- Id: A.29
  function "rem" (L : STD_ULOGIC_VECTOR; R : NATURAL) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(L'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where L is an UNSIGNED vector and R is a
  --         non-negative INTEGER.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.30
  function "rem" (L : NATURAL; R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(R'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where R is an UNSIGNED vector and L is a
  --         non-negative INTEGER.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  --============================================================================
  --
  -- NOTE: If second argument is zero for "mod" operator, a severity level
  --       of ERROR is issued.

  -- Id: A.33
  function "mod" (L, R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(R'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L and R are UNSIGNED vectors.

  -- Id: A.35
  function "mod" (L : STD_ULOGIC_VECTOR; R : NATURAL) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(L'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L is an UNSIGNED vector and R
  --         is a non-negative INTEGER.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.36
  function "mod" (L : NATURAL; R : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(R'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where R is an UNSIGNED vector and L
  --         is a non-negative INTEGER.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  --============================================================================
  -- Id: A.39
  function find_leftmost (ARG : STD_ULOGIC_VECTOR; Y : STD_ULOGIC) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Finds the leftmost occurrence of the value of Y in ARG.
  --         Returns the index of the occurrence if it exists, or -1 otherwise.

  -- Id: A.41
  function find_rightmost (ARG : STD_ULOGIC_VECTOR; Y : STD_ULOGIC) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Finds the leftmost occurrence of the value of Y in ARG.
  --         Returns the index of the occurrence if it exists, or -1 otherwise.

  --============================================================================
  -- Comparison Operators
  --============================================================================

  -- Id: C.1
  function ">" (L, R : STD_ULOGIC_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.3
  function ">" (L : NATURAL; R : STD_ULOGIC_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L is a non-negative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.5
  function ">" (L : STD_ULOGIC_VECTOR; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L is an UNSIGNED vector and
  --         R is a non-negative INTEGER.

  --============================================================================

  -- Id: C.7
  function "<" (L, R : STD_ULOGIC_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.9
  function "<" (L : NATURAL; R : STD_ULOGIC_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L is a non-negative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.11
  function "<" (L : STD_ULOGIC_VECTOR; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L is an UNSIGNED vector and
  --         R is a non-negative INTEGER.

  --============================================================================

  -- Id: C.13
  function "<=" (L, R : STD_ULOGIC_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.15
  function "<=" (L : NATURAL; R : STD_ULOGIC_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L is a non-negative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.17
  function "<=" (L : STD_ULOGIC_VECTOR; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L is an UNSIGNED vector and
  --         R is a non-negative INTEGER.

  --============================================================================

  -- Id: C.19
  function ">=" (L, R : STD_ULOGIC_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.21
  function ">=" (L : NATURAL; R : STD_ULOGIC_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L is a non-negative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.23
  function ">=" (L : STD_ULOGIC_VECTOR; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L is an UNSIGNED vector and
  --         R is a non-negative INTEGER.

  --============================================================================

  -- Id: C.25
  function "=" (L, R : STD_ULOGIC_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.27
  function "=" (L : NATURAL; R : STD_ULOGIC_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L is a non-negative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.29
  function "=" (L : STD_ULOGIC_VECTOR; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L is an UNSIGNED vector and
  --         R is a non-negative INTEGER.

  --============================================================================

  -- Id: C.31
  function "/=" (L, R : STD_ULOGIC_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.33
  function "/=" (L : NATURAL; R : STD_ULOGIC_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L is a non-negative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.35
  function "/=" (L : STD_ULOGIC_VECTOR; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L is an UNSIGNED vector and
  --         R is a non-negative INTEGER.

  --============================================================================
  -- Shift and Rotate Functions
  --============================================================================

  -- Id: S.1
  function SHIFT_LEFT (ARG : STD_ULOGIC_VECTOR; COUNT : NATURAL)
    return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(ARG'LENGTH-1 downto 0)
  -- Result: Performs a shift-left on an UNSIGNED vector COUNT times.
  --         The vacated positions are filled with '0'.
  --         The COUNT leftmost elements are lost.

  -- Id: S.2
  function SHIFT_RIGHT (ARG : STD_ULOGIC_VECTOR; COUNT : NATURAL)
    return STD_ULOGIC_VECTOR;
  -- Result subtype: UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a shift-right on an UNSIGNED vector COUNT times.
  --         The vacated positions are filled with '0'.
  --         The COUNT rightmost elements are lost.
  --============================================================================

  -- Id: S.5
  function ROTATE_LEFT (ARG : STD_ULOGIC_VECTOR; COUNT : NATURAL)
    return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(ARG'LENGTH-1 downto 0)
  -- Result: Performs a rotate-left of an UNSIGNED vector COUNT times.

  -- Id: S.6
  function ROTATE_RIGHT (ARG : STD_ULOGIC_VECTOR; COUNT : NATURAL)
    return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(ARG'LENGTH-1 downto 0)
  -- Result: Performs a rotate-right of an UNSIGNED vector COUNT times.


  --============================================================================

  ------------------------------------------------------------------------------
  -- Note: Function S.17 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.17
  function "sla" (ARG : STD_ULOGIC_VECTOR; COUNT : INTEGER) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.19 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.19
  function "sra" (ARG : STD_ULOGIC_VECTOR; COUNT : INTEGER) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_RIGHT(ARG, COUNT)


  --============================================================================
  --   RESIZE Functions
  --============================================================================

  -- Id: R.2
  function RESIZE (ARG : STD_ULOGIC_VECTOR; NEW_SIZE : NATURAL)
    return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(NEW_SIZE-1 downto 0)
  -- Result: Resizes the UNSIGNED vector ARG to the specified size.
  --         To create a larger vector, the new [leftmost] bit positions
  --         are filled with '0'. When truncating, the leftmost bits
  --         are dropped.

  function RESIZE (ARG, SIZE_RES : STD_ULOGIC_VECTOR) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR (SIZE_RES'length-1 downto 0)

  --============================================================================
  -- Conversion Functions
  --============================================================================

  -- Id: D.1
  function TO_INTEGER (ARG : STD_ULOGIC_VECTOR) return NATURAL;
  -- Result subtype: NATURAL. Value cannot be negative since parameter is an
  --             UNSIGNED vector.
  -- Result: Converts the UNSIGNED vector to an INTEGER.

  -- Id: D.3
  function To_StdLogicVector (ARG, SIZE : NATURAL) return STD_LOGIC_VECTOR;
  -- Result subtype: STD_LOGIC_VECTOR(SIZE-1 downto 0)
  -- Result: Converts a non-negative INTEGER to an UNSIGNED vector with
  --         the specified SIZE.

  function To_StdLogicVector (ARG : NATURAL; SIZE_RES : STD_ULOGIC_VECTOR)
    return STD_LOGIC_VECTOR;
  -- Result subtype: STD_LOGIC_VECTOR(SIZE_RES'length-1 downto 0)

  alias To_Std_Logic_Vector is
    To_StdLogicVector[NATURAL, NATURAL return STD_LOGIC_VECTOR];
  alias To_SLV is
    To_StdLogicVector[NATURAL, NATURAL return STD_LOGIC_VECTOR];
  alias To_Std_Logic_Vector is
    To_StdLogicVector[NATURAL, STD_ULOGIC_VECTOR return STD_LOGIC_VECTOR];
  alias To_SLV is
    To_StdLogicVector[NATURAL, STD_ULOGIC_VECTOR return STD_LOGIC_VECTOR];

  -- Id: D.5
  function To_StdULogicVector (ARG, SIZE : NATURAL) return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_ULOGIC_VECTOR(SIZE-1 downto 0)
  -- Result: Converts a non-negative INTEGER to an UNSIGNED vector with
  --         the specified SIZE.

  function To_StdULogicVector (ARG : NATURAL; SIZE_RES : STD_ULOGIC_VECTOR)
    return STD_ULOGIC_VECTOR;
  -- Result subtype: STD_LOGIC_VECTOR(SIZE_RES'length-1 downto 0)

  alias To_Std_ULogic_Vector is
    To_StdULogicVector[NATURAL, NATURAL return STD_ULOGIC_VECTOR];
  alias To_SULV is
    To_StdULogicVector[NATURAL, NATURAL return STD_ULOGIC_VECTOR];
  alias To_Std_ULogic_Vector is
    To_StdULogicVector[NATURAL, STD_ULOGIC_VECTOR return STD_ULOGIC_VECTOR];
  alias To_SULV is
    To_StdULogicVector[NATURAL, STD_ULOGIC_VECTOR return STD_ULOGIC_VECTOR];

end package NUMERIC_STD_UNSIGNED;

use std.standard.all;
package fixed_float_types is
-- $Date: 2008-04-10 17:16:09 +0930 (Thu, 10 Apr 2008) $

  type fixed_round_style_type is (fixed_round, fixed_truncate);

  type fixed_overflow_style_type is (fixed_saturate, fixed_wrap);

  type round_type is (round_nearest,
                      round_inf,
                      round_neginf,
                      round_zero);

end package fixed_float_types;

use std.standard.all;
use STD.TEXTIO.all;
library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.NUMERIC_STD.all;
use IEEE.fixed_float_types.all;

package fixed_generic_pkg is
  generic (
    -- Rounding routine to use in fixed point, fixed_round or fixed_truncate
    fixed_round_style    : fixed_round_style_type    := fixed_round;
    -- Overflow routine to use in fixed point, fixed_saturate or fixed_wrap
    fixed_overflow_style : fixed_overflow_style_type := fixed_saturate;
    -- Extra bits used in divide routines
    fixed_guard_bits     : NATURAL                   := 3;
    -- If TRUE, then turn off warnings on "X" propagation
    no_warning           : BOOLEAN                   := false
    );
-- $Date: 2008-04-10 17:16:09 +0930 (Thu, 10 Apr 2008) $

  -- base Unsigned fixed point type, downto direction assumed
  type UNRESOLVED_ufixed is array (INTEGER range <>) of STD_ULOGIC;
  -- base Signed fixed point type, downto direction assumed
  type UNRESOLVED_sfixed is array (INTEGER range <>) of STD_ULOGIC;

  alias U_ufixed is UNRESOLVED_ufixed;
  alias U_sfixed is UNRESOLVED_sfixed;

  subtype ufixed is (resolved) UNRESOLVED_ufixed;
  subtype sfixed is (resolved) UNRESOLVED_sfixed;

  -- This version of divide gives the user more control
  -- ufixed(a downto b) / ufixed(c downto d) = ufixed(a-d downto b-c-1)
  function divide (
    l, r                 : UNRESOLVED_ufixed;
    constant round_style : fixed_round_style_type := fixed_round_style;
    constant guard_bits  : NATURAL                := fixed_guard_bits)
    return UNRESOLVED_ufixed;

  -- This version of divide gives the user more control
  -- sfixed(a downto b) / sfixed(c downto d) = sfixed(a-d+1 downto b-c)
  function divide (
    l, r                 : UNRESOLVED_sfixed;
    constant round_style : fixed_round_style_type := fixed_round_style;
    constant guard_bits  : NATURAL                := fixed_guard_bits)
    return UNRESOLVED_sfixed;

  -- These functions return 1/X
  -- 1 / ufixed(a downto b) = ufixed(-b downto -a-1)
  function reciprocal (
    arg                  : UNRESOLVED_ufixed;
    constant round_style : fixed_round_style_type := fixed_round_style;
    constant guard_bits  : NATURAL                := fixed_guard_bits)
    return UNRESOLVED_ufixed;

  -- 1 / sfixed(a downto b) = sfixed(-b+1 downto -a)
  function reciprocal (
    arg                  : UNRESOLVED_sfixed;
    constant round_style : fixed_round_style_type := fixed_round_style;
    constant guard_bits  : NATURAL                := fixed_guard_bits)
    return UNRESOLVED_sfixed;

  -- REM function
  -- ufixed (a downto b) rem ufixed (c downto d)
  --   = ufixed (minimum(a,c) downto minimum(b,d))
  function remainder (
    l, r                 : UNRESOLVED_ufixed;
    constant round_style : fixed_round_style_type := fixed_round_style;
    constant guard_bits  : NATURAL                := fixed_guard_bits)
    return UNRESOLVED_ufixed;

  -- sfixed (a downto b) rem sfixed (c downto d)
  --   = sfixed (minimum(a,c) downto minimum(b,d))
  function remainder (
    l, r                 : UNRESOLVED_sfixed;
    constant round_style : fixed_round_style_type := fixed_round_style;
    constant guard_bits  : NATURAL                := fixed_guard_bits)
    return UNRESOLVED_sfixed;

  -- mod function
  -- ufixed (a downto b) mod ufixed (c downto d)
  --        = ufixed (minimum(a,c) downto minimum(b, d))
  function modulo (
    l, r                 : UNRESOLVED_ufixed;
    constant round_style : fixed_round_style_type := fixed_round_style;
    constant guard_bits  : NATURAL                := fixed_guard_bits)
    return UNRESOLVED_ufixed;

  -- sfixed (a downto b) mod sfixed (c downto d)
  --        = sfixed (c downto minimum(b, d))
  function modulo (
    l, r                    : UNRESOLVED_sfixed;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style;
    constant guard_bits     : NATURAL                   := fixed_guard_bits)
    return UNRESOLVED_sfixed;

  -- Procedure for those who need an "accumulator" function.
  -- add_carry (ufixed(a downto b), ufixed (c downto d))
  --         = ufixed (maximum(a,c) downto minimum(b,d))
  procedure add_carry (
    L, R   : in  UNRESOLVED_ufixed;
    c_in   : in  STD_ULOGIC;
    result : out UNRESOLVED_ufixed;
    c_out  : out STD_ULOGIC);

  -- add_carry (sfixed(a downto b), sfixed (c downto d))
  --         = sfixed (maximum(a,c) downto minimum(b,d))
  procedure add_carry (
    L, R   : in  UNRESOLVED_sfixed;
    c_in   : in  STD_ULOGIC;
    result : out UNRESOLVED_sfixed;
    c_out  : out STD_ULOGIC);

  -- Scales the result by a power of 2.  Width of input = width of output with
  -- the binary point moved.
  function scalb (y : UNRESOLVED_ufixed; N : INTEGER) return UNRESOLVED_ufixed;
  function scalb (y : UNRESOLVED_ufixed; N : UNRESOLVED_SIGNED) return UNRESOLVED_ufixed;
  function scalb (y : UNRESOLVED_sfixed; N : INTEGER) return UNRESOLVED_sfixed;
  function scalb (y : UNRESOLVED_sfixed; N : UNRESOLVED_SIGNED) return UNRESOLVED_sfixed;

  function Is_Negative (arg : UNRESOLVED_sfixed) return BOOLEAN;

  function std_match (l, r : UNRESOLVED_ufixed) return BOOLEAN;
  function std_match (l, r : UNRESOLVED_sfixed) return BOOLEAN;

  function SHIFT_LEFT  (ARG : UNRESOLVED_ufixed; COUNT : NATURAL)
    return UNRESOLVED_ufixed;
  function SHIFT_RIGHT (ARG : UNRESOLVED_ufixed; COUNT : NATURAL)
    return UNRESOLVED_ufixed;
  function SHIFT_LEFT  (ARG : UNRESOLVED_sfixed; COUNT : NATURAL)
    return UNRESOLVED_sfixed;
  function SHIFT_RIGHT (ARG : UNRESOLVED_sfixed; COUNT : NATURAL)
    return UNRESOLVED_sfixed;

  function find_leftmost (arg : UNRESOLVED_ufixed; y : STD_ULOGIC)
    return INTEGER;
  function find_leftmost (arg : UNRESOLVED_sfixed; y : STD_ULOGIC)
    return INTEGER;

  function find_rightmost (arg : UNRESOLVED_ufixed; y : STD_ULOGIC)
    return INTEGER;
  function find_rightmost (arg : UNRESOLVED_sfixed; y : STD_ULOGIC)
    return INTEGER;

  function resize (
    arg                     : UNRESOLVED_ufixed;
    constant left_index     : INTEGER;
    constant right_index    : INTEGER;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_ufixed;

  -- "size_res" functions create the size of the output from the indices
  -- of the "size_res" input.  The actual value of "size_res" is not used.
  function resize (
    arg                     : UNRESOLVED_ufixed;
    size_res                : UNRESOLVED_ufixed;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_ufixed;

  -- Note that in "wrap" mode the sign bit is not replicated.  Thus the
  -- resize of a negative number can have a positive result in wrap mode.
  function resize (
    arg                     : UNRESOLVED_sfixed;
    constant left_index     : INTEGER;
    constant right_index    : INTEGER;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_sfixed;

  function resize (
    arg                     : UNRESOLVED_sfixed;
    size_res                : UNRESOLVED_sfixed;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_sfixed;

  -- integer (natural) to unsigned fixed point.
  -- arguments are the upper and lower bounds of the number, thus
  -- ufixed (7 downto -3) <= to_ufixed (int, 7, -3);
  function to_ufixed (
    arg                     : NATURAL;
    constant left_index     : INTEGER;
    constant right_index    : INTEGER                   := 0;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_ufixed;

  function to_ufixed (
    arg                     : NATURAL;
    size_res                : UNRESOLVED_ufixed;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_ufixed;

  -- real to unsigned fixed point
  function to_ufixed (
    arg                     : REAL;
    constant left_index     : INTEGER;
    constant right_index    : INTEGER;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style;
    constant guard_bits     : NATURAL                   := fixed_guard_bits)
    return UNRESOLVED_ufixed;

  function to_ufixed (
    arg                     : REAL;
    size_res                : UNRESOLVED_ufixed;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style;
    constant guard_bits     : NATURAL                   := fixed_guard_bits)
    return UNRESOLVED_ufixed;

  -- unsigned to unsigned fixed point
  function to_ufixed (
    arg                     : UNRESOLVED_UNSIGNED;
    constant left_index     : INTEGER;
    constant right_index    : INTEGER                   := 0;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_ufixed;

  function to_ufixed (
    arg                     : UNRESOLVED_UNSIGNED;
    size_res                : UNRESOLVED_ufixed;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_ufixed;

  -- Performs a conversion.  ufixed (arg'range) is returned
  function to_ufixed (
    arg : UNRESOLVED_UNSIGNED)
    return UNRESOLVED_ufixed;

  -- unsigned fixed point to unsigned
  function to_unsigned (
    arg                     : UNRESOLVED_ufixed;
    constant size           : NATURAL;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_UNSIGNED;

  -- unsigned fixed point to unsigned
  function to_unsigned (
    arg                     : UNRESOLVED_ufixed;
    size_res                : UNRESOLVED_UNSIGNED;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_UNSIGNED;

  -- unsigned fixed point to real
  function to_real (
    arg : UNRESOLVED_ufixed)
    return REAL;

  -- unsigned fixed point to integer
  function to_integer (
    arg                     : UNRESOLVED_ufixed;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return NATURAL;

  -- Integer to UNRESOLVED_sfixed
  function to_sfixed (
    arg                     : INTEGER;
    constant left_index     : INTEGER;
    constant right_index    : INTEGER                   := 0;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_sfixed;

  function to_sfixed (
    arg                     : INTEGER;
    size_res                : UNRESOLVED_sfixed;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_sfixed;

  -- Real to sfixed
  function to_sfixed (
    arg                     : REAL;
    constant left_index     : INTEGER;
    constant right_index    : INTEGER;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style;
    constant guard_bits     : NATURAL                   := fixed_guard_bits)
    return UNRESOLVED_sfixed;

  function to_sfixed (
    arg                     : REAL;
    size_res                : UNRESOLVED_sfixed;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style;
    constant guard_bits     : NATURAL                   := fixed_guard_bits)
    return UNRESOLVED_sfixed;

  -- signed to sfixed
  function to_sfixed (
    arg                     : UNRESOLVED_SIGNED;
    constant left_index     : INTEGER;
    constant right_index    : INTEGER                   := 0;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_sfixed;

  function to_sfixed (
    arg                     : UNRESOLVED_SIGNED;
    size_res                : UNRESOLVED_sfixed;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_sfixed;

  -- signed to sfixed (output assumed to be size of signed input)
  function to_sfixed (
    arg : UNRESOLVED_SIGNED)
    return UNRESOLVED_sfixed;

  -- Conversion from ufixed to sfixed
  function to_sfixed (
    arg : UNRESOLVED_ufixed)
    return UNRESOLVED_sfixed;

  -- signed fixed point to signed
  function to_signed (
    arg                     : UNRESOLVED_sfixed;
    constant size           : NATURAL;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_SIGNED;

  -- signed fixed point to signed
  function to_signed (
    arg                     : UNRESOLVED_sfixed;
    size_res                : UNRESOLVED_SIGNED;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return UNRESOLVED_SIGNED;

  -- signed fixed point to real
  function to_real (
    arg : UNRESOLVED_sfixed)
    return REAL;

  -- signed fixed point to integer
  function to_integer (
    arg                     : UNRESOLVED_sfixed;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style)
    return INTEGER;

  -- Because of the fairly complicated sizing rules in the fixed point
  -- packages these functions are provided to compute the result ranges
  -- Example:
  -- signal uf1 : ufixed (3 downto -3);
  -- signal uf2 : ufixed (4 downto -2);
  -- signal uf1multuf2 : ufixed (ufixed_high (3, -3, '*', 4, -2) downto
  --                             ufixed_low (3, -3, '*', 4, -2));
  -- uf1multuf2 <= uf1 * uf2;
  -- Valid characters: '+', '-', '*', '/', 'r' or 'R' (rem), 'm' or 'M' (mod),
  --                   '1' (reciprocal), 'a' or 'A' (abs), 'n' or 'N' (unary -)
  function ufixed_high (left_index, right_index   : INTEGER;
                        operation                 : CHARACTER := 'X';
                        left_index2, right_index2 : INTEGER   := 0)
    return INTEGER;

  function ufixed_low (left_index, right_index   : INTEGER;
                       operation                 : CHARACTER := 'X';
                       left_index2, right_index2 : INTEGER   := 0)
    return INTEGER;

  function sfixed_high (left_index, right_index   : INTEGER;
                        operation                 : CHARACTER := 'X';
                        left_index2, right_index2 : INTEGER   := 0)
    return INTEGER;

  function sfixed_low (left_index, right_index   : INTEGER;
                       operation                 : CHARACTER := 'X';
                       left_index2, right_index2 : INTEGER   := 0)
    return INTEGER;

  -- Same as above, but using the "size_res" input only for their ranges:
  -- signal uf1multuf2 : ufixed (ufixed_high (uf1, '*', uf2) downto
  --                             ufixed_low (uf1, '*', uf2));
  -- uf1multuf2 <= uf1 * uf2;
  --
  function ufixed_high (size_res  : UNRESOLVED_ufixed;
                        operation : CHARACTER := 'X';
                        size_res2 : UNRESOLVED_ufixed)
    return INTEGER;

  function ufixed_low (size_res  : UNRESOLVED_ufixed;
                       operation : CHARACTER := 'X';
                       size_res2 : UNRESOLVED_ufixed)
    return INTEGER;

  function sfixed_high (size_res  : UNRESOLVED_sfixed;
                        operation : CHARACTER := 'X';
                        size_res2 : UNRESOLVED_sfixed)
    return INTEGER;

  function sfixed_low (size_res  : UNRESOLVED_sfixed;
                       operation : CHARACTER := 'X';
                       size_res2 : UNRESOLVED_sfixed)
    return INTEGER;

  -- purpose: returns a saturated number
  function saturate (
    constant left_index  : INTEGER;
    constant right_index : INTEGER)
    return UNRESOLVED_ufixed;

  -- purpose: returns a saturated number
  function saturate (
    constant left_index  : INTEGER;
    constant right_index : INTEGER)
    return UNRESOLVED_sfixed;

  function saturate (
    size_res : UNRESOLVED_ufixed)
    return UNRESOLVED_ufixed;

  function saturate (
    size_res : UNRESOLVED_sfixed)
    return UNRESOLVED_sfixed;

  --===========================================================================
  -- Translation Functions
  --===========================================================================

  -- maps meta-logical values
  function to_01 (
    s             : UNRESOLVED_ufixed;
    constant XMAP : STD_ULOGIC := '0')
    return UNRESOLVED_ufixed;

  -- maps meta-logical values
  function to_01 (
    s             : UNRESOLVED_sfixed;
    constant XMAP : STD_ULOGIC := '0')
    return UNRESOLVED_sfixed;

  function Is_X    (arg : UNRESOLVED_ufixed) return BOOLEAN;
  function Is_X    (arg : UNRESOLVED_sfixed) return BOOLEAN;
  function to_X01  (arg : UNRESOLVED_ufixed) return UNRESOLVED_ufixed;
  function to_X01  (arg : UNRESOLVED_sfixed) return UNRESOLVED_sfixed;
  function to_X01Z (arg : UNRESOLVED_ufixed) return UNRESOLVED_ufixed;
  function to_X01Z (arg : UNRESOLVED_sfixed) return UNRESOLVED_sfixed;
  function to_UX01 (arg : UNRESOLVED_ufixed) return UNRESOLVED_ufixed;
  function to_UX01 (arg : UNRESOLVED_sfixed) return UNRESOLVED_sfixed;

  -- straight vector conversion routines, needed for synthesis.
  -- These functions are here so that a std_logic_vector can be
  -- converted to and from sfixed and ufixed.  Note that you can
  -- not convert these vectors because of their negative index.

  function to_slv (
    arg : UNRESOLVED_ufixed)
    return STD_LOGIC_VECTOR;
  alias to_StdLogicVector is to_slv [UNRESOLVED_ufixed
                                     return STD_LOGIC_VECTOR];
  alias to_Std_Logic_Vector is to_slv [UNRESOLVED_ufixed
                                       return STD_LOGIC_VECTOR];

  function to_slv (
    arg : UNRESOLVED_sfixed)
    return STD_LOGIC_VECTOR;
  alias to_StdLogicVector is to_slv [UNRESOLVED_sfixed
                                     return STD_LOGIC_VECTOR];
  alias to_Std_Logic_Vector is to_slv [UNRESOLVED_sfixed
                                       return STD_LOGIC_VECTOR];

  function to_sulv (
    arg : UNRESOLVED_ufixed)
    return STD_ULOGIC_VECTOR;
  alias to_StdULogicVector is to_sulv [UNRESOLVED_ufixed
                                      return STD_ULOGIC_VECTOR];
  alias to_Std_ULogic_Vector is to_sulv [UNRESOLVED_ufixed
                                        return STD_ULOGIC_VECTOR];

  function to_sulv (
    arg : UNRESOLVED_sfixed)
    return STD_ULOGIC_VECTOR;
  alias to_StdULogicVector is to_sulv [UNRESOLVED_sfixed
                                      return STD_ULOGIC_VECTOR];
  alias to_Std_ULogic_Vector is to_sulv [UNRESOLVED_sfixed
                                        return STD_ULOGIC_VECTOR];

  function to_ufixed (
    arg                  : STD_ULOGIC_VECTOR;
    constant left_index  : INTEGER;
    constant right_index : INTEGER)
    return UNRESOLVED_ufixed;

  function to_ufixed (
    arg      : STD_ULOGIC_VECTOR;
    size_res : UNRESOLVED_ufixed)
    return UNRESOLVED_ufixed;

  function to_sfixed (
    arg                  : STD_ULOGIC_VECTOR;
    constant left_index  : INTEGER;
    constant right_index : INTEGER)
    return UNRESOLVED_sfixed;

  function to_sfixed (
    arg      : STD_ULOGIC_VECTOR;
    size_res : UNRESOLVED_sfixed)
    return UNRESOLVED_sfixed;

  -- As a concession to those who use a graphical DSP environment,
  -- these functions take parameters in those tools format and create
  -- fixed point numbers.  These functions are designed to convert from
  -- a std_logic_vector to the VHDL fixed point format using the conventions
  -- of these packages.  In a pure VHDL environment you should use the
  -- "to_ufixed" and "to_sfixed" routines.

  -- unsigned fixed point
  function to_UFix (
    arg      : STD_ULOGIC_VECTOR;
    width    : NATURAL;
    fraction : NATURAL)
    return UNRESOLVED_ufixed;

  -- signed fixed point
  function to_SFix (
    arg      : STD_ULOGIC_VECTOR;
    width    : NATURAL;
    fraction : NATURAL)
    return UNRESOLVED_sfixed;

  -- finding the bounds of a number.  These functions can be used like this:
  -- signal xxx : ufixed (7 downto -3);
  -- -- Which is the same as "ufixed (UFix_high (11,3) downto UFix_low(11,3))"
  -- signal yyy : ufixed (UFix_high (11, 3, "+", 11, 3)
  --               downto UFix_low(11, 3, "+", 11, 3));
  -- Where "11" is the width of xxx (xxx'length),
  -- and 3 is the lower bound (abs (xxx'low))
  -- In a pure VHDL environment use "ufixed_high" and "ufixed_low"

  function UFix_high (width, fraction   : NATURAL;
                      operation         : CHARACTER := 'X';
                      width2, fraction2 : NATURAL   := 0)
    return INTEGER;

  function UFix_low (width, fraction   : NATURAL;
                     operation         : CHARACTER := 'X';
                     width2, fraction2 : NATURAL   := 0)
    return INTEGER;

  -- Same as above but for signed fixed point.  Note that the width
  -- of a signed fixed point number ignores the sign bit, thus
  -- width = sxxx'length-1

  function SFix_high (width, fraction   : NATURAL;
                      operation         : CHARACTER := 'X';
                      width2, fraction2 : NATURAL   := 0)
    return INTEGER;

  function SFix_low (width, fraction   : NATURAL;
                     operation         : CHARACTER := 'X';
                     width2, fraction2 : NATURAL   := 0)
    return INTEGER;

  --===========================================================================
  -- string and textio Functions
  --===========================================================================

  -- purpose: writes fixed point into a line
  procedure WRITE (
    L         : inout LINE;
    VALUE     : in    UNRESOLVED_ufixed;
    JUSTIFIED : in    SIDE  := right;
    FIELD     : in    WIDTH := 0);

  -- purpose: writes fixed point into a line
  procedure WRITE (
    L         : inout LINE;
    VALUE     : in    UNRESOLVED_sfixed;
    JUSTIFIED : in    SIDE  := right;
    FIELD     : in    WIDTH := 0);

  procedure READ(L     : inout LINE;
                 VALUE : out   UNRESOLVED_ufixed);

  procedure READ(L     : inout LINE;
                 VALUE : out   UNRESOLVED_ufixed;
                 GOOD  : out   BOOLEAN);

  procedure READ(L     : inout LINE;
                 VALUE : out   UNRESOLVED_sfixed);

  procedure READ(L     : inout LINE;
                 VALUE : out   UNRESOLVED_sfixed;
                 GOOD  : out   BOOLEAN);

  alias bwrite is WRITE [LINE, UNRESOLVED_ufixed, SIDE, width];
  alias bwrite is WRITE [LINE, UNRESOLVED_sfixed, SIDE, width];
  alias bread is READ [LINE, UNRESOLVED_ufixed];
  alias bread is READ [LINE, UNRESOLVED_ufixed, BOOLEAN];
  alias bread is READ [LINE, UNRESOLVED_sfixed];
  alias bread is READ [LINE, UNRESOLVED_sfixed, BOOLEAN];
  alias BINARY_WRITE is WRITE [LINE, UNRESOLVED_ufixed, SIDE, width];
  alias BINARY_WRITE is WRITE [LINE, UNRESOLVED_sfixed, SIDE, width];
  alias BINARY_READ is READ [LINE, UNRESOLVED_ufixed, BOOLEAN];
  alias BINARY_READ is READ [LINE, UNRESOLVED_ufixed];
  alias BINARY_READ is READ [LINE, UNRESOLVED_sfixed, BOOLEAN];
  alias BINARY_READ is READ [LINE, UNRESOLVED_sfixed];

  -- octal read and write
  procedure OWRITE (
    L         : inout LINE;
    VALUE     : in    UNRESOLVED_ufixed;
    JUSTIFIED : in    SIDE  := right;
    FIELD     : in    WIDTH := 0);

  procedure OWRITE (
    L         : inout LINE;
    VALUE     : in    UNRESOLVED_sfixed;
    JUSTIFIED : in    SIDE  := right;
    FIELD     : in    WIDTH := 0);

  procedure OREAD(L     : inout LINE;
                  VALUE : out   UNRESOLVED_ufixed);

  procedure OREAD(L     : inout LINE;
                  VALUE : out   UNRESOLVED_ufixed;
                  GOOD  : out   BOOLEAN);

  procedure OREAD(L     : inout LINE;
                  VALUE : out   UNRESOLVED_sfixed);

  procedure OREAD(L     : inout LINE;
                  VALUE : out   UNRESOLVED_sfixed;
                  GOOD  : out   BOOLEAN);
  alias OCTAL_READ is OREAD [LINE, UNRESOLVED_ufixed, BOOLEAN];
  alias OCTAL_READ is OREAD [LINE, UNRESOLVED_ufixed];
  alias OCTAL_READ is OREAD [LINE, UNRESOLVED_sfixed, BOOLEAN];
  alias OCTAL_READ is OREAD [LINE, UNRESOLVED_sfixed];
  alias OCTAL_WRITE is OWRITE [LINE, UNRESOLVED_ufixed, SIDE, WIDTH];
  alias OCTAL_WRITE is OWRITE [LINE, UNRESOLVED_sfixed, SIDE, WIDTH];

  -- hex read and write
  procedure HWRITE (
    L         : inout LINE;
    VALUE     : in    UNRESOLVED_ufixed;
    JUSTIFIED : in    SIDE  := right;
    FIELD     : in    WIDTH := 0);

  -- purpose: writes fixed point into a line
  procedure HWRITE (
    L         : inout LINE;
    VALUE     : in    UNRESOLVED_sfixed;
    JUSTIFIED : in    SIDE  := right;
    FIELD     : in    WIDTH := 0);

  procedure HREAD(L     : inout LINE;
                  VALUE : out   UNRESOLVED_ufixed);

  procedure HREAD(L     : inout LINE;
                  VALUE : out   UNRESOLVED_ufixed;
                  GOOD  : out   BOOLEAN);

  procedure HREAD(L     : inout LINE;
                  VALUE : out   UNRESOLVED_sfixed);

  procedure HREAD(L     : inout LINE;
                  VALUE : out   UNRESOLVED_sfixed;
                  GOOD  : out   BOOLEAN);
  alias HEX_READ is HREAD [LINE, UNRESOLVED_ufixed, BOOLEAN];
  alias HEX_READ is HREAD [LINE, UNRESOLVED_sfixed, BOOLEAN];
  alias HEX_READ is HREAD [LINE, UNRESOLVED_ufixed];
  alias HEX_READ is HREAD [LINE, UNRESOLVED_sfixed];
  alias HEX_WRITE is HWRITE [LINE, UNRESOLVED_ufixed, SIDE, WIDTH];
  alias HEX_WRITE is HWRITE [LINE, UNRESOLVED_sfixed, SIDE, WIDTH];

  -- returns a string, useful for:
  -- assert (x = y) report "error found " & to_string(x) severity error;

  -- From string functions allow you to convert a string into a fixed
  -- point number.  Example:
  --  signal uf1 : ufixed (3 downto -3);
  --  uf1 <= from_string ("0110.100", uf1'high, uf1'low); -- 6.5
  -- The "." is optional in this syntax, however it exist and is
  -- in the wrong location an error is produced.  Overflow will
  -- result in saturation.

  function from_string (
    bstring              : STRING;
    constant left_index  : INTEGER;
    constant right_index : INTEGER)
    return UNRESOLVED_ufixed;
  alias from_bstring is from_string [STRING, INTEGER, INTEGER
                                     return UNRESOLVED_ufixed];
  alias from_binary_string is from_string [STRING, INTEGER, INTEGER
                                           return UNRESOLVED_ufixed];

  -- Octal and hex conversions work as follows:
  -- uf1 <= from_hstring ("6.8", 3, -3); -- 6.5 (bottom zeros dropped)
  -- uf1 <= from_ostring ("06.4", 3, -3); -- 6.5 (top zeros dropped)

  function from_ostring (
    ostring              : STRING;
    constant left_index  : INTEGER;
    constant right_index : INTEGER)
    return UNRESOLVED_ufixed;
  alias from_octal_string is from_ostring [STRING, INTEGER, INTEGER
                                           return UNRESOLVED_ufixed];

  function from_hstring (
    hstring              : STRING;
    constant left_index  : INTEGER;
    constant right_index : INTEGER)
    return UNRESOLVED_ufixed;
  alias from_hex_string is from_hstring [STRING, INTEGER, INTEGER
                                         return UNRESOLVED_ufixed];

  function from_string (
    bstring              : STRING;
    constant left_index  : INTEGER;
    constant right_index : INTEGER)
    return UNRESOLVED_sfixed;
  alias from_bstring is from_string [STRING, INTEGER, INTEGER
                                     return UNRESOLVED_sfixed];
  alias from_binary_string is from_string [STRING, INTEGER, INTEGER
                                           return UNRESOLVED_sfixed];

  function from_ostring (
    ostring              : STRING;
    constant left_index  : INTEGER;
    constant right_index : INTEGER)
    return UNRESOLVED_sfixed;
  alias from_octal_string is from_ostring [STRING, INTEGER, INTEGER
                                           return UNRESOLVED_sfixed];

  function from_hstring (
    hstring              : STRING;
    constant left_index  : INTEGER;
    constant right_index : INTEGER)
    return UNRESOLVED_sfixed;
  alias from_hex_string is from_hstring [STRING, INTEGER, INTEGER
                                         return UNRESOLVED_sfixed];

  -- Same as above, "size_res" is used for it's range only.
  function from_string (
    bstring  : STRING;
    size_res : UNRESOLVED_ufixed)
    return UNRESOLVED_ufixed;
  alias from_bstring is from_string [STRING, UNRESOLVED_ufixed
                                     return UNRESOLVED_ufixed];
  alias from_binary_string is from_string [STRING, UNRESOLVED_ufixed
                                           return UNRESOLVED_ufixed];

  function from_ostring (
    ostring  : STRING;
    size_res : UNRESOLVED_ufixed)
    return UNRESOLVED_ufixed;
  alias from_octal_string is from_ostring [STRING, UNRESOLVED_ufixed
                                           return UNRESOLVED_ufixed];

  function from_hstring (
    hstring  : STRING;
    size_res : UNRESOLVED_ufixed)
    return UNRESOLVED_ufixed;
  alias from_hex_string is from_hstring [STRING, UNRESOLVED_ufixed
                                         return UNRESOLVED_ufixed];

  function from_string (
    bstring  : STRING;
    size_res : UNRESOLVED_sfixed)
    return UNRESOLVED_sfixed;
  alias from_bstring is from_string [STRING, UNRESOLVED_sfixed
                                     return UNRESOLVED_sfixed];
  alias from_binary_string is from_string [STRING, UNRESOLVED_sfixed
                                           return UNRESOLVED_sfixed];

  function from_ostring (
    ostring  : STRING;
    size_res : UNRESOLVED_sfixed)
    return UNRESOLVED_sfixed;
  alias from_octal_string is from_ostring [STRING, UNRESOLVED_sfixed
                                           return UNRESOLVED_sfixed];

  function from_hstring (
    hstring  : STRING;
    size_res : UNRESOLVED_sfixed)
    return UNRESOLVED_sfixed;
  alias from_hex_string is from_hstring [STRING, UNRESOLVED_sfixed
                                         return UNRESOLVED_sfixed];

  -- Direct conversion functions.  Example:
  --  signal uf1 : ufixed (3 downto -3);
  --  uf1 <= from_string ("0110.100"); -- 6.5
  -- In this case the "." is not optional, and the size of
  -- the output must match exactly.

  function from_string (
    bstring : STRING)
    return UNRESOLVED_ufixed;
  alias from_bstring is from_string [STRING return UNRESOLVED_ufixed];
  alias from_binary_string is from_string [STRING return UNRESOLVED_ufixed];

  -- Direct octal and hex conversion functions.  In this case
  -- the string lengths must match.  Example:
  -- signal sf1 := sfixed (5 downto -3);
  -- sf1 <= from_ostring ("71.4") -- -6.5

  function from_ostring (
    ostring : STRING)
    return UNRESOLVED_ufixed;
  alias from_octal_string is from_ostring [STRING return UNRESOLVED_ufixed];

  function from_hstring (
    hstring : STRING)
    return UNRESOLVED_ufixed;
  alias from_hex_string is from_hstring [STRING return UNRESOLVED_ufixed];

  function from_string (
    bstring : STRING)
    return UNRESOLVED_sfixed;
  alias from_bstring is from_string [STRING return UNRESOLVED_sfixed];
  alias from_binary_string is from_string [STRING return UNRESOLVED_sfixed];

  function from_ostring (
    ostring : STRING)
    return UNRESOLVED_sfixed;
  alias from_octal_string is from_ostring [STRING return UNRESOLVED_sfixed];

  function from_hstring (
    hstring : STRING)
    return UNRESOLVED_sfixed;
  alias from_hex_string is from_hstring [STRING return UNRESOLVED_sfixed];

end package fixed_generic_pkg;

use std.standard.all;
library IEEE;

package fixed_pkg is new IEEE.fixed_generic_pkg
  generic map (
    fixed_round_style    => IEEE.fixed_float_types.fixed_round,
    fixed_overflow_style => IEEE.fixed_float_types.fixed_saturate,
    fixed_guard_bits     => 3,
    no_warning           => false
    );

use std.standard.all;
use STD.TEXTIO.all;
library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.NUMERIC_STD.all;
use IEEE.fixed_float_types.all;
use IEEE.fixed_pkg.all;

package float_generic_pkg is
  generic (
    -- Defaults for sizing routines, when you do a "to_float" this will be
    -- the default size.  Example float32 would be 8 and 23 (8 downto -23)
    float_exponent_width : NATURAL    := 8;
    float_fraction_width : NATURAL    := 23;
    -- Rounding algorithm, "round_nearest" is default, other valid values
    -- are "round_zero" (truncation), "round_inf" (round up), and
    -- "round_neginf" (round down)
    float_round_style    : round_type := round_nearest;
    -- Denormal numbers (very small numbers near zero) true or false
    float_denormalize    : BOOLEAN    := true;
    -- Turns on NAN processing (invalid numbers and overflow) true of false
    float_check_error    : BOOLEAN    := true;
    -- Guard bits are added to the bottom of every operation for rounding.
    -- any natural number (including 0) are valid.
    float_guard_bits     : NATURAL    := 3;
    -- If TRUE, then turn off warnings on "X" propagation
    no_warning           : BOOLEAN    := false;
    package fixed_pkg is new IEEE.fixed_generic_pkg
                           generic map (<>) );
  -- Note that this is "INTEGER range <>", thus if you use a literal, then the
  -- default range will be (INTEGER'low to INTEGER'low + X)
  type UNRESOLVED_float is array (INTEGER range <>) of STD_ULOGIC;
  alias U_float is UNRESOLVED_float;

  subtype float is (resolved) UNRESOLVED_float;
  -----------------------------------------------------------------------------
  -- Use the float type to define your own floating point numbers.
  -- There must be a negative index or the packages will error out.
  -- Minimum supported is "subtype float7 is float (3 downto -3);"
  -- "subtype float16 is float (6 downto -9);" is probably the smallest
  -- practical one to use.
  -----------------------------------------------------------------------------

  -- IEEE 754 single precision
  subtype UNRESOLVED_float32 is UNRESOLVED_float (8 downto -23);
  alias U_float32 is UNRESOLVED_float32;
  subtype float32 is float (8 downto -23);
  -----------------------------------------------------------------------------
  -- IEEE-754 single precision floating point.  This is a "float"
  -- in C, and a FLOAT in Fortran.  The exponent is 8 bits wide, and
  -- the fraction is 23 bits wide.  This format can hold roughly 7 decimal
  -- digits.  Infinity is 2**127 = 1.7E38 in this number system.
  -- The bit representation is as follows:
  -- 1 09876543 21098765432109876543210
  -- 8 76543210 12345678901234567890123
  -- 0 00000000 00000000000000000000000
  -- 8 7      0 -1                  -23
  -- +/-   exp.  fraction
  -----------------------------------------------------------------------------

  -- IEEE 754 double precision
  subtype UNRESOLVED_float64 is UNRESOLVED_float (11 downto -52);
  alias U_float64 is UNRESOLVED_float64;
  subtype float64 is float (11 downto -52);
  -----------------------------------------------------------------------------
  -- IEEE-754 double precision floating point.  This is a "double float"
  -- in C, and a FLOAT*8 in Fortran.  The exponent is 11 bits wide, and
  -- the fraction is 52 bits wide.  This format can hold roughly 15 decimal
  -- digits.  Infinity is 2**2047 in this number system.
  -- The bit representation is as follows:
  --  3 21098765432 1098765432109876543210987654321098765432109876543210
  --  1 09876543210 1234567890123456789012345678901234567890123456789012
  --  S EEEEEEEEEEE FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
  -- 11 10        0 -1                                               -52
  -- +/-  exponent    fraction
  -----------------------------------------------------------------------------

  -- IEEE 854 & C extended precision
  subtype UNRESOLVED_float128 is UNRESOLVED_float (15 downto -112);
  alias U_float128 is UNRESOLVED_float128;
  subtype float128 is float (15 downto -112);
  -----------------------------------------------------------------------------
  -- The 128 bit floating point number is "long double" in C (on
  -- some systems this is a 70 bit floating point number) and FLOAT*32
  -- in Fortran.  The exponent is 15 bits wide and the fraction is 112
  -- bits wide. This number can handle approximately 33 decimal digits.
  -- Infinity is 2**32,767 in this number system.
  -----------------------------------------------------------------------------

  -- purpose: Checks for a valid floating point number
  type valid_fpstate is (nan,
                         quiet_nan,
                         neg_inf,
                         neg_normal,
                         neg_denormal,
                         neg_zero,
                         pos_zero,
                         pos_denormal,
                         pos_normal,
                         pos_inf,
                         isx);

  -- This deferred constant will tell you if the package body is synthesizable
  -- or implemented as real numbers.
  constant fphdlsynth_or_real : BOOLEAN;

  -- Returns the class which X falls into
  function Classfp (
    x           : UNRESOLVED_float;
    check_error : BOOLEAN := float_check_error)
    return valid_fpstate;

  -- Basic parameter list
  -- round_style - Selects the rounding algorithm to use
  -- guard - extra bits added to the end if the operation to add precision
  -- check_error - When "false" turns off NAN and overflow checks
  -- denormalize - When "false" turns off denormal number processing
  function add (
    l, r                 : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant guard       : NATURAL    := float_guard_bits;
    constant check_error : BOOLEAN    := float_check_error;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  function subtract (
    l, r                 : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant guard       : NATURAL    := float_guard_bits;
    constant check_error : BOOLEAN    := float_check_error;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  function multiply (
    l, r                 : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant guard       : NATURAL    := float_guard_bits;
    constant check_error : BOOLEAN    := float_check_error;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  function divide (
    l, r                 : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant guard       : NATURAL    := float_guard_bits;
    constant check_error : BOOLEAN    := float_check_error;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  function remainder (
    l, r                 : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant guard       : NATURAL    := float_guard_bits;
    constant check_error : BOOLEAN    := float_check_error;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  function modulo (
    l, r                 : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant guard       : NATURAL    := float_guard_bits;
    constant check_error : BOOLEAN    := float_check_error;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  -- reciprocal
  function reciprocal (
    arg                  : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant guard       : NATURAL    := float_guard_bits;
    constant check_error : BOOLEAN    := float_check_error;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  function dividebyp2 (
    l, r                 : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant guard       : NATURAL    := float_guard_bits;
    constant check_error : BOOLEAN    := float_check_error;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  -- Multiply accumulate  result = l*r + c
  function mac (
    l, r, c              : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant guard       : NATURAL    := float_guard_bits;
    constant check_error : BOOLEAN    := float_check_error;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  -- Square root (all 754 based implementations need this)
  function sqrt (
    arg                  : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant guard       : NATURAL    := float_guard_bits;
    constant check_error : BOOLEAN    := float_check_error;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  function Is_Negative (arg : UNRESOLVED_float) return BOOLEAN;

  -- compare functions
  -- =, /=, >=, <=, <, >, maximum, minimum

  function eq (
    l, r                 : UNRESOLVED_float;
    constant check_error : BOOLEAN := float_check_error;
    constant denormalize : BOOLEAN := float_denormalize)
    return BOOLEAN;

  function ne (
    l, r                 : UNRESOLVED_float;
    constant check_error : BOOLEAN := float_check_error;
    constant denormalize : BOOLEAN := float_denormalize)
    return BOOLEAN;

  function lt (
    l, r                 : UNRESOLVED_float;
    constant check_error : BOOLEAN := float_check_error;
    constant denormalize : BOOLEAN := float_denormalize)
    return BOOLEAN;

  function gt (
    l, r                 : UNRESOLVED_float;
    constant check_error : BOOLEAN := float_check_error;
    constant denormalize : BOOLEAN := float_denormalize)
    return BOOLEAN;

  function le (
    l, r                 : UNRESOLVED_float;
    constant check_error : BOOLEAN := float_check_error;
    constant denormalize : BOOLEAN := float_denormalize)
    return BOOLEAN;

  function ge (
    l, r                 : UNRESOLVED_float;
    constant check_error : BOOLEAN := float_check_error;
    constant denormalize : BOOLEAN := float_denormalize)
    return BOOLEAN;

  function std_match (l, r : UNRESOLVED_float) return BOOLEAN;
  function find_rightmost (arg : UNRESOLVED_float; y : STD_ULOGIC)
    return INTEGER;
  function find_leftmost (arg : UNRESOLVED_float; y : STD_ULOGIC)
    return INTEGER;

  -- conversion functions
  -- Converts one floating point number into another.

  function resize (
    arg                     : UNRESOLVED_float;
    constant exponent_width : NATURAL    := float_exponent_width;
    constant fraction_width : NATURAL    := float_fraction_width;
    constant round_style    : round_type := float_round_style;
    constant check_error    : BOOLEAN    := float_check_error;
    constant denormalize_in : BOOLEAN    := float_denormalize;
    constant denormalize    : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  function resize (
    arg                     : UNRESOLVED_float;
    size_res                : UNRESOLVED_float;
    constant round_style    : round_type := float_round_style;
    constant check_error    : BOOLEAN    := float_check_error;
    constant denormalize_in : BOOLEAN    := float_denormalize;
    constant denormalize    : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  function to_float32 (
    arg                     : UNRESOLVED_float;
    constant round_style    : round_type := float_round_style;
    constant check_error    : BOOLEAN    := float_check_error;
    constant denormalize_in : BOOLEAN    := float_denormalize;
    constant denormalize    : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float32;

  function to_float64 (
    arg                     : UNRESOLVED_float;
    constant round_style    : round_type := float_round_style;
    constant check_error    : BOOLEAN    := float_check_error;
    constant denormalize_in : BOOLEAN    := float_denormalize;
    constant denormalize    : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float64;

  function to_float128 (
    arg                     : UNRESOLVED_float;
    constant round_style    : round_type := float_round_style;
    constant check_error    : BOOLEAN    := float_check_error;
    constant denormalize_in : BOOLEAN    := float_denormalize;
    constant denormalize    : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float128;

  -- Converts an fp into an SLV (needed for synthesis)
  function to_slv (arg : UNRESOLVED_float) return STD_LOGIC_VECTOR;
  alias to_StdLogicVector is to_slv [UNRESOLVED_float return STD_LOGIC_VECTOR];
  alias to_Std_Logic_Vector is to_slv [UNRESOLVED_float return STD_LOGIC_VECTOR];

  -- Converts an fp into an std_ulogic_vector (sulv)
  function to_sulv (arg : UNRESOLVED_float) return STD_ULOGIC_VECTOR;
  alias to_StdULogicVector is to_sulv [UNRESOLVED_float return STD_ULOGIC_VECTOR];
  alias to_Std_ULogic_Vector is to_sulv [UNRESOLVED_float return STD_ULOGIC_VECTOR];

  -- std_ulogic_vector to float
  function to_float (
    arg                     : STD_ULOGIC_VECTOR;
    constant exponent_width : NATURAL := float_exponent_width;
    constant fraction_width : NATURAL := float_fraction_width)
    return UNRESOLVED_float;

  -- Integer to float
  function to_float (
    arg                     : INTEGER;
    constant exponent_width : NATURAL    := float_exponent_width;
    constant fraction_width : NATURAL    := float_fraction_width;
    constant round_style    : round_type := float_round_style)
    return UNRESOLVED_float;

  -- real to float
  function to_float (
    arg                     : REAL;
    constant exponent_width : NATURAL    := float_exponent_width;
    constant fraction_width : NATURAL    := float_fraction_width;
    constant round_style    : round_type := float_round_style;
    constant denormalize    : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  -- unsigned to float
  function to_float (
    arg                     : UNRESOLVED_UNSIGNED;
    constant exponent_width : NATURAL    := float_exponent_width;
    constant fraction_width : NATURAL    := float_fraction_width;
    constant round_style    : round_type := float_round_style)
    return UNRESOLVED_float;

  -- signed to float
  function to_float (
    arg                     : UNRESOLVED_SIGNED;
    constant exponent_width : NATURAL    := float_exponent_width;
    constant fraction_width : NATURAL    := float_fraction_width;
    constant round_style    : round_type := float_round_style)
    return UNRESOLVED_float;

  -- unsigned fixed point to float
  function to_float (
    arg                     : UNRESOLVED_ufixed;
    constant exponent_width : NATURAL    := float_exponent_width;
    constant fraction_width : NATURAL    := float_fraction_width;
    constant round_style    : round_type := float_round_style;
    constant denormalize    : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  -- signed fixed point to float
  function to_float (
    arg                     : UNRESOLVED_sfixed;
    constant exponent_width : NATURAL    := float_exponent_width;
    constant fraction_width : NATURAL    := float_fraction_width;
    constant round_style    : round_type := float_round_style;
    constant denormalize    : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  -- size_res functions
  -- Integer to float
  function to_float (
    arg                  : INTEGER;
    size_res             : UNRESOLVED_float;
    constant round_style : round_type := float_round_style)
    return UNRESOLVED_float;

  -- real to float
  function to_float (
    arg                  : REAL;
    size_res             : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  -- unsigned to float
  function to_float (
    arg                  : UNRESOLVED_UNSIGNED;
    size_res             : UNRESOLVED_float;
    constant round_style : round_type := float_round_style)
    return UNRESOLVED_float;

  -- signed to float
  function to_float (
    arg                  : UNRESOLVED_SIGNED;
    size_res             : UNRESOLVED_float;
    constant round_style : round_type := float_round_style)
    return UNRESOLVED_float;

  -- sulv to float
  function to_float (
    arg      : STD_ULOGIC_VECTOR;
    size_res : UNRESOLVED_float)
    return UNRESOLVED_float;

  -- unsigned fixed point to float
  function to_float (
    arg                  : UNRESOLVED_ufixed;
    size_res             : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  -- signed fixed point to float
  function to_float (
    arg                  : UNRESOLVED_sfixed;
    size_res             : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  -- float to unsigned
  function to_unsigned (
    arg                  : UNRESOLVED_float;
    constant size        : NATURAL;
    constant round_style : round_type := float_round_style;
    constant check_error : BOOLEAN    := float_check_error)
    return UNRESOLVED_UNSIGNED;

  -- float to signed
  function to_signed (
    arg                  : UNRESOLVED_float;
    constant size        : NATURAL;
    constant round_style : round_type := float_round_style;
    constant check_error : BOOLEAN    := float_check_error)
    return UNRESOLVED_SIGNED;

  -- purpose: Converts a float to unsigned fixed point
  function to_ufixed (
    arg                     : UNRESOLVED_float;
    constant left_index     : INTEGER;
    constant right_index    : INTEGER;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style;
    constant check_error    : BOOLEAN                   := float_check_error;
    constant denormalize    : BOOLEAN                   := float_denormalize)
    return UNRESOLVED_ufixed;

  -- float to signed fixed point
  function to_sfixed (
    arg                     : UNRESOLVED_float;
    constant left_index     : INTEGER;
    constant right_index    : INTEGER;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style;
    constant check_error    : BOOLEAN                   := float_check_error;
    constant denormalize    : BOOLEAN                   := float_denormalize)
    return UNRESOLVED_sfixed;

  -- size_res versions
  -- float to unsigned
  function to_unsigned (
    arg                  : UNRESOLVED_float;
    size_res             : UNRESOLVED_UNSIGNED;
    constant round_style : round_type := float_round_style;
    constant check_error : BOOLEAN    := float_check_error)
    return UNRESOLVED_UNSIGNED;

  -- float to signed
  function to_signed (
    arg                  : UNRESOLVED_float;
    size_res             : UNRESOLVED_SIGNED;
    constant round_style : round_type := float_round_style;
    constant check_error : BOOLEAN    := float_check_error)
    return UNRESOLVED_SIGNED;

  -- purpose: Converts a float to unsigned fixed point
  function to_ufixed (
    arg                     : UNRESOLVED_float;
    size_res                : UNRESOLVED_ufixed;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style;
    constant check_error    : BOOLEAN                   := float_check_error;
    constant denormalize    : BOOLEAN                   := float_denormalize)
    return UNRESOLVED_ufixed;

  -- float to signed fixed point
  function to_sfixed (
    arg                     : UNRESOLVED_float;
    size_res                : UNRESOLVED_sfixed;
    constant overflow_style : fixed_overflow_style_type := fixed_overflow_style;
    constant round_style    : fixed_round_style_type    := fixed_round_style;
    constant check_error    : BOOLEAN                   := float_check_error;
    constant denormalize    : BOOLEAN                   := float_denormalize)
    return UNRESOLVED_sfixed;

  -- float to real
  function to_real (
    arg                  : UNRESOLVED_float;
    constant check_error : BOOLEAN    := float_check_error;
    constant denormalize : BOOLEAN    := float_denormalize)
    return REAL;

  -- float to integer
  function to_integer (
    arg                  : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant check_error : BOOLEAN    := float_check_error)
    return INTEGER;

  -- For Verilog compatability
  function realtobits (arg : REAL) return STD_ULOGIC_VECTOR;
  function bitstoreal (arg : STD_ULOGIC_VECTOR) return REAL;

  -- Maps metalogical values
  function to_01 (
    arg  : UNRESOLVED_float;
    XMAP : STD_LOGIC := '0')
    return UNRESOLVED_float;

  function Is_X (arg    : UNRESOLVED_float) return BOOLEAN;
  function to_X01 (arg  : UNRESOLVED_float) return UNRESOLVED_float;
  function to_X01Z (arg : UNRESOLVED_float) return UNRESOLVED_float;
  function to_UX01 (arg : UNRESOLVED_float) return UNRESOLVED_float;

  -- These two procedures were copied out of the body because they proved
  -- very useful for vendor specific algorithm development
  -- Break_number converts a floating point number into it's parts
  -- Exponent is biased by -1

  procedure break_number (
    arg         : in  UNRESOLVED_float;
    denormalize : in  BOOLEAN := float_denormalize;
    check_error : in  BOOLEAN := float_check_error;
    fract       : out UNRESOLVED_UNSIGNED;
    expon       : out UNRESOLVED_SIGNED;
    sign        : out STD_ULOGIC);

  procedure break_number (
    arg         : in  UNRESOLVED_float;
    denormalize : in  BOOLEAN := float_denormalize;
    check_error : in  BOOLEAN := float_check_error;
    fract       : out UNRESOLVED_ufixed;
    expon       : out UNRESOLVED_SIGNED;
    sign        : out STD_ULOGIC);

  -- Normalize takes a fraction and and exponent and converts them into
  -- a floating point number.  Does the shifting and the rounding.
  -- Exponent is assumed to be biased by -1

  function normalize (
    fract                   : UNRESOLVED_UNSIGNED;
    expon                   : UNRESOLVED_SIGNED;
    sign                    : STD_ULOGIC;
    sticky                  : STD_ULOGIC := '0';
    constant exponent_width : NATURAL    := float_exponent_width;
    constant fraction_width : NATURAL    := float_fraction_width;
    constant round_style    : round_type := float_round_style;
    constant denormalize    : BOOLEAN    := float_denormalize;
    constant nguard         : NATURAL    := float_guard_bits)
    return UNRESOLVED_float;

  -- Exponent is assumed to be biased by -1
  function normalize (
    fract                   : UNRESOLVED_ufixed;
    expon                   : UNRESOLVED_SIGNED;
    sign                    : STD_ULOGIC;
    sticky                  : STD_ULOGIC := '0';
    constant exponent_width : NATURAL    := float_exponent_width;
    constant fraction_width : NATURAL    := float_fraction_width;
    constant round_style    : round_type := float_round_style;
    constant denormalize    : BOOLEAN    := float_denormalize;
    constant nguard         : NATURAL    := float_guard_bits)
    return UNRESOLVED_float;

  function normalize (
    fract                : UNRESOLVED_UNSIGNED;
    expon                : UNRESOLVED_SIGNED;
    sign                 : STD_ULOGIC;
    sticky               : STD_ULOGIC := '0';
    size_res             : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant denormalize : BOOLEAN    := float_denormalize;
    constant nguard      : NATURAL    := float_guard_bits)
    return UNRESOLVED_float;

  -- Exponent is assumed to be biased by -1
  function normalize (
    fract                : UNRESOLVED_ufixed;
    expon                : UNRESOLVED_SIGNED;
    sign                 : STD_ULOGIC;
    sticky               : STD_ULOGIC := '0';
    size_res             : UNRESOLVED_float;
    constant round_style : round_type := float_round_style;
    constant denormalize : BOOLEAN    := float_denormalize;
    constant nguard      : NATURAL    := float_guard_bits)
    return UNRESOLVED_float;

  -- returns x with the sign of y.
  function Copysign (x, y : UNRESOLVED_float) return UNRESOLVED_float;
  -- Returns y * 2**n for integral values of N without computing 2**n
  function Scalb (
    y                    : UNRESOLVED_float;
    N                    : INTEGER;
    constant round_style : round_type := float_round_style;
    constant check_error : BOOLEAN    := float_check_error;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  -- Returns y * 2**n for integral values of N without computing 2**n
  function Scalb (
    y                    : UNRESOLVED_float;
    N                    : UNRESOLVED_SIGNED;
    constant round_style : round_type := float_round_style;
    constant check_error : BOOLEAN    := float_check_error;
    constant denormalize : BOOLEAN    := float_denormalize)
    return UNRESOLVED_float;

  -- returns the unbiased exponent of x
  function Logb (x : UNRESOLVED_float) return INTEGER;
  function Logb (x : UNRESOLVED_float) return UNRESOLVED_SIGNED;

  -- returns the next representable neighbor of x in the direction toward y
  function Nextafter (
    x, y                 : UNRESOLVED_float;
    constant check_error : BOOLEAN := float_check_error;
    constant denormalize : BOOLEAN := float_denormalize)
    return UNRESOLVED_float;

  -- Returns TRUE if X is unordered with Y.
  function Unordered (x, y : UNRESOLVED_float) return BOOLEAN;
  function Finite (x       : UNRESOLVED_float) return BOOLEAN;
  function Isnan (x        : UNRESOLVED_float) return BOOLEAN;

  -- Function to return constants.
  function zerofp (
    constant exponent_width : NATURAL := float_exponent_width;
    constant fraction_width : NATURAL := float_fraction_width)
    return UNRESOLVED_float;
  function nanfp (
    constant exponent_width : NATURAL := float_exponent_width;
    constant fraction_width : NATURAL := float_fraction_width)
    return UNRESOLVED_float;
  function qnanfp (
    constant exponent_width : NATURAL := float_exponent_width;
    constant fraction_width : NATURAL := float_fraction_width)
    return UNRESOLVED_float;
  function pos_inffp (
    constant exponent_width : NATURAL := float_exponent_width;
    constant fraction_width : NATURAL := float_fraction_width)
    return UNRESOLVED_float;
  function neg_inffp (
    constant exponent_width : NATURAL := float_exponent_width;
    constant fraction_width : NATURAL := float_fraction_width)
    return UNRESOLVED_float;
  function neg_zerofp (
    constant exponent_width : NATURAL := float_exponent_width;
    constant fraction_width : NATURAL := float_fraction_width)
    return UNRESOLVED_float;
  -- size_res versions
  function zerofp (
    size_res : UNRESOLVED_float)
    return UNRESOLVED_float;
  function nanfp (
    size_res : UNRESOLVED_float)
    return UNRESOLVED_float;
  function qnanfp (
    size_res : UNRESOLVED_float)
    return UNRESOLVED_float;
  function pos_inffp (
    size_res : UNRESOLVED_float)
    return UNRESOLVED_float;
  function neg_inffp (
    size_res : UNRESOLVED_float)
    return UNRESOLVED_float;
  function neg_zerofp (
    size_res : UNRESOLVED_float)
    return UNRESOLVED_float;

  --===========================================================================
  -- string and textio Functions
  --===========================================================================

  -- writes S:EEEE:FFFFFFFF
  procedure WRITE (
    L         : inout LINE;
    VALUE     : in    UNRESOLVED_float;
    JUSTIFIED : in    SIDE  := right;
    FIELD     : in    WIDTH := 0);

  -- Reads SEEEEFFFFFFFF, "." and ":" are ignored
  procedure READ (L    : inout LINE; VALUE : out UNRESOLVED_float);
  procedure READ (L    : inout LINE; VALUE : out UNRESOLVED_float;
                  GOOD : out   BOOLEAN);

  alias BREAD is READ [LINE, UNRESOLVED_float, BOOLEAN];
  alias BREAD is READ [LINE, UNRESOLVED_float];
  alias BWRITE is WRITE [LINE, UNRESOLVED_float, SIDE, WIDTH];
  alias BINARY_READ is READ [LINE, UNRESOLVED_FLOAT, BOOLEAN];
  alias BINARY_READ is READ [LINE, UNRESOLVED_FLOAT];
  alias BINARY_WRITE is WRITE [LINE, UNRESOLVED_float, SIDE, WIDTH];

  procedure OWRITE (
    L         : inout LINE;
    VALUE     : in    UNRESOLVED_float;
    JUSTIFIED : in    SIDE  := right;
    FIELD     : in    WIDTH := 0);

  -- Octal read with padding, no separators used
  procedure OREAD (L    : inout LINE; VALUE : out UNRESOLVED_float);
  procedure OREAD (L    : inout LINE; VALUE : out UNRESOLVED_float;
                   GOOD : out   BOOLEAN);
  alias OCTAL_READ is OREAD [LINE, UNRESOLVED_FLOAT, BOOLEAN];
  alias OCTAL_READ is OREAD [LINE, UNRESOLVED_FLOAT];
  alias OCTAL_WRITE is OWRITE [LINE, UNRESOLVED_FLOAT, SIDE, WIDTH];

  -- Hex write with padding, no separators
  procedure HWRITE (
    L         : inout LINE;
    VALUE     : in    UNRESOLVED_float;
    JUSTIFIED : in    SIDE  := right;
    FIELD     : in    WIDTH := 0);

  -- Hex read with padding, no separators used
  procedure HREAD (L : inout LINE; VALUE : out UNRESOLVED_float);
  procedure HREAD (L    : inout LINE; VALUE : out UNRESOLVED_float;
                   GOOD : out   BOOLEAN);
  alias HEX_READ is HREAD [LINE, UNRESOLVED_FLOAT, BOOLEAN];
  alias HEX_READ is HREAD [LINE, UNRESOLVED_FLOAT];
  alias HEX_WRITE is HWRITE [LINE, UNRESOLVED_FLOAT, SIDE, WIDTH];

  -- returns "S:EEEE:FFFFFFFF"

  function from_string (
    bstring                 : STRING;
    constant exponent_width : NATURAL := float_exponent_width;
    constant fraction_width : NATURAL := float_fraction_width)
    return UNRESOLVED_float;
  alias from_bstring is from_string [STRING, NATURAL, NATURAL
                                     return UNRESOLVED_float];
  alias from_binary_string is from_string [STRING, NATURAL, NATURAL
                                           return UNRESOLVED_float];
  function from_ostring (
    ostring                 : STRING;
    constant exponent_width : NATURAL := float_exponent_width;
    constant fraction_width : NATURAL := float_fraction_width)
    return UNRESOLVED_float;
  alias from_octal_string is from_ostring [STRING, NATURAL, NATURAL
                                           return UNRESOLVED_float];

  function from_hstring (
    hstring                 : STRING;
    constant exponent_width : NATURAL := float_exponent_width;
    constant fraction_width : NATURAL := float_fraction_width)
    return UNRESOLVED_float;
  alias from_hex_string is from_hstring [STRING, NATURAL, NATURAL
                                         return UNRESOLVED_float];

  function from_string (
    bstring  : STRING;
    size_res : UNRESOLVED_float)
    return UNRESOLVED_float;
  alias from_bstring is from_string [STRING, UNRESOLVED_float
                                     return UNRESOLVED_float];
  alias from_binary_string is from_string [STRING, UNRESOLVED_float
                                           return UNRESOLVED_float];

  function from_ostring (
    ostring  : STRING;
    size_res : UNRESOLVED_float)
    return UNRESOLVED_float;
  alias from_octal_string is from_ostring [STRING, UNRESOLVED_float
                                           return UNRESOLVED_float];

  function from_hstring (
    hstring  : STRING;
    size_res : UNRESOLVED_float)
    return UNRESOLVED_float;
  alias from_hex_string is from_hstring [STRING, UNRESOLVED_float
                                         return UNRESOLVED_float];

end package float_generic_pkg;

use std.standard.all;
library ieee;

package float_pkg is new IEEE.float_generic_pkg
  generic map (
    float_exponent_width => 8,
    float_fraction_width => 23,
    float_round_style    => IEEE.fixed_float_types.round_nearest,
    float_denormalize    => true,
    float_check_error    => true,
    float_guard_bits     => 3,
    no_warning           => false,
    fixed_pkg            => IEEE.fixed_pkg
    );

use std.standard.all;
package NUMERIC_BIT_UNSIGNED is

  -- Id: A.3
  function "+" (L, R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: bit_vector(MAXIMUM(L'LENGTH, R'LENGTH)-1 downto 0).
  -- Result: Adds two UNSIGNED vectors that may be of different lengths.

  -- Id: A.3R
  function "+"(L : BIT_VECTOR; R : BIT) return BIT_VECTOR;
  -- Result subtype: bit_vector(L'LENGTH-1 downto 0)
  -- Result: Similar to A.3 where R is a one bit bit_vector

  -- Id: A.3L
  function "+"(L : BIT; R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: bit_vector(R'LENGTH-1 downto 0)
  -- Result: Similar to A.3 where L is a one bit UNSIGNED

  -- Id: A.5
  function "+" (L : BIT_VECTOR; R : NATURAL) return BIT_VECTOR;
  -- Result subtype: bit_vector(L'LENGTH-1 downto 0).
  -- Result: Adds an UNSIGNED vector, L, with a non-negative INTEGER, R.

  -- Id: A.6
  function "+" (L : NATURAL; R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: bit_vector(R'LENGTH-1 downto 0).
  -- Result: Adds a non-negative INTEGER, L, with an UNSIGNED vector, R.

  --============================================================================

  -- Id: A.9
  function "-" (L, R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: UNSIGNED(MAXIMUM(L'LENGTH, R'LENGTH)-1 downto 0).
  -- Result: Subtracts two UNSIGNED vectors that may be of different lengths.

  -- Id: A.9R
  function "-"(L : BIT_VECTOR; R : BIT) return BIT_VECTOR;
  -- Result subtype: bit_vector(L'LENGTH-1 downto 0)
  -- Result: Similar to A.9 where R is a one bit UNSIGNED

  -- Id: A.9L
  function "-"(L : BIT; R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: bit_vector(R'LENGTH-1 downto 0)
  -- Result: Similar to A.9 where L is a one bit UNSIGNED

  -- Id: A.11
  function "-" (L : BIT_VECTOR; R : NATURAL) return BIT_VECTOR;
  -- Result subtype: bit_vector(L'LENGTH-1 downto 0).
  -- Result: Subtracts a non-negative INTEGER, R, from an UNSIGNED vector, L.

  -- Id: A.12
  function "-" (L : NATURAL; R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: bit_vector(R'LENGTH-1 downto 0).
  -- Result: Subtracts an UNSIGNED vector, R, from a non-negative INTEGER, L.

  --============================================================================

  -- Id: A.15
  function "*" (L, R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: bit_vector((L'LENGTH+R'LENGTH-1) downto 0).
  -- Result: Performs the multiplication operation on two UNSIGNED vectors
  --         that may possibly be of different lengths.

  -- Id: A.17
  function "*" (L : BIT_VECTOR; R : NATURAL) return BIT_VECTOR;
  -- Result subtype: bit_vector((L'LENGTH+L'LENGTH-1) downto 0).
  -- Result: Multiplies an UNSIGNED vector, L, with a non-negative
  --         INTEGER, R. R is converted to an UNSIGNED vector of
  --         SIZE L'LENGTH before multiplication.

  -- Id: A.18
  function "*" (L : NATURAL; R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: bit_vector((R'LENGTH+R'LENGTH-1) downto 0).
  -- Result: Multiplies an UNSIGNED vector, R, with a non-negative
  --         INTEGER, L. L is converted to an UNSIGNED vector of
  --         SIZE R'LENGTH before multiplication.

  --============================================================================
  --
  -- NOTE: If second argument is zero for "/" operator, a severity level
  --       of ERROR is issued.

  -- Id: A.21
  function "/" (L, R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: bit_vector(L'LENGTH-1 downto 0)
  -- Result: Divides an UNSIGNED vector, L, by another UNSIGNED vector, R.

  -- Id: A.23
  function "/" (L : BIT_VECTOR; R : NATURAL) return BIT_VECTOR;
  -- Result subtype: bit_vector(L'LENGTH-1 downto 0)
  -- Result: Divides an UNSIGNED vector, L, by a non-negative INTEGER, R.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.24
  function "/" (L : NATURAL; R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: bit_vector(R'LENGTH-1 downto 0)
  -- Result: Divides a non-negative INTEGER, L, by an UNSIGNED vector, R.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  --============================================================================
  --
  -- NOTE: If second argument is zero for "rem" operator, a severity level
  --       of ERROR is issued.

  -- Id: A.27
  function "rem" (L, R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: bit_vector(R'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where L and R are UNSIGNED vectors.

  -- Id: A.29
  function "rem" (L : BIT_VECTOR; R : NATURAL) return BIT_VECTOR;
  -- Result subtype: bit_vector(L'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where L is an UNSIGNED vector and R is a
  --         non-negative INTEGER.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.30
  function "rem" (L : NATURAL; R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: bit_vector(R'LENGTH-1 downto 0)
  -- Result: Computes "L rem R" where R is an UNSIGNED vector and L is a
  --         non-negative INTEGER.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  --============================================================================
  --
  -- NOTE: If second argument is zero for "mod" operator, a severity level
  --       of ERROR is issued.

  -- Id: A.33
  function "mod" (L, R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: bit_vector(R'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L and R are UNSIGNED vectors.

  -- Id: A.35
  function "mod" (L : BIT_VECTOR; R : NATURAL) return BIT_VECTOR;
  -- Result subtype: bit_vector(L'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where L is an UNSIGNED vector and R
  --         is a non-negative INTEGER.
  --         If NO_OF_BITS(R) > L'LENGTH, result is truncated to L'LENGTH.

  -- Id: A.36
  function "mod" (L : NATURAL; R : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: bit_vector(R'LENGTH-1 downto 0)
  -- Result: Computes "L mod R" where R is an UNSIGNED vector and L
  --         is a non-negative INTEGER.
  --         If NO_OF_BITS(L) > R'LENGTH, result is truncated to R'LENGTH.

  --============================================================================
  -- Id: A.39
  function find_leftmost (ARG : BIT_VECTOR; Y : BIT) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Finds the leftmost occurrence of the value of Y in ARG.
  --         Returns the index of the occurrence if it exists, or -1 otherwise.

  -- Id: A.41
  function find_rightmost (ARG : BIT_VECTOR; Y : BIT) return INTEGER;
  -- Result subtype: INTEGER
  -- Result: Finds the leftmost occurrence of the value of Y in ARG.
  --         Returns the index of the occurrence if it exists, or -1 otherwise.

  --============================================================================
  -- Comparison Operators
  --============================================================================
  -- Id: C.1
  function ">" (L, R : BIT_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.3
  function ">" (L : NATURAL; R : BIT_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L is a non-negative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.5
  function ">" (L : BIT_VECTOR; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L > R" where L is an UNSIGNED vector and
  --         R is a non-negative INTEGER.

  --============================================================================
  -- Id: C.7
  function "<" (L, R : BIT_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.9
  function "<" (L : NATURAL; R : BIT_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L is a non-negative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.11
  function "<" (L : BIT_VECTOR; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L < R" where L is an UNSIGNED vector and
  --         R is a non-negative INTEGER.

  --============================================================================
  -- Id: C.13
  function "<=" (L, R : BIT_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.15
  function "<=" (L : NATURAL; R : BIT_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L is a non-negative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.17
  function "<=" (L : BIT_VECTOR; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L <= R" where L is an UNSIGNED vector and
  --         R is a non-negative INTEGER.

  --============================================================================
  -- Id: C.19
  function ">=" (L, R : BIT_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.21
  function ">=" (L : NATURAL; R : BIT_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L is a non-negative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.23
  function ">=" (L : BIT_VECTOR; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L >= R" where L is an UNSIGNED vector and
  --         R is a non-negative INTEGER.

  --============================================================================
  -- Id: C.25
  function "=" (L, R : BIT_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.27
  function "=" (L : NATURAL; R : BIT_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L is a non-negative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.29
  function "=" (L : BIT_VECTOR; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L = R" where L is an UNSIGNED vector and
  --         R is a non-negative INTEGER.

  --============================================================================

  -- Id: C.31
  function "/=" (L, R : BIT_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L and R are UNSIGNED vectors possibly
  --         of different lengths.

  -- Id: C.33
  function "/=" (L : NATURAL; R : BIT_VECTOR) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L is a non-negative INTEGER and
  --         R is an UNSIGNED vector.

  -- Id: C.35
  function "/=" (L : BIT_VECTOR; R : NATURAL) return BOOLEAN;
  -- Result subtype: BOOLEAN
  -- Result: Computes "L /= R" where L is an UNSIGNED vector and
  --         R is a non-negative INTEGER.

  --============================================================================
  -- Shift and Rotate Functions
  --============================================================================

  -- Id: S.1
  function SHIFT_LEFT (ARG : BIT_VECTOR; COUNT : NATURAL) return BIT_VECTOR;
  -- Result subtype: bit_vector(ARG'LENGTH-1 downto 0)
  -- Result: Performs a shift-left on an UNSIGNED vector COUNT times.
  --         The vacated positions are filled with '0'.
  --         The COUNT leftmost elements are lost.

  -- Id: S.2
  function SHIFT_RIGHT (ARG : BIT_VECTOR; COUNT : NATURAL) return BIT_VECTOR;
  -- Result subtype: UNSIGNED(ARG'LENGTH-1 downto 0)
  -- Result: Performs a shift-right on an UNSIGNED vector COUNT times.
  --         The vacated positions are filled with '0'.
  --         The COUNT rightmost elements are lost.
  --============================================================================

  -- Id: S.5
  function ROTATE_LEFT (ARG : BIT_VECTOR; COUNT : NATURAL) return BIT_VECTOR;
  -- Result subtype: bit_vector(ARG'LENGTH-1 downto 0)
  -- Result: Performs a rotate-left of an UNSIGNED vector COUNT times.

  -- Id: S.6
  function ROTATE_RIGHT (ARG : BIT_VECTOR; COUNT : NATURAL) return BIT_VECTOR;
  -- Result subtype: bit_vector(ARG'LENGTH-1 downto 0)
  -- Result: Performs a rotate-right of an UNSIGNED vector COUNT times.


  --============================================================================

  ------------------------------------------------------------------------------
  -- Note: Function S.9 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.9
  function "sll" (ARG : BIT_VECTOR; COUNT : INTEGER) return BIT_VECTOR;
  -- Result subtype: BIT_VECTOR(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.11 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.11
  function "srl" (ARG : BIT_VECTOR; COUNT : INTEGER) return BIT_VECTOR;
  -- Result subtype: BIT_VECTOR(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_RIGHT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.13 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.13
  function "rol" (ARG : BIT_VECTOR; COUNT : INTEGER) return BIT_VECTOR;
  -- Result subtype: BIT_VECTOR(ARG'LENGTH-1 downto 0)
  -- Result: ROTATE_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.15 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.15
  function "ror" (ARG : BIT_VECTOR; COUNT : INTEGER) return BIT_VECTOR;
  -- Result subtype: BIT_VECTOR(ARG'LENGTH-1 downto 0)
  -- Result: ROTATE_RIGHT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.17 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.17
  function "sla" (ARG : BIT_VECTOR; COUNT : INTEGER) return BIT_VECTOR;
  -- Result subtype: BIT_VECTOR(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_LEFT(ARG, COUNT)

  ------------------------------------------------------------------------------
  -- Note: Function S.19 is not compatible with IEEE Std 1076-1987. Comment
  -- out the function (declaration and body) for IEEE Std 1076-1987 compatibility.
  ------------------------------------------------------------------------------
  -- Id: S.19
  function "sra" (ARG : BIT_VECTOR; COUNT : INTEGER) return BIT_VECTOR;
  -- Result subtype: BIT_VECTOR(ARG'LENGTH-1 downto 0)
  -- Result: SHIFT_RIGHT(ARG, COUNT)


  --============================================================================
  --   RESIZE Functions
  --============================================================================

  -- Id: R.2
  function RESIZE (ARG : BIT_VECTOR; NEW_SIZE : NATURAL) return BIT_VECTOR;
  -- Result subtype: bit_vector(NEW_SIZE-1 downto 0)
  -- Result: Resizes the UNSIGNED vector ARG to the specified size.
  --         To create a larger vector, the new [leftmost] bit positions
  --         are filled with '0'. When truncating, the leftmost bits
  --         are dropped.

  function RESIZE (ARG, SIZE_RES : BIT_VECTOR) return BIT_VECTOR;
  -- Result subtype: BIT_VECTOR (SIZE_RES'length-1 downto 0)

  --============================================================================
  -- Conversion Functions
  --============================================================================

  -- Id: D.1
  function TO_INTEGER (ARG : BIT_VECTOR) return NATURAL;
  -- Result subtype: NATURAL. Value cannot be negative since parameter is an
  --             UNSIGNED vector.
  -- Result: Converts the UNSIGNED vector to an INTEGER.

  -- Id: D.3
  function To_BitVector (ARG, SIZE : NATURAL) return BIT_VECTOR;
  -- Result subtype: bit_vector(SIZE-1 downto 0)
  -- Result: Converts a non-negative INTEGER to an UNSIGNED vector with
  --         the specified size.

  function To_BitVector (ARG : NATURAL; SIZE_RES : BIT_VECTOR)
    return BIT_VECTOR;
  -- Result subtype: STD_LOGIC_VECTOR(SIZE_RES'length-1 downto 0)

  -- begin LCS-2006-130
  alias To_Bit_Vector is
    To_BitVector[NATURAL, NATURAL return BIT_VECTOR];
  alias To_BV is
    To_BitVector[NATURAL, NATURAL return BIT_VECTOR];

  alias To_Bit_Vector is
    To_BitVector[NATURAL, BIT_VECTOR return BIT_VECTOR];
  alias To_BV is
    To_BitVector[NATURAL, BIT_VECTOR return BIT_VECTOR];

end package NUMERIC_BIT_UNSIGNED;

use std.standard.all;
use std.textio.all;
package standard_textio_additions is

	procedure deallocate (p : inout line);

	procedure flush (file f : text);

	function justify (value : string; justified : side := right; field : width := 0) return string;

	procedure sread (l    : inout line; value : out string; strlen : out natural);
	alias string_read is sread [line, string, natural];
	alias bread is read [line, bit_vector, boolean];
	alias bread is read [line, bit_vector];
	alias binary_read is read [line, bit_vector, boolean];
	alias binary_read is read [line, bit_vector];
	procedure oread (l    : inout line; value : out bit_vector; good : out boolean);
	procedure oread (l    : inout line; value : out bit_vector);
	alias octal_read is oread [line, bit_vector, boolean];
	alias octal_read is oread [line, bit_vector];
	procedure hread (l    : inout line; value : out bit_vector; good : out boolean);
	procedure hread (l    : inout line; value : out bit_vector);
	alias hex_read is hread [line, bit_vector, boolean];
	alias hex_read is hread [line, bit_vector];
	procedure tee (file f :       text; l : inout line);
	procedure write (l      : inout line; value : in real;
	                 format : in    string);
	alias swrite is write [line, string, side, width];
	alias string_write is write [line, string, side, width];
	alias bwrite is write [line, bit_vector, side, width];
	alias binary_write is write [line, bit_vector, side, width];
	procedure owrite (l         : inout line; value : in bit_vector;
	                  justified : in    side := right; field : in width := 0);
	alias octal_write is owrite [line, bit_vector, side, width];
	procedure hwrite (l         : inout line; value : in bit_vector;
	                  justified : in    side := right; field : in width := 0);
	alias hex_write is hwrite [line, bit_vector, side, width];

end package standard_textio_additions;


-- $Date: 2008-04-10 17:16:09 +0930 (Thu, 10 Apr 2008) $

package body MATH_REAL is

    --
    -- Local Constants for Use in the Package Body Only
    --
    constant  MATH_E_P2 :  REAL := 7.38905_60989_30650;   -- e**2
    constant  MATH_E_P10 :  REAL := 22026.46579_48067_17; -- e**10
    constant  MATH_EIGHT_PI : REAL := 25.13274_12287_18345_90770_115; --8*pi
--    constant  MAX_ITER:  INTEGER := 27;  -- Maximum precision factor for cordic
    constant  MAX_COUNT: INTEGER := 150; -- Maximum count for number of tries
    constant  BASE_EPS: REAL := 0.00001;  -- Factor for convergence criteria
--    constant  KC : REAL := 6.0725293500888142e-01; -- Constant for cordic
    constant REAL_INTEGER_HIGH : REAL := 2147483647.0; -- Constant to replace REAL(INTEGER'HIGH)

    --
    -- Local Type Declarations for Cordic Operations
    --
    type REAL_VECTOR is array (NATURAL range <>) of REAL;
--    type NATURAL_VECTOR is array (NATURAL range <>) of NATURAL;
--    subtype REAL_VECTOR_N is REAL_VECTOR (0 to MAX_ITER);
--    subtype REAL_ARR_2 is REAL_VECTOR (0 to 1);
--    subtype REAL_ARR_3 is REAL_VECTOR (0 to 2);
--    subtype QUADRANT is INTEGER range 0 to 3;
--    type CORDIC_MODE_TYPE is (ROTATION, VECTORING);
--
--    --
--    -- Auxiliary Functions for Cordic Algorithms
--    --
--    function POWER_OF_2_SERIES (D : in NATURAL_VECTOR; INITIAL_VALUE : in REAL;
--                NUMBER_OF_VALUES : in NATURAL) return REAL_VECTOR is
--        -- Description:
--        --        Returns power of two for a vector of values
--       -- Notes:
--        --        None
--        --
--        variable V : REAL_VECTOR (0 to NUMBER_OF_VALUES);
--        variable TEMP : REAL := INITIAL_VALUE;
--        variable FLAG : BOOLEAN := TRUE;
--    begin
--              for I in 0 to NUMBER_OF_VALUES loop
--                 V(I) := TEMP;
--                 for P in D'RANGE loop
--                            if I = D(P) then
--                                FLAG := FALSE;
--                                exit;
--                            end if;
--                 end loop;
--                 if FLAG then
--                            TEMP := TEMP/2.0;
--                 end if;
--                 FLAG := TRUE;
--              end loop;
--              return V;
--    end function POWER_OF_2_SERIES;
--
--
--    constant TWO_AT_MINUS : REAL_VECTOR := POWER_OF_2_SERIES(
--                                               NATURAL_VECTOR'(100, 90),1.0,
--                                                                  MAX_ITER);
--
--    constant EPSILON : REAL_VECTOR_N := (
--                                        7.8539816339744827e-01,
--                                        4.6364760900080606e-01,
--                                        2.4497866312686413e-01,
--                                        1.2435499454676144e-01,
--                                        6.2418809995957351e-02,
--                                        3.1239833430268277e-02,
--                                        1.5623728620476830e-02,
--                                        7.8123410601011116e-03,
--                                        3.9062301319669717e-03,
--                                        1.9531225164788189e-03,
--                                        9.7656218955931937e-04,
--                                        4.8828121119489829e-04,
--                                        2.4414062014936175e-04,
--                                        1.2207031189367021e-04,
--                                        6.1035156174208768e-05,
--                                        3.0517578115526093e-05,
--                                        1.5258789061315760e-05,
--                                        7.6293945311019699e-06,
--                                        3.8146972656064960e-06,
--                                        1.9073486328101870e-06,
--                                        9.5367431640596080e-07,
--                                        4.7683715820308876e-07,
--                                        2.3841857910155801e-07,
--                                        1.1920928955078067e-07,
--                                        5.9604644775390553e-08,
--                                        2.9802322387695303e-08,
--                                        1.4901161193847654e-08,
--                                        7.4505805969238281e-09
--                                       );
--
--    function CORDIC ( X0 : in REAL;
--                      Y0 : in REAL;
--                      Z0 : in REAL;
--                      N : in NATURAL;                 --  Precision factor
--            CORDIC_MODE : in CORDIC_MODE_TYPE         --  Rotation (Z -> 0)
--                                                      --  or vectoring (Y -> 0)
--                    ) return REAL_ARR_3 is
--        -- Description:
--        --        Compute cordic values
--        -- Notes:
--        --         None
--             variable X : REAL := X0;
--             variable Y : REAL := Y0;
--             variable Z : REAL := Z0;
--             variable X_TEMP : REAL;
--    begin
--       if CORDIC_MODE = ROTATION then
--           for K in 0 to N loop
--                      X_TEMP := X;
--                      if ( Z >= 0.0) then
--                               X := X - Y * TWO_AT_MINUS(K);
--                               Y := Y + X_TEMP * TWO_AT_MINUS(K);
--                               Z := Z - EPSILON(K);
--                      else
--                               X := X + Y * TWO_AT_MINUS(K);
--                               Y := Y - X_TEMP * TWO_AT_MINUS(K);
--                               Z := Z + EPSILON(K);
--                      end if;
--            end loop;
--        else
--            for K in 0 to N loop
--                    X_TEMP := X;
--                    if ( Y < 0.0) then
--                               X := X - Y * TWO_AT_MINUS(K);
--                               Y := Y + X_TEMP * TWO_AT_MINUS(K);
--                               Z := Z - EPSILON(K);
--                    else
--                               X := X + Y * TWO_AT_MINUS(K);
--                               Y := Y - X_TEMP * TWO_AT_MINUS(K);
--                               Z := Z + EPSILON(K);
--                    end if;
--            end loop;
--        end if;
--        return REAL_ARR_3'(X, Y, Z);
--    end function CORDIC;

    --
    -- Bodies for Global Mathematical Functions Start Here
    --
    function SIGN (X: in REAL ) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --        None
    begin
           if  ( X > 0.0 )  then
                return 1.0;
           elsif ( X < 0.0 )  then
                return -1.0;
           else
                return 0.0;
           end if;
    end function SIGN;

    function CEIL (X : in REAL ) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --        a) No conversion to an INTEGER type is expected, so truncate
        --           cannot overflow for large arguments
        --        b) The domain supported by this function is X <= LARGE
        --        c) Returns X if ABS(X) >= LARGE

        constant LARGE: REAL  := REAL_INTEGER_HIGH;
        variable RD: REAL;

    begin
         if ABS(X) >= LARGE then
               return X;
         end if;

         RD := REAL ( INTEGER(X));
         if RD = X then
            return X;
         end if;

            if X > 0.0 then
                       if RD >= X then
                                  return RD;
                       else
                                  return RD + 1.0;
                       end if;
            elsif  X = 0.0  then
                return 0.0;
            else
                       if RD <= X then
                                  return RD + 1.0;
                       else
                                  return RD;
                       end if;
            end if;
    end function CEIL;

    function FLOOR (X : in REAL ) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --        a) No conversion to an INTEGER type is expected, so truncate
        --           cannot overflow for large arguments
        --        b) The domain supported by this function is ABS(X) <= LARGE
        --        c) Returns X if ABS(X) >= LARGE

        constant LARGE: REAL  := REAL_INTEGER_HIGH;
        variable RD: REAL;

    begin
        if ABS( X ) >= LARGE then
                    return X;
        end if;

        RD := REAL ( INTEGER(X));
        if RD = X then
                return X;
        end if;

        if X > 0.0 then
                      if RD <= X then
                                  return RD;
                       else
                                  return RD - 1.0;
                       end if;
        elsif  X = 0.0  then
                return 0.0;
        else
                   if RD >= X then
                                  return RD - 1.0;
                   else
                                  return RD;
                   end if;
        end if;
    end function FLOOR;

    function ROUND (X : in REAL ) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --         a) Returns 0.0 if X = 0.0
        --         b) Returns FLOOR(X + 0.5) if X > 0
        --         c) Returns CEIL(X - 0.5) if X < 0

    begin
           if  X > 0.0  then
                return FLOOR(X + 0.5);
           elsif  X < 0.0  then
                return CEIL( X - 0.5);
           else
                return 0.0;
           end if;
    end function ROUND;

    function TRUNC (X : in REAL ) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --         a) Returns 0.0 if X = 0.0
        --         b) Returns FLOOR(X) if X > 0
        --         c) Returns CEIL(X) if X < 0

    begin
           if  X > 0.0  then
                return FLOOR(X);
           elsif  X < 0.0  then
                return CEIL( X);
           else
                return 0.0;
           end if;
    end function TRUNC;




    function "MOD" (X, Y: in REAL ) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --        a) Returns 0.0 on error

        constant XNEGATIVE : BOOLEAN := X < 0.0;
        constant YNEGATIVE : BOOLEAN := Y < 0.0;
        variable VALUE : REAL;
    begin
        -- Check validity of input arguments
            if (Y = 0.0) then
                 assert FALSE
                        report "MOD(X, 0.0) is undefined"
                        severity ERROR;
                 return 0.0;
              end if;

        -- Compute value
        if ( XNEGATIVE ) then
                if ( YNEGATIVE ) then
                        VALUE := X + (FLOOR(ABS(X)/ABS(Y)))*ABS(Y);
                else
                        VALUE := X + (CEIL(ABS(X)/ABS(Y)))*ABS(Y);
                end if;
        else
                if ( YNEGATIVE ) then
                        VALUE := X - (CEIL(ABS(X)/ABS(Y)))*ABS(Y);
                else
                        VALUE := X - (FLOOR(ABS(X)/ABS(Y)))*ABS(Y);
                end if;
        end if;

        return VALUE;
    end function "MOD";


    function REALMAX (X, Y : in REAL ) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --        a) REALMAX(X,Y) = X when X = Y
        --
    begin
        if X >= Y then
           return X;
        else
           return Y;
        end if;
    end function REALMAX;

    function REALMIN (X, Y : in REAL ) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --        a) REALMIN(X,Y) = X when X = Y
        --
    begin
        if X <= Y then
           return X;
        else
           return Y;
        end if;
    end function REALMIN;


--    procedure UNIFORM(variable SEED1,SEED2:inout POSITIVE;variable X:out REAL)
--                                                                         is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --        a) Returns 0.0 on error
--        --
--        variable Z, K: INTEGER;
--        variable TSEED1 : INTEGER := INTEGER'(SEED1);
--        variable TSEED2 : INTEGER := INTEGER'(SEED2);
--    begin
--        -- Check validity of arguments
--        if SEED1 > 2147483562 then
--                assert FALSE
--                        report "SEED1 > 2147483562 in UNIFORM"
--                        severity ERROR;
--                X := 0.0;
--                return;
--        end if;
--
--        if SEED2 > 2147483398 then
--                assert FALSE
--                        report "SEED2 > 2147483398 in UNIFORM"
--                        severity ERROR;
--                X := 0.0;
--                return;
--        end if;
--
--        -- Compute new seed values and pseudo-random number
--        K := TSEED1/53668;
--        TSEED1 := 40014 * (TSEED1 - K * 53668) - K * 12211;
--
--        if TSEED1 < 0  then
--                TSEED1 := TSEED1 + 2147483563;
--        end if;
--
--        K := TSEED2/52774;
--        TSEED2 := 40692 * (TSEED2 - K * 52774) - K * 3791;
--
--        if TSEED2 < 0  then
--                TSEED2 := TSEED2 + 2147483399;
--        end if;
--
--        Z := TSEED1 - TSEED2;
--        if Z < 1 then
--                Z := Z + 2147483562;
--        end if;
--
--        -- Get output values
--        SEED1 := POSITIVE'(TSEED1);
--        SEED2 := POSITIVE'(TSEED2);
--        X :=  REAL(Z)*4.656613e-10;
--    end procedure UNIFORM;



    function SQRT (X : in REAL ) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --        a) Uses the Newton-Raphson approximation:
        --            F(n+1) = 0.5*[F(n) + x/F(n)]
        --        b) Returns 0.0 on error
        --

        constant EPS : REAL := BASE_EPS*BASE_EPS; -- Convergence factor

        variable INIVAL: REAL;
        variable OLDVAL : REAL ;
        variable NEWVAL : REAL ;
        variable COUNT : INTEGER := 1;

    begin
        -- Check validity of argument
        if ( X < 0.0 ) then
                assert FALSE
                        report "X < 0.0 in SQRT(X)"
                        severity ERROR;
                return 0.0;
        end if;

        -- Get the square root for special cases
        if X = 0.0 then
                  return 0.0;
        else
                if ( X = 1.0 ) then
                        return 1.0;
                end if;
        end if;

        -- Get the square root for general cases
        INIVAL := EXP(LOG(X)*(0.5)); -- Mathematically correct but imprecise
        OLDVAL := INIVAL;
        NEWVAL := (X/OLDVAL + OLDVAL)*0.5;

        -- Check for  relative and absolute error and max count
        while  ( ( (ABS((NEWVAL -OLDVAL)/NEWVAL) > EPS) OR
                   (ABS(NEWVAL - OLDVAL) > EPS) ) AND
                   (COUNT < MAX_COUNT) )  loop
                OLDVAL := NEWVAL;
                NEWVAL := (X/OLDVAL + OLDVAL)*0.5;
                COUNT := COUNT + 1;
        end loop;
        return NEWVAL;
    end function SQRT;

--    function CBRT (X : in REAL ) return REAL is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --        a) Uses the Newton-Raphson approximation:
--        --            F(n+1) = (1/3)*[2*F(n) + x/F(n)**2];
--        --
--        constant EPS : REAL := BASE_EPS*BASE_EPS;
--
--        variable INIVAL: REAL;
--        variable XLOCAL : REAL := X;
--        constant NEGATIVE : BOOLEAN := X < 0.0;
--        variable OLDVAL : REAL ;
--        variable NEWVAL : REAL ;
--        variable COUNT : INTEGER := 1;
--
--    begin
--
--        -- Compute root for special cases
--        if X = 0.0 then
--                return 0.0;
--        elsif ( X = 1.0 ) then
--                return 1.0;
--        else
--                if X = -1.0 then
--                        return -1.0;
--                end if;
--        end if;
--
--        -- Compute root for general cases
--        if NEGATIVE then
--                XLOCAL := -X;
--        end if;
--
--        INIVAL := EXP(LOG(XLOCAL)/(3.0)); -- Mathematically correct but
--                                          -- imprecise
--        OLDVAL := INIVAL;
--        NEWVAL := (XLOCAL/(OLDVAL*OLDVAL) + 2.0*OLDVAL)/3.0;
--
--        -- Check for relative and absolute errors and max count
--        while ( (  (ABS((NEWVAL -OLDVAL)/NEWVAL) > EPS ) OR
--                   (ABS(NEWVAL - OLDVAL) > EPS ) )  AND
--                   ( COUNT < MAX_COUNT ) ) loop
--                OLDVAL := NEWVAL;
--                NEWVAL :=(XLOCAL/(OLDVAL*OLDVAL) + 2.0*OLDVAL)/3.0;
--                COUNT := COUNT + 1;
--        end loop;
--
--        if NEGATIVE then
--                NEWVAL := -NEWVAL;
--        end if;
--
--        return NEWVAL;
--    end function CBRT;

    function "**" (X : in INTEGER; Y : in REAL) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --        a) Returns 0.0 on error condition

    begin
        -- Check validity of argument
        if ( ( X < 0  ) and ( Y /= 0.0 ) ) then
                assert FALSE
                        report "X < 0 and Y /= 0.0 in X**Y"
                        severity ERROR;
                return 0.0;
        end if;

        if ( ( X = 0  ) and ( Y <= 0.0 ) ) then
                assert FALSE
                        report "X = 0 and Y <= 0.0 in X**Y"
                        severity ERROR;
                return 0.0;
        end if;

        -- Get value for special cases
        if ( X = 0  and  Y > 0.0 ) then
                return 0.0;
        end if;

        if ( X = 1 ) then
                return 1.0;
        end if;

        if ( Y = 0.0 and X /= 0 ) then
                return 1.0;
        end if;

        if ( Y = 1.0) then
                return (REAL(X));
        end if;

        -- Get value for general case
        return EXP (Y * LOG (REAL(X)));
    end function "**";

    function "**" (X : in REAL; Y : in REAL) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --        a) Returns 0.0 on error condition

    begin
        -- Check validity of argument
        if ( ( X < 0.0  ) and ( Y /= 0.0 ) ) then
                assert FALSE
                        report "X < 0.0 and Y /= 0.0 in X**Y"
                        severity ERROR;
                return 0.0;
        end if;

        if ( ( X = 0.0  ) and ( Y <= 0.0 ) ) then
                assert FALSE
                        report "X = 0.0 and Y <= 0.0 in X**Y"
                        severity ERROR;
                return 0.0;
        end if;

        -- Get value for special cases
        if ( X = 0.0  and  Y > 0.0 ) then
                return 0.0;
        end if;

        if ( X = 1.0 ) then
                return 1.0;
        end if;

        if ( Y = 0.0 and X /= 0.0 ) then
                return 1.0;
        end if;

        if ( Y = 1.0) then
                return (X);
        end if;

        -- Get value for general case
        return EXP (Y * LOG (X));
    end function "**";

    function EXP  (X : in REAL ) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --        a) This function computes the exponential using the following
        --           series:
        --                exp(x) = 1 + x + x**2/2! + x**3/3! + ... ; |x| < 1.0
        --           and reduces argument X to take advantage of exp(x+y) =
        --           exp(x)*exp(y)
        --
        --        b) This implementation limits X to be less than LOG(REAL'HIGH)
        --           to avoid overflow.  Returns REAL'HIGH when X reaches that
        --           limit
        --
        constant EPS : REAL := BASE_EPS*BASE_EPS*BASE_EPS;-- Precision criteria
        constant MAX_POW : REAL := 30.0; -- FIXME after X=30.0, results aren't accurate
        constant MAX_VALUE : REAL := 10686474581524.46; -- exp(MAX_POW)

            constant RECIPROCAL: BOOLEAN := X < 0.0;-- Check sign of argument
            variable XLOCAL : REAL := ABS(X);       -- Use positive value
            variable OLDVAL: REAL ;
            variable COUNT: INTEGER ;
            variable NEWVAL: REAL ;
            variable LAST_TERM: REAL ;
        variable FACTOR : REAL := 1.0;

     begin
            -- Compute value for special cases
        if X = 0.0 then
                return 1.0;
        end if;

        if  XLOCAL = 1.0  then
                if RECIPROCAL then
                        return MATH_1_OVER_E;
                else
                        return MATH_E;
                end if;
        end if;

        if  XLOCAL = 2.0  then
                if RECIPROCAL then
                        return 1.0/MATH_E_P2;
                else
                        return MATH_E_P2;
                end if;
        end if;

        if  XLOCAL = 10.0  then
                if RECIPROCAL then
                        return 1.0/MATH_E_P10;
                else
                        return MATH_E_P10;
                end if;
        end if;

        if XLOCAL > MAX_POW then
                if RECIPROCAL then
                        return 0.0;
                else
                        assert FALSE
                                report "X > LOG(REAL'HIGH) in EXP(X)"
                                severity NOTE;
                        return MAX_VALUE;
                end if;
        end if;

        -- Reduce argument to ABS(X) < 1.0
        while XLOCAL > 10.0 loop
                XLOCAL := XLOCAL - 10.0;
                FACTOR := FACTOR*MATH_E_P10;
        end loop;

        while XLOCAL > 1.0 loop
                XLOCAL := XLOCAL - 1.0;
                FACTOR := FACTOR*MATH_E;
        end loop;

        -- Compute value for case 0 < XLOCAL < 1
        OLDVAL := 1.0;
        LAST_TERM := XLOCAL;
        NEWVAL:= OLDVAL + LAST_TERM;
        COUNT := 2;

        -- Check for relative and absolute errors and max count
        while ( ( (ABS((NEWVAL - OLDVAL)/NEWVAL) > EPS) OR
                  (ABS(NEWVAL - OLDVAL) > EPS) ) AND
                  (COUNT < MAX_COUNT ) ) loop
                OLDVAL := NEWVAL;
                LAST_TERM := LAST_TERM*(XLOCAL / (REAL(COUNT)));
                NEWVAL := OLDVAL + LAST_TERM;
                COUNT := COUNT + 1;
        end loop;

        -- Compute final value using exp(x+y) = exp(x)*exp(y)
        NEWVAL := NEWVAL*FACTOR;

        if RECIPROCAL then
                NEWVAL := 1.0/NEWVAL;
        end if;

        return NEWVAL;
     end function EXP;


    --
    -- Auxiliary Functions to Compute LOG
    --
    function ILOGB(X: in REAL) return INTEGER IS
        -- Description:
        --        Returns n such that -1 <= ABS(X)/2^n < 2
        -- Notes:
        --        None

        variable N: INTEGER := 0;
        variable Y: REAL := ABS(X);

    begin
        if(Y = 1.0 or Y = 0.0) then
                return 0;
        end if;

        if( Y > 1.0) then
                while Y >= 2.0 loop
                        Y := Y/2.0;
                        N := N+1;
                end loop;
                return N;
        end if;

        -- O < Y < 1
        while Y < 1.0 loop
                Y := Y*2.0;
                N := N -1;
        end loop;
        return N;
    end function ILOGB;

    function LDEXP(X: in REAL; N: in INTEGER) RETURN REAL IS
        -- Description:
        --        Returns X*2^n
        -- Notes:
        --         None
    begin
        return X*(2.0 ** N);
    end function LDEXP;

    function LOG (X : in REAL ) return REAL IS
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        --
        -- Notes:
        --        a) Returns REAL'LOW on error
        --
        -- Copyright (c) 1992 Regents of the University of California.
        -- All rights reserved.
        --
        -- Redistribution and use in source and binary forms, with or without
        -- modification, are permitted provided that the following conditions
        -- are met:
        -- 1. Redistributions of source code must retain the above copyright
        -- notice, this list of conditions and the following disclaimer.
        -- 2. Redistributions in binary form must reproduce the above copyright
        -- notice, this list of conditions and the following disclaimer in the
        -- documentation and/or other materials provided with the distribution.
        -- 3. Neither the name of the University nor the names of its
        -- contributors may be used to endorse or promote products derived
        -- from this software without specific prior written permission.
        --
        -- THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS''
        -- AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
        -- THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
        -- PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
        -- CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
        -- EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
        -- PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
        -- PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
        -- OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
        -- (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
        -- USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
        -- DAMAGE.
        --
        -- NOTE: This VHDL version was generated using the C version of the
        --         original function by the IEEE VHDL Mathematical Package
        --         Working Group (CS/JT)

        constant N: INTEGER := 128;

        -- Table of log(Fj) = logF_head[j] + logF_tail[j], for Fj = 1+j/128.
        -- Used for generation of extend precision logarithms.
        -- The constant 35184372088832 is 2^45, so the divide is exact.
        -- It ensures correct reading of logF_head, even for inaccurate
        -- decimal-to-binary conversion routines. (Everybody gets the
        -- right answer for INTEGERs less than 2^53.)
        -- Values for LOG(F) were generated using error < 10^-57 absolute
        -- with the bc -l package.

        type REAL_VECTOR is array (NATURAL range <>) of REAL;

        constant A1:REAL := 0.08333333333333178827;
        constant A2:REAL := 0.01250000000377174923;
        constant A3:REAL := 0.002232139987919447809;
        constant A4:REAL := 0.0004348877777076145742;

        constant LOGF_HEAD: REAL_VECTOR(0 TO N) := (
                0.0,
                0.007782140442060381246,
                0.015504186535963526694,
                0.023167059281547608406,
                0.030771658666765233647,
                0.038318864302141264488,
                0.045809536031242714670,
                0.053244514518837604555,
                0.060624621816486978786,
                0.067950661908525944454,
                0.075223421237524235039,
                0.082443669210988446138,
                0.089612158689760690322,
                0.096729626458454731618,
                0.103796793681567578460,
                0.110814366340264314203,
                0.117783035656430001836,
                0.124703478501032805070,
                0.131576357788617315236,
                0.138402322859292326029,
                0.145182009844575077295,
                0.151916042025732167530,
                0.158605030176659056451,
                0.165249572895390883786,
                0.171850256926518341060,
                0.178407657472689606947,
                0.184922338493834104156,
                0.191394852999565046047,
                0.197825743329758552135,
                0.204215541428766300668,
                0.210564769107350002741,
                0.216873938300523150246,
                0.223143551314024080056,
                0.229374101064877322642,
                0.235566071312860003672,
                0.241719936886966024758,
                0.247836163904594286577,
                0.253915209980732470285,
                0.259957524436686071567,
                0.265963548496984003577,
                0.271933715484010463114,
                0.277868451003087102435,
                0.283768173130738432519,
                0.289633292582948342896,
                0.295464212893421063199,
                0.301261330578199704177,
                0.307025035294827830512,
                0.312755710004239517729,
                0.318453731118097493890,
                0.324119468654316733591,
                0.329753286372579168528,
                0.335355541920762334484,
                0.340926586970454081892,
                0.346466767346100823488,
                0.351976423156884266063,
                0.357455888922231679316,
                0.362905493689140712376,
                0.368325561158599157352,
                0.373716409793814818840,
                0.379078352934811846353,
                0.384411698910298582632,
                0.389716751140440464951,
                0.394993808240542421117,
                0.400243164127459749579,
                0.405465108107819105498,
                0.410659924985338875558,
                0.415827895143593195825,
                0.420969294644237379543,
                0.426084395310681429691,
                0.431173464818130014464,
                0.436236766774527495726,
                0.441274560805140936281,
                0.446287102628048160113,
                0.451274644139630254358,
                0.456237433481874177232,
                0.461175715122408291790,
                0.466089729924533457960,
                0.470979715219073113985,
                0.475845904869856894947,
                0.480688529345570714212,
                0.485507815781602403149,
                0.490303988045525329653,
                0.495077266798034543171,
                0.499827869556611403822,
                0.504556010751912253908,
                0.509261901790523552335,
                0.513945751101346104405,
                0.518607764208354637958,
                0.523248143765158602036,
                0.527867089620485785417,
                0.532464798869114019908,
                0.537041465897345915436,
                0.541597282432121573947,
                0.546132437597407260909,
                0.550647117952394182793,
                0.555141507540611200965,
                0.559615787935399566777,
                0.564070138285387656651,
                0.568504735352689749561,
                0.572919753562018740922,
                0.577315365035246941260,
                0.581691739635061821900,
                0.586049045003164792433,
                0.590387446602107957005,
                0.594707107746216934174,
                0.599008189645246602594,
                0.603290851438941899687,
                0.607555250224322662688,
                0.611801541106615331955,
                0.616029877215623855590,
                0.620240409751204424537,
                0.624433288012369303032,
                0.628608659422752680256,
                0.632766669570628437213,
                0.636907462236194987781,
                0.641031179420679109171,
                0.645137961373620782978,
                0.649227946625615004450,
                0.653301272011958644725,
                0.657358072709030238911,
                0.661398482245203922502,
                0.665422632544505177065,
                0.669430653942981734871,
                0.673422675212350441142,
                0.677398823590920073911,
                0.681359224807238206267,
                0.685304003098281100392,
                0.689233281238557538017,
                0.693147180560117703862);

        constant LOGF_TAIL: REAL_VECTOR(0 TO N) := (
                0.0,
                -0.00000000000000543229938420049,
                0.00000000000000172745674997061,
                -0.00000000000001323017818229233,
                -0.00000000000001154527628289872,
                -0.00000000000000466529469958300,
                0.00000000000005148849572685810,
                -0.00000000000002532168943117445,
                -0.00000000000005213620639136504,
                -0.00000000000001819506003016881,
                0.00000000000006329065958724544,
                0.00000000000008614512936087814,
                -0.00000000000007355770219435028,
                0.00000000000009638067658552277,
                0.00000000000007598636597194141,
                0.00000000000002579999128306990,
                -0.00000000000004654729747598444,
                -0.00000000000007556920687451336,
                0.00000000000010195735223708472,
                -0.00000000000017319034406422306,
                -0.00000000000007718001336828098,
                0.00000000000010980754099855238,
                -0.00000000000002047235780046195,
                -0.00000000000008372091099235912,
                0.00000000000014088127937111135,
                0.00000000000012869017157588257,
                0.00000000000017788850778198106,
                0.00000000000006440856150696891,
                0.00000000000016132822667240822,
                -0.00000000000007540916511956188,
                -0.00000000000000036507188831790,
                0.00000000000009120937249914984,
                0.00000000000018567570959796010,
                -0.00000000000003149265065191483,
                -0.00000000000009309459495196889,
                0.00000000000017914338601329117,
                -0.00000000000001302979717330866,
                0.00000000000023097385217586939,
                0.00000000000023999540484211737,
                0.00000000000015393776174455408,
                -0.00000000000036870428315837678,
                0.00000000000036920375082080089,
                -0.00000000000009383417223663699,
                0.00000000000009433398189512690,
                0.00000000000041481318704258568,
                -0.00000000000003792316480209314,
                0.00000000000008403156304792424,
                -0.00000000000034262934348285429,
                0.00000000000043712191957429145,
                -0.00000000000010475750058776541,
                -0.00000000000011118671389559323,
                0.00000000000037549577257259853,
                0.00000000000013912841212197565,
                0.00000000000010775743037572640,
                0.00000000000029391859187648000,
                -0.00000000000042790509060060774,
                0.00000000000022774076114039555,
                0.00000000000010849569622967912,
                -0.00000000000023073801945705758,
                0.00000000000015761203773969435,
                0.00000000000003345710269544082,
                -0.00000000000041525158063436123,
                0.00000000000032655698896907146,
                -0.00000000000044704265010452446,
                0.00000000000034527647952039772,
                -0.00000000000007048962392109746,
                0.00000000000011776978751369214,
                -0.00000000000010774341461609578,
                0.00000000000021863343293215910,
                0.00000000000024132639491333131,
                0.00000000000039057462209830700,
                -0.00000000000026570679203560751,
                0.00000000000037135141919592021,
                -0.00000000000017166921336082431,
                -0.00000000000028658285157914353,
                -0.00000000000023812542263446809,
                0.00000000000006576659768580062,
                -0.00000000000028210143846181267,
                0.00000000000010701931762114254,
                0.00000000000018119346366441110,
                0.00000000000009840465278232627,
                -0.00000000000033149150282752542,
                -0.00000000000018302857356041668,
                -0.00000000000016207400156744949,
                0.00000000000048303314949553201,
                -0.00000000000071560553172382115,
                0.00000000000088821239518571855,
                -0.00000000000030900580513238244,
                -0.00000000000061076551972851496,
                0.00000000000035659969663347830,
                0.00000000000035782396591276383,
                -0.00000000000046226087001544578,
                0.00000000000062279762917225156,
                0.00000000000072838947272065741,
                0.00000000000026809646615211673,
                -0.00000000000010960825046059278,
                0.00000000000002311949383800537,
                -0.00000000000058469058005299247,
                -0.00000000000002103748251144494,
                -0.00000000000023323182945587408,
                -0.00000000000042333694288141916,
                -0.00000000000043933937969737844,
                0.00000000000041341647073835565,
                0.00000000000006841763641591466,
                0.00000000000047585534004430641,
                0.00000000000083679678674757695,
                -0.00000000000085763734646658640,
                0.00000000000021913281229340092,
                -0.00000000000062242842536431148,
                -0.00000000000010983594325438430,
                0.00000000000065310431377633651,
                -0.00000000000047580199021710769,
                -0.00000000000037854251265457040,
                0.00000000000040939233218678664,
                0.00000000000087424383914858291,
                0.00000000000025218188456842882,
                -0.00000000000003608131360422557,
                -0.00000000000050518555924280902,
                0.00000000000078699403323355317,
                -0.00000000000067020876961949060,
                0.00000000000016108575753932458,
                0.00000000000058527188436251509,
                -0.00000000000035246757297904791,
                -0.00000000000018372084495629058,
                0.00000000000088606689813494916,
                0.00000000000066486268071468700,
                0.00000000000063831615170646519,
                0.00000000000025144230728376072,
                -0.00000000000017239444525614834);

        variable M, J:INTEGER;
        variable F1, F2, G, Q, U, U2, V: REAL;

        -- double logb(), ldexp();

        variable U1:REAL;

     begin

        -- Check validity of argument
        if ( X <= 0.0 ) then
                assert FALSE
                        report "X <= 0.0 in LOG(X)"
                        severity ERROR;
                return 0.0;
        end if;

        -- Compute value for special cases
        if ( X = 1.0 ) then
                return 0.0;
        end if;

        if ( X = MATH_E ) then
                return 1.0;
        end if;

        -- Argument reduction: 1 <= g < 2; x/2^m = g;
        -- y = F*(1 + f/F) for |f| <= 2^-8

        M := ILOGB(X);
        G := LDEXP(X, -M);
        J := INTEGER(REAL(N)*(G-1.0)); -- C code adds 0.5 for rounding
        F1 := (1.0/REAL(N)) * REAL(J) + 1.0; --F1*128 is an INTEGER in [128,512]
        F2 := G - F1;

        -- Approximate expansion for log(1+f2/F1) ~= u + q
        G := 1.0/(2.0*F1+F2);
        U := 2.0*F2*G;
        V := U*U;
        Q := U*V*(A1 + V*(A2 + V*(A3 + V*A4)));

        -- Case 1: u1 = u rounded to 2^-43 absolute. Since u < 2^-8,
        --       u1 has at most 35 bits, and F1*u1 is exact, as F1 has < 8 bits.
        --       It also adds exactly to |m*log2_hi + log_F_head[j] | < 750.
        --
        if ( J /= 0 or M /= 0) then
                U1 := U + 513.0;
                U1 := U1 - 513.0;

                -- Case 2: |1-x| < 1/256. The m- and j- dependent terms are zero
                --        u1 = u to 24 bits.
                --
        else
                U1 := U;
                --TRUNC(U1); --In c this is u1 = (double) (float) (u1)
        end if;

        U2 := (2.0*(F2 - F1*U1) - U1*F2) * G;
        -- u1 + u2 = 2f/(2F+f) to extra precision.

        -- log(x) = log(2^m*F1*(1+f2/F1)) =
        -- (m*log2_hi+LOGF_HEAD(j)+u1) + (m*log2_lo+LOGF_TAIL(j)+q);
        -- (exact) + (tiny)

        U1 := U1 + REAL(M)*LOGF_HEAD(N) + LOGF_HEAD(J);        -- Exact
        U2 := (U2 + LOGF_TAIL(J)) + Q;        -- Tiny
        U2 := U2 + LOGF_TAIL(N)*REAL(M);
        return (U1 + U2);
    end function LOG;


    function LOG2 (X: in REAL) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --        a) Returns REAL'LOW on error
    begin
        -- Check validity of arguments
        if ( X <= 0.0 )  then
                assert FALSE
                        report "X <= 0.0 in LOG2(X)"
                        severity ERROR;
                return 0.0;
        end if;

        -- Compute value for special cases
        if ( X = 1.0 ) then
                return 0.0;
        end if;

        if ( X = 2.0 ) then
                return 1.0;
        end if;

        -- Compute value for general case
        return ( MATH_LOG2_OF_E*LOG(X) );
    end function LOG2;


    function LOG10 (X: in REAL) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --        a) Returns REAL'LOW on error
    begin
        -- Check validity of arguments
        if ( X <= 0.0 )  then
                   assert FALSE
                        report "X <= 0.0 in LOG10(X)"
                        severity ERROR;
                   return 0.0;
        end if;

        -- Compute value for special cases
        if ( X = 1.0 ) then
                return 0.0;
        end if;

        if ( X = 10.0 ) then
                return 1.0;
        end if;

        -- Compute value for general case
        return ( MATH_LOG10_OF_E*LOG(X) );
    end function LOG10;


    function LOG (X: in REAL; BASE: in REAL) return REAL is
        -- Description:
        --        See function declaration in IEEE Std 1076.2-1996
        -- Notes:
        --        a) Returns REAL'LOW on error
    begin
        -- Check validity of arguments
        if ( X <= 0.0 )  then
                 assert FALSE
                        report "X <= 0.0 in LOG(X, BASE)"
                        severity ERROR;
                 return 0.0;
        end if;

        if ( BASE <= 0.0 or BASE = 1.0 )  then
                 assert FALSE
                        report "BASE <= 0.0 or BASE = 1.0 in LOG(X, BASE)"
                        severity ERROR;
                 return 0.0;
        end if;

        -- Compute value for special cases
        if ( X = 1.0 ) then
                return 0.0;
        end if;

        if ( X = BASE ) then
                return 1.0;
        end if;

        -- Compute value for general case
        return ( LOG(X)/LOG(BASE));
    end function LOG;


--    function  SIN (X : in REAL ) return REAL is
--        -- Description:
--        --         See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --         a) SIN(-X) = -SIN(X)
--        --         b) SIN(X) = X if ABS(X) < EPS
--        --         c) SIN(X) = X - X**3/3! if EPS < ABS(X) < BASE_EPS
--        --         d) SIN(MATH_PI_OVER_2 - X) = COS(X)
--        --         e) COS(X) = 1.0 - 0.5*X**2 if ABS(X) < EPS
--        --         f) COS(X) = 1.0 - 0.5*X**2 + (X**4)/4! if
--        --                                         EPS< ABS(X) <BASE_EPS
--
--        constant EPS : REAL := BASE_EPS*BASE_EPS; -- Convergence criteria
--
--        variable N : INTEGER;
--        constant NEGATIVE : BOOLEAN := X < 0.0;
--        variable XLOCAL : REAL := ABS(X) ;
--        variable VALUE: REAL;
--        variable TEMP : REAL;
--
--    begin
--        -- Make XLOCAL < MATH_2_PI
--        if XLOCAL > MATH_2_PI then
--                TEMP := FLOOR(XLOCAL/MATH_2_PI);
--                XLOCAL := XLOCAL - TEMP*MATH_2_PI;
--        end if;
--
--        if XLOCAL < 0.0 then
--                -- adjust for rounding error
--                XLOCAL := 0.0;
--        end if;
--
--        -- Compute value for special cases
--        if XLOCAL = 0.0  or XLOCAL = MATH_2_PI or XLOCAL = MATH_PI  then
--                return 0.0;
--        end if;
--
--        if  XLOCAL = MATH_PI_OVER_2 then
--                if NEGATIVE then
--                        return -1.0;
--                else
--                        return 1.0;
--                end if;
--        end if;
--
--        if  XLOCAL = MATH_3_PI_OVER_2 then
--                if NEGATIVE then
--                        return 1.0;
--                else
--                        return -1.0;
--                end if;
--        end if;
--
--        if XLOCAL < EPS then
--                if NEGATIVE then
--                        return -XLOCAL;
--                else
--                        return XLOCAL;
--                end if;
--        else
--                if XLOCAL < BASE_EPS then
--                        TEMP := XLOCAL - (XLOCAL*XLOCAL*XLOCAL)/6.0;
--                        if NEGATIVE then
--                                return -TEMP;
--                        else
--                                return TEMP;
--                        end if;
--                end if;
--        end if;
--
--        TEMP := MATH_PI - XLOCAL;
--        if ABS(TEMP) < EPS then
--                if NEGATIVE then
--                        return -TEMP;
--                else
--                        return TEMP;
--                end if;
--        else
--                if ABS(TEMP) < BASE_EPS then
--                        TEMP := TEMP - (TEMP*TEMP*TEMP)/6.0;
--                        if NEGATIVE then
--                                return -TEMP;
--                        else
--                                return TEMP;
--                        end if;
--                end if;
--        end if;
--
--        TEMP := MATH_2_PI - XLOCAL;
--        if ABS(TEMP) < EPS then
--                if NEGATIVE then
--                        return TEMP;
--                else
--                        return -TEMP;
--                end if;
--        else
--                if ABS(TEMP) < BASE_EPS then
--                        TEMP := TEMP - (TEMP*TEMP*TEMP)/6.0;
--                        if NEGATIVE then
--                                return TEMP;
--                        else
--                                return -TEMP;
--                        end if;
--                end if;
--        end if;
--
--        TEMP := ABS(MATH_PI_OVER_2 - XLOCAL);
--        if TEMP < EPS then
--                TEMP := 1.0 - TEMP*TEMP*0.5;
--                if NEGATIVE then
--                        return -TEMP;
--                else
--                        return TEMP;
--                end if;
--        else
--                if TEMP < BASE_EPS then
--                        TEMP := 1.0 -TEMP*TEMP*0.5 + TEMP*TEMP*TEMP*TEMP/24.0;
--                        if NEGATIVE then
--                                return -TEMP;
--                        else
--                                return TEMP;
--                        end if;
--                end if;
--        end if;
--
--        TEMP := ABS(MATH_3_PI_OVER_2 - XLOCAL);
--        if TEMP < EPS then
--                TEMP := 1.0 - TEMP*TEMP*0.5;
--                if NEGATIVE then
--                        return TEMP;
--                else
--                        return -TEMP;
--                end if;
--        else
--                if TEMP < BASE_EPS then
--                        TEMP := 1.0 -TEMP*TEMP*0.5 + TEMP*TEMP*TEMP*TEMP/24.0;
--                        if NEGATIVE then
--                                return TEMP;
--                        else
--                                return -TEMP;
--                        end if;
--                end if;
--        end if;
--
--        -- Compute value for general cases
--        if ((XLOCAL < MATH_PI_OVER_2 ) and (XLOCAL > 0.0)) then
--                 VALUE:=  CORDIC( KC, 0.0, X, 27, ROTATION)(1);
--        end if;
--
--        N := INTEGER ( FLOOR(XLOCAL/MATH_PI_OVER_2));
--        case QUADRANT( N mod 4) is
--           when 0 =>
--                VALUE := CORDIC( KC, 0.0, XLOCAL, 27, ROTATION)(1);
--           when 1 =>
--                VALUE := CORDIC( KC, 0.0, XLOCAL - MATH_PI_OVER_2, 27,
--                                                                ROTATION)(0);
--           when 2 =>
--                VALUE := -CORDIC( KC, 0.0, XLOCAL - MATH_PI, 27, ROTATION)(1);
--           when 3 =>
--                VALUE := -CORDIC( KC, 0.0, XLOCAL - MATH_3_PI_OVER_2, 27,
--                                                                ROTATION)(0);
--        end case;
--
--        if NEGATIVE then
--                return -VALUE;
--        else
--                return VALUE;
--        end if;
--    end function SIN;
--
--
--   function COS (X : in REAL) return REAL is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --        a) COS(-X) = COS(X)
--        --        b) COS(X) = SIN(MATH_PI_OVER_2 - X)
--        --        c) COS(MATH_PI + X)  = -COS(X)
--        --        d) COS(X) = 1.0 - X*X/2.0 if ABS(X) < EPS
--        --        e) COS(X) = 1.0 - 0.5*X**2 + (X**4)/4! if
--        --                                           EPS< ABS(X) <BASE_EPS
--        --
--        constant EPS : REAL := BASE_EPS*BASE_EPS;
--
--        variable XLOCAL : REAL := ABS(X);
--        variable TEMP : REAL;
--
--    begin
--        -- Make XLOCAL < MATH_2_PI
--        if XLOCAL > MATH_2_PI then
--                TEMP := FLOOR(XLOCAL/MATH_2_PI);
--                XLOCAL := XLOCAL - TEMP*MATH_2_PI;
--        end if;
--
--        if XLOCAL < 0.0 then
--                -- adjust for rounding error
--                XLOCAL := 0.0;
--        end if;
--
--        -- Compute value for special cases
--        if XLOCAL = 0.0  or XLOCAL = MATH_2_PI then
--                return 1.0;
--        end if;
--
--        if  XLOCAL = MATH_PI then
--                return -1.0;
--        end if;
--
--        if XLOCAL = MATH_PI_OVER_2 or XLOCAL = MATH_3_PI_OVER_2 then
--                return 0.0;
--        end if;
--
--        TEMP := ABS(XLOCAL);
--        if ( TEMP < EPS) then
--                return (1.0 - 0.5*TEMP*TEMP);
--        else
--                if (TEMP < BASE_EPS) then
--                        return (1.0 -0.5*TEMP*TEMP + TEMP*TEMP*TEMP*TEMP/24.0);
--                end if;
--        end if;
--
--        TEMP := ABS(XLOCAL -MATH_2_PI);
--        if ( TEMP < EPS) then
--                return (1.0 - 0.5*TEMP*TEMP);
--        else
--                if (TEMP < BASE_EPS) then
--                        return (1.0 -0.5*TEMP*TEMP + TEMP*TEMP*TEMP*TEMP/24.0);
--                end if;
--        end if;
--
--        TEMP := ABS (XLOCAL - MATH_PI);
--        if TEMP < EPS then
--                return (-1.0 + 0.5*TEMP*TEMP);
--        else
--                if (TEMP < BASE_EPS) then
--                        return (-1.0 +0.5*TEMP*TEMP - TEMP*TEMP*TEMP*TEMP/24.0);
--                end if;
--        end if;
--
--        -- Compute value for general cases
--        return SIN(MATH_PI_OVER_2 - XLOCAL);
--   end function COS;
--
--   function TAN (X : in REAL) return REAL is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --        a) TAN(0.0) = 0.0
--        --        b) TAN(-X) = -TAN(X)
--        --        c) Returns REAL'LOW on error if X < 0.0
--        --        d) Returns REAL'HIGH on error if X > 0.0
--
--        constant NEGATIVE : BOOLEAN := X < 0.0;
--        variable XLOCAL : REAL := ABS(X) ;
--        variable VALUE: REAL;
--        variable TEMP : REAL;
--
--    begin
--        -- Make 0.0 <= XLOCAL <= MATH_2_PI
--        if XLOCAL > MATH_2_PI then
--                TEMP := FLOOR(XLOCAL/MATH_2_PI);
--                XLOCAL := XLOCAL - TEMP*MATH_2_PI;
--        end if;
--
--        if XLOCAL < 0.0 then
--                -- adjust for rounding error
--                XLOCAL := 0.0;
--        end if;
--
--        -- Check validity of argument
--        if XLOCAL = MATH_PI_OVER_2 then
--                assert FALSE
--                        report "X is a multiple of MATH_PI_OVER_2 in TAN(X)"
--                        severity ERROR;
--                if NEGATIVE then
--                        return(REAL'LOW);
--                else
--                        return(REAL'HIGH);
--                end if;
--        end if;
--
--        if XLOCAL = MATH_3_PI_OVER_2 then
--                assert FALSE
--                        report "X is a multiple of MATH_3_PI_OVER_2 in TAN(X)"
--                        severity ERROR;
--                if NEGATIVE then
--                        return(REAL'HIGH);
--                else
--                        return(REAL'LOW);
--                end if;
--        end if;
--
--        -- Compute value for special cases
--        if XLOCAL = 0.0 or XLOCAL = MATH_PI then
--                return 0.0;
--        end if;
--
--        -- Compute value for general cases
--        VALUE := SIN(XLOCAL)/COS(XLOCAL);
--        if NEGATIVE then
--                return -VALUE;
--        else
--                return VALUE;
--        end if;
--   end function TAN;
--
--   function ARCSIN (X : in REAL ) return REAL is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --        a) ARCSIN(-X) = -ARCSIN(X)
--        --        b) Returns X on error
--
--        constant NEGATIVE : BOOLEAN := X < 0.0;
--        constant XLOCAL : REAL := ABS(X);
--        variable VALUE : REAL;
--
--   begin
--      -- Check validity of arguments
--      if XLOCAL > 1.0 then
--         assert FALSE
--                report "ABS(X) > 1.0 in ARCSIN(X)"
--                severity ERROR;
--         return X;
--      end if;
--
--      -- Compute value for special cases
--      if XLOCAL = 0.0 then
--         return 0.0;
--      elsif XLOCAL = 1.0 then
--         if NEGATIVE then
--                return -MATH_PI_OVER_2;
--         else
--                return MATH_PI_OVER_2;
--         end if;
--      end if;
--
--      -- Compute value for general cases
--      if XLOCAL < 0.9 then
--         VALUE := ARCTAN(XLOCAL/(SQRT(1.0 - XLOCAL*XLOCAL)));
--      else
--         VALUE := MATH_PI_OVER_2 - ARCTAN(SQRT(1.0 - XLOCAL*XLOCAL)/XLOCAL);
--      end if;
--
--      if NEGATIVE then
--         VALUE := -VALUE;
--      end if;
--
--      return VALUE;
--   end function ARCSIN;
--
--   function ARCCOS (X : in REAL) return REAL is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --        a) ARCCOS(-X) = MATH_PI - ARCCOS(X)
--        --        b) Returns X on error
--
--        constant NEGATIVE : BOOLEAN := X < 0.0;
--        constant XLOCAL : REAL := ABS(X);
--        variable VALUE : REAL;
--
--   begin
--      -- Check validity of argument
--      if XLOCAL > 1.0 then
--         assert FALSE
--                report "ABS(X) > 1.0 in ARCCOS(X)"
--                severity ERROR;
--         return X;
--      end if;
--
--      -- Compute value for special cases
--      if X = 1.0 then
--         return 0.0;
--      elsif X = 0.0 then
--         return MATH_PI_OVER_2;
--      elsif X = -1.0 then
--         return MATH_PI;
--      end if;
--
--      -- Compute value for general cases
--      if XLOCAL > 0.9 then
--         VALUE := ARCTAN(SQRT(1.0 - XLOCAL*XLOCAL)/XLOCAL);
--      else
--         VALUE := MATH_PI_OVER_2 - ARCTAN(XLOCAL/SQRT(1.0 - XLOCAL*XLOCAL));
--      end if;
--
--
--      if NEGATIVE then
--         VALUE := MATH_PI - VALUE;
--      end if;
--
--      return VALUE;
--   end function ARCCOS;
--
--
--   function ARCTAN (Y : in REAL) return REAL is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --        a) ARCTAN(-Y) = -ARCTAN(Y)
--        --        b) ARCTAN(Y) = -ARCTAN(1.0/Y) + MATH_PI_OVER_2 for |Y| > 1.0
--        --        c) ARCTAN(Y) = Y for |Y| < EPS
--
--        constant EPS : REAL := BASE_EPS*BASE_EPS*BASE_EPS;
--
--        constant NEGATIVE : BOOLEAN := Y < 0.0;
--        variable RECIPROCAL : BOOLEAN;
--        variable YLOCAL : REAL := ABS(Y);
--        variable VALUE : REAL;
--
--   begin
--      -- Make argument |Y| <=1.0
--      if YLOCAL > 1.0 then
--                YLOCAL := 1.0/YLOCAL;
--                RECIPROCAL := TRUE;
--      else
--                RECIPROCAL := FALSE;
--      end if;
--
--      -- Compute value for special cases
--      if YLOCAL = 0.0 then
--         if RECIPROCAL then
--                if NEGATIVE then
--                        return (-MATH_PI_OVER_2);
--                else
--                        return (MATH_PI_OVER_2);
--                end if;
--         else
--                return 0.0;
--         end if;
--      end if;
--
--      if YLOCAL < EPS then
--         if NEGATIVE then
--                if RECIPROCAL then
--                        return (-MATH_PI_OVER_2 + YLOCAL);
--                else
--                        return -YLOCAL;
--                end if;
--         else
--                if RECIPROCAL then
--                        return (MATH_PI_OVER_2 - YLOCAL);
--                else
--                        return YLOCAL;
--                end if;
--         end if;
--      end if;
--
--      -- Compute value for general cases
--      VALUE :=  CORDIC( 1.0, YLOCAL, 0.0, 27, VECTORING )(2);
--
--      if RECIPROCAL then
--         VALUE := MATH_PI_OVER_2 - VALUE;
--      end if;
--
--      if NEGATIVE then
--        VALUE := -VALUE;
--      end if;
--
--      return VALUE;
--   end function ARCTAN;
--
--
--   function ARCTAN (Y : in REAL; X : in REAL) return REAL is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --         a) Returns 0.0 on error
--
--        variable YLOCAL : REAL;
--        variable VALUE : REAL;
--   begin
--
--     -- Check validity of arguments
--     if (Y = 0.0 and X = 0.0 ) then
--           assert FALSE report
--                "ARCTAN(0.0, 0.0) is undetermined"
--                severity ERROR;
--           return 0.0;
--     end if;
--
--     -- Compute value for special cases
--     if Y = 0.0 then
--        if X > 0.0 then
--           return 0.0;
--        else
--           return MATH_PI;
--        end if;
--     end if;
--
--     if X = 0.0 then
--        if Y > 0.0 then
--           return MATH_PI_OVER_2;
--        else
--           return -MATH_PI_OVER_2;
--        end if;
--     end if;
--
--
--     -- Compute value for general cases
--     YLOCAL := ABS(Y/X);
--
--     VALUE := ARCTAN(YLOCAL);
--
--     if X < 0.0 then
--         VALUE := MATH_PI - VALUE;
--     end if;
--
--     if Y < 0.0 then
--         VALUE := -VALUE;
--     end if;
--
--     return VALUE;
--   end function ARCTAN;
--
--
--    function SINH (X : in REAL) return REAL is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --        a) Returns (EXP(X) - EXP(-X))/2.0
--        --        b) SINH(-X) = SINH(X)
--
--        constant NEGATIVE : BOOLEAN := X < 0.0;
--        constant XLOCAL : REAL := ABS(X);
--        variable TEMP : REAL;
--        variable VALUE : REAL;
--
--    begin
--        -- Compute value for special cases
--        if XLOCAL = 0.0 then
--                return 0.0;
--        end if;
--
--        -- Compute value for general cases
--        TEMP := EXP(XLOCAL);
--        VALUE := (TEMP - 1.0/TEMP)*0.5;
--
--         if NEGATIVE then
--                VALUE := -VALUE;
--        end if;
--
--        return VALUE;
--    end function SINH;
--
--    function  COSH (X : in REAL) return REAL is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --        a) Returns (EXP(X) + EXP(-X))/2.0
--        --        b) COSH(-X) = COSH(X)
--
--        constant XLOCAL : REAL := ABS(X);
--        variable TEMP : REAL;
--        variable VALUE : REAL;
--    begin
--        -- Compute value for special cases
--        if XLOCAL = 0.0 then
--                return 1.0;
--        end if;
--
--
--        -- Compute value for general cases
--        TEMP := EXP(XLOCAL);
--        VALUE := (TEMP + 1.0/TEMP)*0.5;
--
--        return VALUE;
--    end function COSH;
--
--    function  TANH (X : in REAL) return REAL is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --        a) Returns (EXP(X) - EXP(-X))/(EXP(X) + EXP(-X))
--        --        b) TANH(-X) = -TANH(X)
--
--        constant NEGATIVE : BOOLEAN := X < 0.0;
--        constant XLOCAL : REAL := ABS(X);
--        variable TEMP : REAL;
--        variable VALUE : REAL;
--
--    begin
--        -- Compute value for special cases
--        if XLOCAL = 0.0 then
--                return 0.0;
--        end if;
--
--        -- Compute value for general cases
--        TEMP := EXP(XLOCAL);
--        VALUE := (TEMP - 1.0/TEMP)/(TEMP + 1.0/TEMP);
--
--        if NEGATIVE then
--            return -VALUE;
--        else
--            return VALUE;
--        end if;
--    end function TANH;
--
--    function ARCSINH (X : in REAL) return REAL is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --        a) Returns LOG( X + SQRT( X*X + 1.0))
--
--    begin
--        -- Compute value for special cases
--        if X = 0.0 then
--                return 0.0;
--        end if;
--
--        -- Compute value for general cases
--        return ( LOG( X + SQRT( X*X + 1.0)) );
--    end function ARCSINH;
--
--
--
--   function ARCCOSH (X : in REAL) return REAL is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --        a) Returns LOG( X + SQRT( X*X - 1.0));   X >= 1.0
--        --        b) Returns X on error
--
--    begin
--        -- Check validity of arguments
--        if X < 1.0 then
--                 assert FALSE
--                        report "X < 1.0 in ARCCOSH(X)"
--                        severity ERROR;
--                 return X;
--        end if;
--
--        -- Compute value for special cases
--        if X = 1.0 then
--                return 0.0;
--        end if;
--
--        -- Compute value for general cases
--        return ( LOG( X + SQRT( X*X - 1.0)));
--    end function ARCCOSH;
--
--    function ARCTANH (X : in REAL) return REAL is
--        -- Description:
--        --        See function declaration in IEEE Std 1076.2-1996
--        -- Notes:
--        --        a) Returns (LOG( (1.0 + X)/(1.0 - X)))/2.0 ; | X | < 1.0
--        --        b) Returns X on error
--    begin
--        -- Check validity of arguments
--        if ABS(X) >= 1.0 then
--                assert FALSE
--                        report "ABS(X) >= 1.0 in ARCTANH(X)"
--                        severity ERROR;
--                return X;
--        end if;
--
--        -- Compute value for special cases
--        if X = 0.0 then
--                return 0.0;
--        end if;
--
--        -- Compute value for general cases
--        return( 0.5*LOG( (1.0+X)/(1.0-X) ) );
--    end function ARCTANH;


end package body MATH_REAL;
