#include <dvt_ucis.h>

vector<module*>* module::get_all_modules(sqlite3 *db) {
    sqlite3_stmt *stmt;
    const char *selectModules = "SELECT moduleId, name FROM UCIS_DU_MODULE;";
    int rc = sqlite3_prepare_v2(db, selectModules, -1, &stmt, NULL);
    if (rc != SQLITE_OK)
    {
        printf("Prepare failed 1: %s\n", sqlite3_errmsg(db));
        sqlite3_close(db);
        return NULL;
    }

    vector<module*> *modules = new vector<module*>();
    while ((rc = sqlite3_step(stmt)) == SQLITE_ROW)
    {
        int moduleId = sqlite3_column_int(stmt, 0);
        const char *name = (const char *)sqlite3_column_text(stmt, 1);
        modules->push_back(new module(moduleId, string(name)));
    }

    return modules;
}

vector<module_instance*>* module_instance::get_all_instances(sqlite3 *db) {
    sqlite3_stmt *stmt;
    const char *selectInstances =
        "SELECT i.instanceId, i.name, i.fullPath, i.moduleId, m.name "
        "FROM UCIS_INSTANCE AS i JOIN UCIS_DU_MODULE AS m "
        "ON i.moduleId = m.moduleId "
        "WHERE i.parentId IS NULL;";

    int rc = sqlite3_prepare_v2(db, selectInstances, -1, &stmt, NULL);
    if (rc != SQLITE_OK)
    {
        printf("Prepare failed 2: %s\n", sqlite3_errmsg(db));
        sqlite3_close(db);
        return NULL;
    }

    vector<module_instance*>* instances = new vector<module_instance*>();
    while ((rc = sqlite3_step(stmt)) == SQLITE_ROW)
    {
        int instanceId = sqlite3_column_int(stmt, 0);
        const char *instanceName = (const char *)sqlite3_column_text(stmt, 1);
        const char *fullPath = (const char *)sqlite3_column_text(stmt, 2);
        int moduleId = sqlite3_column_int(stmt, 3);
        const char *moduleName = (const char *)sqlite3_column_text(stmt, 4);

        instances->push_back(new module_instance(instanceId, instanceName, fullPath, new module(moduleId, string(moduleName))));
    }

    return instances;
}

vector<covergroup_type*>* covergroup_type::get_all_cg_types_from_instance(sqlite3 *db, module_instance *instance) {
    sqlite3_stmt *stmt;
    const char *selectCoverDef = "SELECT covergroupId, name FROM UCIS_COVERGROUP WHERE moduleId = ?;";

    int rc = sqlite3_prepare_v2(db, selectCoverDef, -1, &stmt, 0);
    if (rc != SQLITE_OK)
    {
        fprintf(stderr, "Failed to prepare statement 3: %s\n", sqlite3_errmsg(db));
        sqlite3_close(db);
        return NULL;
    }

    sqlite3_bind_int(stmt, 1, instance->typeModule->id);

    vector<covergroup_type*>* cg_types = new vector<covergroup_type*>();
    while ((rc = sqlite3_step(stmt)) == SQLITE_ROW)
    {
        int cgDefId = sqlite3_column_int(stmt, 0);
        const char *name = (const char *)sqlite3_column_text(stmt, 1);

        cg_types->push_back(new covergroup_type(cgDefId, string(name), instance->typeModule));
    }

    return cg_types;
}

vector<coverpoint_type*>* coverpoint_type::get_all_cp_types_from_cg_type(sqlite3 *db, covergroup_type* cg_type) {
    sqlite3_stmt *stmt;
    const char *selectCoverPointsDef = "SELECT coverpointId, name, expression, isCross FROM UCIS_COVERPOINT WHERE covergroupId = ?;";

    int rc = sqlite3_prepare_v2(db, selectCoverPointsDef, -1, &stmt, 0);
    if (rc != SQLITE_OK)
    {
        fprintf(stderr, "Failed to prepare statement 4: %s\n", sqlite3_errmsg(db));
        sqlite3_close(db);
        return NULL;
    }

    sqlite3_bind_int(stmt, 1, cg_type->id);

    vector<coverpoint_type*>* cps = new vector<coverpoint_type*>();
    while ((rc = sqlite3_step(stmt)) == SQLITE_ROW)
    {
        int cpDefId = sqlite3_column_int(stmt, 0);
        const char *name = (const char *)sqlite3_column_text(stmt, 1);
        const char *expression = (const char *)sqlite3_column_text(stmt, 2);
        const bool isCross = sqlite3_column_int(stmt, 3); 
        cps->push_back(new coverpoint_type(cpDefId, string(name), string(expression), isCross, cg_type));
    }

    return cps;
}

vector<covergroup_instance*>* covergroup_instance::get_all_cg_instances_from_instance(sqlite3 *db, module_instance *instance) {
    sqlite3_stmt *stmt;
    const char *selectCoverInstances =
        "SELECT ci.cgInstanceId, ci.name, ci.coverage, ci.weight, ci.covergroupId, cg.name "
        "FROM UCIS_COVERGROUP_INSTANCE AS ci JOIN UCIS_COVERGROUP as cg "
        "ON ci.covergroupId = cg.covergroupId "
        "WHERE ci.instanceId = ?;";

    int rc = sqlite3_prepare_v2(db, selectCoverInstances, -1, &stmt, 0);
    if (rc != SQLITE_OK)
    {
        fprintf(stderr, "Failed to prepare statement 7: %s\n", sqlite3_errmsg(db));
        sqlite3_close(db);
        return NULL;
    }

    sqlite3_bind_int(stmt, 1, instance->id);
    vector<covergroup_instance*>* cgs = new vector<covergroup_instance*>();
    while ((rc = sqlite3_step(stmt)) == SQLITE_ROW)
    {
        int cgInstanceId = sqlite3_column_int(stmt, 0);
        const char *cgInstanceName = (const char *)sqlite3_column_text(stmt, 1);
        double coverage = sqlite3_column_double(stmt, 2);
        int weight = sqlite3_column_int(stmt, 3);
        int cgId = sqlite3_column_int(stmt, 4);
        const char *cgName = (const char *)sqlite3_column_text(stmt, 5);

        cgs->push_back(new covergroup_instance(cgInstanceId, string(cgInstanceName), coverage, weight, 
                            new covergroup_type(cgId, cgInstanceName, instance->typeModule), instance));
    }

    return cgs;
}

coverpoint_instance* coverpoint_instance::get_cp_instance_by_name(sqlite3 *db, string name) {
    sqlite3_stmt *stmt;
    const char *selectCoverPointsDef =
        "SELECT ci.cpInstanceId, cp.name, ci.coverpointId, ci.coverage, cp.isCross, cp.expression "
        "FROM UCIS_COVERPOINT_INSTANCE as ci JOIN UCIS_COVERPOINT as cp "
        "ON ci.coverpointId = cp.coverpointId "
        "WHERE ci.cgInstanceId = ? AND cp.name = ?;";

    int rc = sqlite3_prepare_v2(db, selectCoverPointsDef, -1, &stmt, 0);
    if (rc != SQLITE_OK)
    {
        fprintf(stderr, "Failed to prepare statement 6: %s\n", sqlite3_errmsg(db));
        sqlite3_close(db);
        return NULL;
    }

    sqlite3_bind_int(stmt, 1, cg_instance->id);
    sqlite3_bind_text(stmt, 2, name.c_str(), -1, 0);

    if ((rc = sqlite3_step(stmt)) == SQLITE_ROW)
    {
        int cpInstanceId = sqlite3_column_int(stmt, 0);
        const char *cpName = (const char *)sqlite3_column_text(stmt, 1);
        int cpId = sqlite3_column_int(stmt, 2);
        double coverage = sqlite3_column_double(stmt, 3);
        const bool isCross = sqlite3_column_int(stmt, 4);
        const char *expression = (const char *)sqlite3_column_text(stmt, 5);

        return new coverpoint_instance(cpInstanceId, coverage, new coverpoint_type(cpId, string(cpName), string(expression), isCross, cg_instance->cg_type), cg_instance);
    } 

    return new coverpoint_instance(-1, 0, new coverpoint_type(-1, string(name), "", false, cg_instance->cg_type), cg_instance);
}

vector<coverpoint_instance*>* coverpoint_instance::get_all_cp_instances_from_cg_instance(sqlite3 *db, covergroup_instance* cg_instance) {
    sqlite3_stmt *stmt;
    const char *selectCoverPointsDef =
        "SELECT ci.cpInstanceId, cp.name, ci.coverpointId, ci.coverage, cp.isCross, cp.expression "
        "FROM UCIS_COVERPOINT_INSTANCE as ci JOIN UCIS_COVERPOINT as cp "
        "ON ci.coverpointId = cp.coverpointId "
        "WHERE ci.cgInstanceId = ?;";

    int rc = sqlite3_prepare_v2(db, selectCoverPointsDef, -1, &stmt, 0);
    if (rc != SQLITE_OK)
    {
        fprintf(stderr, "Failed to prepare statement 6: %s\n", sqlite3_errmsg(db));
        sqlite3_close(db);
        return NULL;
    }

    sqlite3_bind_int(stmt, 1, cg_instance->id);

    vector<coverpoint_instance*>* cps = new vector<coverpoint_instance*>();
    while ((rc = sqlite3_step(stmt)) == SQLITE_ROW)
    {
        int cpInstanceId = sqlite3_column_int(stmt, 0);
        const char *cpName = (const char *)sqlite3_column_text(stmt, 1);
        int cpId = sqlite3_column_int(stmt, 2);
        double coverage = sqlite3_column_double(stmt, 3);
        const bool isCross = sqlite3_column_int(stmt, 4);
        const char *expression = (const char *)sqlite3_column_text(stmt, 5);

        cps->push_back(new coverpoint_instance(cpInstanceId, coverage, new coverpoint_type(cpId, string(cpName), string(expression), isCross, cg_instance->cg_type), cg_instance));
    }

    return cps;
}

vector<bin_instance*>* bin_instance::get_all_bins_from_cp_instance(sqlite3 *db, coverpoint_instance *cp_instance) {
    sqlite3_stmt *stmt;
    const char *selectBins =
        "SELECT binInstanceId, name, hitCount, type "
        "FROM UCIS_BIN "
        "WHERE cpInstanceId = ?;";

    int rc = sqlite3_prepare_v2(db, selectBins, -1, &stmt, 0);
    if (rc != SQLITE_OK)
    {
        fprintf(stderr, "Failed to prepare statement 5: %s\n", sqlite3_errmsg(db));
        sqlite3_close(db);
        return NULL;
    }

    sqlite3_bind_int(stmt, 1, cp_instance->id);
    vector<bin_instance*>* bins = new vector<bin_instance*>();
    while ((rc = sqlite3_step(stmt)) == SQLITE_ROW)
    {
        int binId = sqlite3_column_int(stmt, 0);
        const char *name = (const char *)sqlite3_column_text(stmt, 1);
        int hitCount = sqlite3_column_int(stmt, 2);
        int type = sqlite3_column_int(stmt, 3);

        bins->push_back(new bin_instance(binId, string(name), hitCount, type, cp_instance));
    }

    return bins;
}

bin_instance* bin_instance::get_bin_from_id(sqlite3 *db, int binId) {
    sqlite3_stmt *stmt;
    const char *selectBinInst =
        "SELECT binInstanceId, name, hitCount, type "
        "FROM UCIS_BIN "
        "WHERE binInstanceId = ?;";

    int rc = sqlite3_prepare_v2(db, selectBinInst, -1, &stmt, 0);

    if (rc != SQLITE_OK)
    {
        fprintf(stderr, "Failed to prepare statement 3: %s\n", sqlite3_errmsg(db));
        sqlite3_close(db);
        return NULL;
    }

    sqlite3_bind_int(stmt, 1, binId);
    if ((rc = sqlite3_step(stmt)) != SQLITE_ROW)
        return NULL;

    binId = sqlite3_column_int(stmt, 0);
    const char *name = (const char *)sqlite3_column_text(stmt, 1);
    int hitCount = sqlite3_column_int(stmt, 2);
    int type = sqlite3_column_int(stmt, 3);

    return new bin_instance(binId, string(name), hitCount, type, NULL);
}

string* module::get_string_property(ucisStringPropertyEnumT property) {
    switch (property)
    {
    case UCIS_STR_SCOPE_HIER_NAME:
        return new string(name);
    default:
        return NULL;
    }
}

string* module_instance::get_string_property(ucisStringPropertyEnumT property) {
    switch (property)
    {
    case UCIS_STR_SCOPE_HIER_NAME:
        return new string(fullPath);
    case UCIS_STR_SCOPE_NAME:
        return new string(name);
    default:
        return NULL;
    }
}

string* covergroup_instance::get_string_property(ucisStringPropertyEnumT property) {
    switch (property)
    {
    case UCIS_STR_SCOPE_HIER_NAME:
        return new string("/" + cg_type->name);
    case UCIS_STR_SCOPE_NAME:
        return new string(name);
    default:
        return NULL;
    }
}

string* coverpoint_instance::get_string_property(ucisStringPropertyEnumT property) {
    switch (property)
    {
    case UCIS_STR_SCOPE_HIER_NAME:
        return new string(*(cg_instance->get_string_property(property)) + "/" + cp_type->name);
    case UCIS_STR_SCOPE_NAME:
        return new string(cp_type->name);
    default:
        return NULL;
    }
}

string* bin_instance::get_string_property(ucisStringPropertyEnumT property) {
    switch (property)
    {
    case UCIS_STR_SCOPE_HIER_NAME:
        return cp_instance->get_string_property(property);
    case UCIS_STR_SCOPE_NAME:
        return new string(name);
    default:
        return NULL;
    }
}

string strip(const string& str) {
    auto start = find_if_not(str.begin(), str.end(), [](unsigned char ch) { return isspace(ch); });
    auto end = find_if_not(str.rbegin(), str.rend(), [](unsigned char ch) { return isspace(ch); }).base();
    
    return (start < end) ? string(start, end) : string();
}

vector<string> split(string s, string delimiter) {
    size_t pos_start = 0, pos_end, delim_len = delimiter.length();
    string token;
    vector<string> res;

    while ((pos_end = s.find(delimiter, pos_start)) != string::npos) {
        token = s.substr (pos_start, pos_end - pos_start);
        pos_start = pos_end + delim_len;
        res.push_back (token);
    }

    res.push_back (s.substr (pos_start));
    return res;
}

void coverpoint_instance::init_cross_items(sqlite3 *db) {
    vector<string> cross_cvps_names = split(cp_type->expression, ",");
    cross_items = new vector<coverpoint_instance*>();

    for (string cvp_name : cross_cvps_names) 
        cross_items->push_back(get_cp_instance_by_name(db, strip(cvp_name)));
}

string reverseItems(const std::string& input) {
    // Remove the angle brackets
    string content = input.substr(1, input.length() - 2);
    
    // Parse items into a vector
    vector<std::string> items;
    stringstream ss(content);
    string item;
    
    while (getline(ss, item, ',')) 
        items.push_back(strip(item));
    
    reverse(items.begin(), items.end());
    
    // Build the result string
    string result;
    for (size_t i = 0; i < items.size(); ++i) {
        result += items[i];
        if (i < items.size() - 1) {
            result += ",";
        }
    }
    
    return result;
}