#include <string>
#include <iostream>
#include <sqlite3.h>
#include <ucis.h>
#include <vector>
#include <string.h>
#include <sstream>
#include <algorithm>
#include <cctype>

using namespace std;

string reverseItems(const std::string& input);

enum entity_type {
    MODULE,
    MODULE_INSTANCE,
    COVERGROUP_TYPE,
    COVERGROUP_INSTANCE,
    COVERPOINT_TYPE,
    COVERPOINT_INSTANCE,
    CROSS_INSTANCE,
    BIN_INSTANCE
};
class entity {
    public:
        entity_type type;

        entity(entity_type type) : type(type) {}
        virtual string* get_string_property(ucisStringPropertyEnumT property) = 0;
        ucisScopeTypeT get_scope_type() {
            switch (type) {
                case COVERGROUP_TYPE: 
                    return UCIS_COVERGROUP;
                case COVERGROUP_INSTANCE:
                    return UCIS_COVERINSTANCE;
                case COVERPOINT_INSTANCE:
                    return UCIS_COVERPOINT;
                case CROSS_INSTANCE:
                    return UCIS_CROSS;
                case BIN_INSTANCE:
                    return UCIS_CVGBIN;
            }
        }
};

class module : entity {
    public:
        int id;
        string name;

        module(int id, string name) : entity(MODULE), id(id), name(name) {}

        static vector<module*>* get_all_modules(sqlite3 *db);
        string* get_string_property(ucisStringPropertyEnumT property);
};

class module_instance : entity {
    public:
        int id;
        string name;
        string fullPath;
        module* typeModule;

        module_instance(int id, string name, string fullPath, module* typeModule) : 
            entity(MODULE_INSTANCE), id(id), name(name), fullPath(fullPath), typeModule(typeModule) {}

        static vector<module_instance*>* get_all_instances(sqlite3 *db);
        string* get_string_property(ucisStringPropertyEnumT property);
};

class covergroup_type : entity {
    public:
        int id;
        string name;
        module* moduleType;

        covergroup_type(int id, string name, module* moduleType) :
            entity(COVERGROUP_TYPE), id(id), name(name), moduleType(moduleType) {}

        static vector<covergroup_type*>* get_all_cg_types_from_instance(sqlite3 *db, module_instance* parentInstance);
        string* get_string_property(ucisStringPropertyEnumT property) {return NULL;};
};

class covergroup_instance : entity {
    public:
        int id;
        string name;
        double coverage;
        int weight;
        covergroup_type* cg_type;
        module_instance* parentInstance;
    public:
        covergroup_instance(int id, string name, double coverage, int weight, covergroup_type* cg_type, module_instance* parentInstance) : 
            entity(COVERGROUP_INSTANCE), id(id), name(name), coverage(coverage), weight(weight), cg_type(cg_type), parentInstance(parentInstance) {}

        static vector<covergroup_instance*>* get_all_cg_instances_from_instance(sqlite3 *db, module_instance *parentInstance);
        string* get_string_property(ucisStringPropertyEnumT property);
};

class coverpoint_type : entity {
    public:
        int id;
        string name;
        string expression;
        bool isCross;
        covergroup_type* cg_type;
    public:
        coverpoint_type(int id, string name, string expression, bool isCross, covergroup_type* cg_type) : 
            entity(COVERPOINT_TYPE), id(id), name(name), expression(expression), isCross(isCross), cg_type(cg_type) {
            this->type = COVERPOINT_TYPE;
        }

        static vector<coverpoint_type*>* get_all_cp_types_from_cg_type(sqlite3 *db, covergroup_type* cg_type);
        string* get_string_property(ucisStringPropertyEnumT property) {return NULL;};
};

class coverpoint_instance : entity {
    public:
        int id;
        double coverage;
        coverpoint_type* cp_type;
        covergroup_instance* cg_instance;
        vector<coverpoint_instance*>* cross_items = nullptr;
    public:
        coverpoint_instance(int id, double coverage, coverpoint_type* cp_type, covergroup_instance* cg_instance) : 
            entity(cp_type->isCross ? CROSS_INSTANCE : COVERPOINT_INSTANCE), id(id), cp_type(cp_type), coverage(coverage), 
            cg_instance(cg_instance) {}

        coverpoint_instance *get_cp_instance_by_name(sqlite3 *db, string name);
        static vector<coverpoint_instance *> *get_all_cp_instances_from_cg_instance(sqlite3 *db, covergroup_instance *cg_instance);
        string* get_string_property(ucisStringPropertyEnumT property);
        void init_cross_items(sqlite3 *db);
};

class bin_instance : entity {
    public:
        int id;
        string name;
        int hitCount;
        int binType;
        int ucisBinType = UCIS_CVGBIN;
        coverpoint_instance* cp_instance;
    public:
        bin_instance(int id, string name, int hitCount, int binType, coverpoint_instance* cp_instance) : 
            entity(BIN_INSTANCE), id(id), hitCount(hitCount), binType(binType), cp_instance(cp_instance) {
                this->name = cp_instance->cp_type->isCross ? reverseItems(name) : name;
            }

        static vector<bin_instance*>* get_all_bins_from_cp_instance(sqlite3 *db, coverpoint_instance *cp_instance);
        static bin_instance* get_bin_from_id(sqlite3 *db, int binId);
        string* get_string_property(ucisStringPropertyEnumT property);
};