import { 
  createEditor, $createTextNode,
  $getRoot, $createParagraphNode, $getSelection, $isRangeSelection, getDOMSelection,
  KEY_DOWN_COMMAND,
  UNDO_COMMAND,
  REDO_COMMAND,
  COPY_COMMAND,
  PASTE_COMMAND,
  CUT_COMMAND,
  COMMAND_PRIORITY_LOW,
  COMMAND_PRIORITY_EDITOR,
  SELECTION_CHANGE_COMMAND,
} from 'lexical';
import { registerHistory, createEmptyHistoryState } from '@lexical/history';
import { registerPlainText } from '@lexical/plain-text';

const containerToEditorMap = new WeakMap();
function getEditorForComment(comment) {
  return containerToEditorMap.get(comment);
}

export function createMarkdownEditor(container, options = {}) {
  const toolBar = container.querySelector('div.toolbar');
  const floatingEditor = container.querySelector('.floating-text-format-popup');
  const floatingToolbar = floatingEditor.querySelector('div.toolbar');
  const previewContainer = container.querySelector('div.panes > div.preview');
  const editorContainer = container.querySelector('div.panes > div.editor');
  const initialText = editorContainer.textContent;
  const individualRegenerateBtn = container.querySelector(REGENERATE_CLS);


  const historyState = createEmptyHistoryState();
  const editor = createEditor({ namespace: 'MarkdownEditor', onError: console.error });
  editor.setRootElement(editorContainer);
  containerToEditorMap.set(container, editor);

  registerPlainText(editor);
  registerHistory(editor, historyState, 1000);

  editor.update(() => {
    const root = $getRoot();
    const paragraphNode = $createParagraphNode()
    paragraphNode.append($createTextNode(initialText));
    root.append(paragraphNode);
  });

  editor.registerTextContentListener(
    (textContent) => {
      const aiGenerated = container.classList.contains(AI_TARGET_RESULT_CLASSNAME);

      if (textContent.trim() !== initialText.trim()) {
        // Update the comment if there are any changes
        addCommentToEditList(container, textContent);
      } else if (!aiGenerated) {
        // Remove only original comments, AI generated are still new, even if their content wasn't edited
        removeCommentFromEditList(container);
      }
    },
  );

  editor.registerCommand(
    KEY_DOWN_COMMAND,
    (event) => {
      const key = (event.key || '').toLowerCase();
  
      if (!event.ctrlKey || event.repeat) {
        return false;
      }
  
  
      if (!event.shiftKey && key === 'z') {
        event.preventDefault();
        editor.dispatchCommand(UNDO_COMMAND, undefined);
        return true;
      }
  
      if (key === 'y' || (key === 'z' && event.shiftKey)) {
        event.preventDefault();
        editor.dispatchCommand(REDO_COMMAND, undefined);
        return true;
      }
      return false;
    },
    COMMAND_PRIORITY_EDITOR
  );

  editor.registerCommand(
    COPY_COMMAND,
    (event) => {
      return false;
    },
    COMMAND_PRIORITY_LOW
  );
  
  editor.registerCommand(
    PASTE_COMMAND,
    (event) => {
      return false;
    },
    COMMAND_PRIORITY_LOW
  );

  editor.registerCommand(
    CUT_COMMAND,
    (event) => {
      return false;
    },
    COMMAND_PRIORITY_LOW
  );

  function bold() {
    editor.update(() => {
      const sel = $getSelection();
      if ($isRangeSelection(sel)) {
        let text = sel.getTextContent();

        if (text.length === 0) {
          sel.insertText("****");
          return;
        }

        const leadingLength = text.length - text.trimStart().length;
        const trailingLength = text.length - text.trimEnd().length;
  
        const leading = text.slice(0, leadingLength);
        const trailing = text.slice(text.length - trailingLength);
        const core = text.slice(leadingLength, text.length - trailingLength);
  
        sel.insertText(`${leading}**${core}**${trailing}`);
      }
    });
  }

  function italic() {
    editor.update(() => {
      const sel = $getSelection();
      if ($isRangeSelection(sel)) {
        let text = sel.getTextContent();

        if (text.length === 0) {
          sel.insertText("****");
          return;
        }
        
        const leadingLength = text.length - text.trimStart().length;
        const trailingLength = text.length - text.trimEnd().length;
  
        const leading = text.slice(0, leadingLength);
        const trailing = text.slice(text.length - trailingLength);
        const core = text.slice(leadingLength, text.length - trailingLength);
  
        sel.insertText(`${leading}*${core}*${trailing}`);
      }
    });
  }

  function heading() {
    editor.update(() => {
      const sel = $getSelection();
      const selectedText = sel.getTextContent();
      if ($isRangeSelection(sel)) {
        if (selectedText.length) {
          sel.insertText(`# ${selectedText}`);
        } else {
          sel.insertLineBreak();
          sel.insertText('# ');
        }
      }
    });
  }

  function unorderedList() {
    editor.update(() => {
      const sel = $getSelection();
      const text = sel.getTextContent();
      if ($isRangeSelection(sel)) sel.insertText(`\n- ${text}`);
    });
  }

  function codeBlock() {
    editor.update(() => {
      const sel = $getSelection();
      if ($isRangeSelection(sel)) {
        const selected = sel.getTextContent();
        sel.insertText(`\`${selected}\``);
      }
    });
  }

  function underline() {
    editor.update(() => {
      const sel = $getSelection();
      if ($isRangeSelection(sel)) {
        const selected = sel.getTextContent();
        sel.insertText(`<u>${selected}</u>`);
      }
    });
  }

  function quote() {
    editor.update(() => {
      const sel = $getSelection();
      if ($isRangeSelection(sel)) {
        const lines = sel.getTextContent().split('\n');
        const quoted = lines.map(l => `> ${l}`).join('\n');
        sel.insertText(quoted);
      }
    });
  }

  for (const buttonSet of [toolBar, floatingToolbar]) {
    buttonSet.querySelector('.bold').onclick = bold;
    buttonSet.querySelector('.italic').onclick = italic;
    buttonSet.querySelector('.h1-btn').onclick = heading;
    buttonSet.querySelector('.ul-btn').onclick = unorderedList;
    buttonSet.querySelector('.code').onclick = codeBlock;
    buttonSet.querySelector('.underline').onclick = underline;
    buttonSet.querySelector('.quote').onclick = quote;
  }

  toolBar.querySelector('.toggle').onclick = () => {
    if (editorContainer.classList.contains('hidden')) {
      editorContainer.classList.remove('hidden');
      toolBar.querySelector('.style-btns').classList.remove('hidden');
      previewContainer.classList.add('hidden');
      toolBar.querySelector('.toggle').textContent = 'Preview';

      // Add dragging listeners
      document.addEventListener('mouseup', () => setDraggingState(false));
      document.addEventListener('mouseleave', () => setDraggingState(false));
    } else {
      editorContainer.classList.add('hidden');
      toolBar.querySelector('.style-btns').classList.add('hidden');
      floatingEditor.classList.add('hidden');
      previewContainer.classList.remove('hidden');
      refreshPreview(editor, previewContainer);
      toolBar.querySelector('.toggle').textContent = 'Continue editing';

      // Remove editor listeners
      document.removeEventListener('mouseup', setDraggingState);
      document.removeEventListener('mouseleave', setDraggingState);
    }
  };

  editor.registerCommand(
    SELECTION_CHANGE_COMMAND,
    () => {
      const selection = $getSelection();

      // hide when there is no text selection
      if (
        !$isRangeSelection(selection) ||
        selection.getTextContent().length === 0
      ) {
        floatingEditor.classList.add('hidden');
        floatingEditor.style.opacity = '0';
        return false;
      }

      // we have a range so we show the toolbar
      floatingEditor.classList.remove('hidden');

      const nativeSel = getDOMSelection(editor._window);
      const rangeRect = getDOMRangeRect(nativeSel, editor.getRootElement());

      setFloatingElemPosition(
        rangeRect,
        floatingEditor,
        editor.getRootElement(),
      );

      return false; // let other listeners run
    },
    COMMAND_PRIORITY_EDITOR,
  );

  function setDraggingState(dragging) {
    if (dragging) {
      floatingEditor.classList.add('dragging');
    } else {
      floatingEditor.classList.remove('dragging');
    }
  }

  const rootEl = editor.getRootElement();

  rootEl.addEventListener('mousedown', (e) => {
    // Only start dragging when the click is *inside* the editable area,
    // not when it lands on a button.
    if (!e.target.closest('.floating-text-format-popup')) {
      setDraggingState(true);
    }
  });

  document.addEventListener('mouseup', () => setDraggingState(false));
  document.addEventListener('mouseleave', () => setDraggingState(false));
}

// function runImproveAction(editor, prompt) {
//   let markdown = '';
// 
//   editor.update(() => {
//     const selection = $getSelection();
//     const root = $getRoot();
//     markdown = root.getTextContent();
//     root.clear();
//   });
// 
//   improveExistingDescription(editor, prompt, markdown);
// }

function refreshPreview(editor, previewContainer) {
  let markdown = '';
  editor.getEditorState().read(() => {
    const root = $getRoot();
    markdown = root.getTextContent();
  });
  updatePreviewEditor(markdown, previewContainer);
}

function setFloatingElemPosition(
  targetRect,
  floatingElem,
  anchorElem,
  verticalGap = 10,
  horizontalOffset = 5,
) {
  const scrollerElem = anchorElem.parentElement; // the scrolling wrapper
  if (!targetRect || !scrollerElem) {
    floatingElem.style.opacity = '0';
    floatingElem.style.transform = 'translate(-10000px, -10000px)';
    return;
  }

  const floatingRect = floatingElem.getBoundingClientRect(); // width/height of the toolbar
  const anchorRect   = anchorElem.getBoundingClientRect();   // editor root (page‑coords)
  const scrollerRect = scrollerElem.getBoundingClientRect(); // scroll container (page‑coords)

  let top  = targetRect.top - floatingRect.height - verticalGap; // above selection
  let left = targetRect.left - horizontalOffset;                // a little left

  const sel = window.getSelection();
  if (sel?.rangeCount) {
    const range   = sel.getRangeAt(0);
    const node    = range.startContainer;
    const element = node.nodeType === Node.ELEMENT_NODE ? node : node.parentElement;

    if (element) {
      const align = window.getComputedStyle(element).textAlign;
      if (align === 'right' || align === 'end') {
        // Align toolbar’s right edge with the right edge of the selection.
        left = targetRect.right - floatingRect.width + horizontalOffset;
      }
    }
  }

  if (left + floatingRect.width > scrollerRect.right) {
    left = scrollerRect.right - floatingRect.width - horizontalOffset;
  }

  // Make sure the toolbar never goes outside the browser window
  const viewportRight = window.innerWidth;
  const viewportLeft  = 0;

  if (left + floatingRect.width > viewportRight) {
    left = viewportRight - floatingRect.width - horizontalOffset;
  }
  if (left < viewportLeft) {
    left = viewportLeft + horizontalOffset;
  }

  top  = top - anchorRect.top + scrollerElem.scrollTop;
  left = left - anchorRect.left + scrollerElem.scrollLeft;

  floatingElem.style.opacity = '1';
  floatingElem.style.transform = `translate(${left}px, ${top}px)`;
}

function getDOMRangeRect(nativeSelection, rootElement) {
  const domRange = nativeSelection.getRangeAt(0);
  let rect;
  if (nativeSelection.anchorNode === rootElement) {
    let inner = rootElement;
    while (inner.firstElementChild != null) inner = inner.firstElementChild;
    rect = inner.getBoundingClientRect();
  } else {
    rect = domRange.getBoundingClientRect();
  }
  return rect;
}

function appendToLexicalEditor(editor, textChunk) {
  editor.update(() => {

    //const selection = $getSelection();

    // ✅ Ensure we have a real cursor/text selection
    //if ($isRangeSelection(selection)) {
      // This automatically inserts at the cursor
    //  selection.insertText(textChunk);
    //  return;
    //}

    const root = $getRoot();

    // Find the last child; if there isn't a paragraph, create one
    let lastNode = root.getLastChild();
    if (!lastNode || lastNode.getType() !== "paragraph") {
      lastNode = $createParagraphNode();
      root.append(lastNode);
    }

    // Append text into that paragraph
    const textNode = $createTextNode(textChunk);
    lastNode.append(textNode);
  });
}

function replaceLexicalEditorContent(editor, newContent) {
  editor.update(() => {
    const root = $getRoot();
    root.clear();
  
    const paragraph = $createParagraphNode();
    paragraph.append($createTextNode(newContent));
    root.append(paragraph);
  });
}

window.appendToLexicalEditor = appendToLexicalEditor;
window.createMarkdownEditor = createMarkdownEditor;
window.replaceLexicalEditorContent = replaceLexicalEditorContent;
window.getEditorForComment = getEditorForComment;