
var hljs = new function () {

    /* Utility functions */
  
    function escape(value) {
      return value.replace(/&/gm, '&amp;').replace(/</gm, '&lt;').replace(/>/gm, '&gt;');
    }
  
    function tag(node) {
      return node.nodeName.toLowerCase();
    }
  
    function testRe(re, lexeme) {
      var match = re && re.exec(lexeme);
      return match && match.index == 0;
    }
  
    function blockLanguage(block) {
      var classes = (block.className + ' ' + (block.parentNode ? block.parentNode.className : '')).split(/\s+/);
      classes = classes.map(function (c) { return c.replace(/^lang(uage)?-/, ''); });
      return classes.filter(function (c) { return getLanguage(c) || c == 'no-highlight'; })[0];
    }
  
    function inherit(parent, obj) {
      var result = {};
      for (var key in parent)
        result[key] = parent[key];
      if (obj)
        for (var key in obj)
          result[key] = obj[key];
      return result;
    };
  
    /* Stream merging */
  
    function nodeStream(node) {
      var result = [];
      (function _nodeStream(node, offset) {
        for (var child = node.firstChild; child; child = child.nextSibling) {
          if (child.nodeType == 3)
            offset += child.nodeValue.length;
          else if (tag(child) == 'br')
            offset += 1;
          else if (child.nodeType == 1) {
            result.push({
              event: 'start',
              offset: offset,
              node: child
            });
            offset = _nodeStream(child, offset);
            result.push({
              event: 'stop',
              offset: offset,
              node: child
            });
          }
        }
        return offset;
      })(node, 0);
      return result;
    }
  
    function mergeStreams(original, highlighted, value) {
      var processed = 0;
      var result = '';
      var nodeStack = [];
  
      function selectStream() {
        if (!original.length || !highlighted.length) {
          return original.length ? original : highlighted;
        }
        if (original[0].offset != highlighted[0].offset) {
          return (original[0].offset < highlighted[0].offset) ? original : highlighted;
        }
  
        /*
        To avoid starting the stream just before it should stop the order is
        ensured that original always starts first and closes last:
  
        if (event1 == 'start' && event2 == 'start')
          return original;
        if (event1 == 'start' && event2 == 'stop')
          return highlighted;
        if (event1 == 'stop' && event2 == 'start')
          return original;
        if (event1 == 'stop' && event2 == 'stop')
          return highlighted;
  
        ... which is collapsed to:
        */
        return highlighted[0].event == 'start' ? original : highlighted;
      }
  
      function open(node) {
        function attr_str(a) { return ' ' + a.nodeName + '="' + escape(a.value) + '"'; }
        result += '<' + tag(node) + Array.prototype.map.call(node.attributes, attr_str).join('') + '>';
      }
  
      function close(node) {
        result += '</' + tag(node) + '>';
      }
  
      function render(event) {
        (event.event == 'start' ? open : close)(event.node);
      }
  
      while (original.length || highlighted.length) {
        var stream = selectStream();
        result += escape(value.substr(processed, stream[0].offset - processed));
        processed = stream[0].offset;
        if (stream == original) {
          /*
          On any opening or closing tag of the original markup we first close
          the entire highlighted node stack, then render the original tag along
          with all the following original tags at the same offset and then
          reopen all the tags on the highlighted stack.
          */
          nodeStack.reverse().forEach(close);
          do {
            render(stream.splice(0, 1)[0]);
            stream = selectStream();
          } while (stream == original && stream.length && stream[0].offset == processed);
          nodeStack.reverse().forEach(open);
        } else {
          if (stream[0].event == 'start') {
            nodeStack.push(stream[0].node);
          } else {
            nodeStack.pop();
          }
          render(stream.splice(0, 1)[0]);
        }
      }
      return result + escape(value.substr(processed));
    }
  
    /* Initialization */
  
    function compileLanguage(language) {
  
      function reStr(re) {
        return (re && re.source) || re;
      }
  
      function langRe(value, global) {
        return RegExp(
          reStr(value),
          'm' + (language.case_insensitive ? 'i' : '') + (global ? 'g' : '')
        );
      }
  
      function compileMode(mode, parent) {
        if (mode.compiled)
          return;
        mode.compiled = true;
  
        mode.keywords = mode.keywords || mode.beginKeywords;
        if (mode.keywords) {
          var compiled_keywords = {};
  
          var flatten = function (className, str) {
            if (language.case_insensitive) {
              str = str.toLowerCase();
            }
            str.split(' ').forEach(function (kw) {
              var pair = kw.split('|');
              compiled_keywords[pair[0]] = [className, pair[1] ? Number(pair[1]) : 1];
            });
          };
  
          if (typeof mode.keywords == 'string') { // string
            flatten('keyword', mode.keywords);
          } else {
            Object.keys(mode.keywords).forEach(function (className) {
              flatten(className, mode.keywords[className]);
            });
          }
          mode.keywords = compiled_keywords;
        }
        mode.lexemesRe = langRe(mode.lexemes || /\b[A-Za-z0-9_]+\b/, true);
  
        if (parent) {
          if (mode.beginKeywords) {
            mode.begin = '\\b(' + mode.beginKeywords.split(' ').join('|') + ')\\b';
          }
          if (!mode.begin)
            mode.begin = /\B|\b/;
          mode.beginRe = langRe(mode.begin);
          if (!mode.end && !mode.endsWithParent)
            mode.end = /\B|\b/;
          if (mode.end)
            mode.endRe = langRe(mode.end);
          mode.terminator_end = reStr(mode.end) || '';
          if (mode.endsWithParent && parent.terminator_end)
            mode.terminator_end += (mode.end ? '|' : '') + parent.terminator_end;
        }
        if (mode.illegal)
          mode.illegalRe = langRe(mode.illegal);
        if (mode.relevance === undefined)
          mode.relevance = 1;
        if (!mode.contains) {
          mode.contains = [];
        }
        var expanded_contains = [];
        mode.contains.forEach(function (c) {
          if (c.variants) {
            c.variants.forEach(function (v) { expanded_contains.push(inherit(c, v)); });
          } else {
            expanded_contains.push(c == 'self' ? mode : c);
          }
        });
        mode.contains = expanded_contains;
        mode.contains.forEach(function (c) { compileMode(c, mode); });
  
        if (mode.starts) {
          compileMode(mode.starts, parent);
        }
  
        var terminators =
          mode.contains.map(function (c) {
            return c.beginKeywords ? '\\.?(' + c.begin + ')\\.?' : c.begin;
          })
            .concat([mode.terminator_end, mode.illegal])
            .map(reStr)
            .filter(Boolean);
        mode.terminators = terminators.length ? langRe(terminators.join('|'), true) : { exec: function (s) { return null; } };
  
        mode.continuation = {};
      }
  
      compileMode(language);
    }
  
    /*
    Core highlighting function. Accepts a language name, or an alias, and a
    string with the code to highlight. Returns an object with the following
    properties:
  
    - relevance (int)
    - value (an HTML string with highlighting markup)
  
    */
    function highlight(name, value, ignore_illegals, continuation) {
  
      function subMode(lexeme, mode) {
        for (var i = 0; i < mode.contains.length; i++) {
          if (testRe(mode.contains[i].beginRe, lexeme)) {
            return mode.contains[i];
          }
        }
      }
  
      function endOfMode(mode, lexeme) {
        if (testRe(mode.endRe, lexeme)) {
          return mode;
        }
        if (mode.endsWithParent) {
          return endOfMode(mode.parent, lexeme);
        }
      }
  
      function isIllegal(lexeme, mode) {
        return !ignore_illegals && testRe(mode.illegalRe, lexeme);
      }
  
      function keywordMatch(mode, match) {
        var match_str = language.case_insensitive ? match[0].toLowerCase() : match[0];
        return mode.keywords.hasOwnProperty(match_str) && mode.keywords[match_str];
      }
  
      function buildSpan(classname, insideSpan, leaveOpen, noPrefix) {
        var classPrefix = noPrefix ? '' : options.classPrefix,
          openSpan = '<span class="' + classPrefix,
          closeSpan = leaveOpen ? '' : '</span>';
  
        openSpan += classname + '">';
  
        return openSpan + insideSpan + closeSpan;
      }
  
      function processKeywords() {
        if (!top.keywords)
          return escape(mode_buffer);
        var result = '';
        var last_index = 0;
        top.lexemesRe.lastIndex = 0;
        var match = top.lexemesRe.exec(mode_buffer);
        while (match) {
          result += escape(mode_buffer.substr(last_index, match.index - last_index));
          var keyword_match = keywordMatch(top, match);
          if (keyword_match) {
            relevance += keyword_match[1];
            result += buildSpan(keyword_match[0], escape(match[0]));
          } else {
            result += escape(match[0]);
          }
          last_index = top.lexemesRe.lastIndex;
          match = top.lexemesRe.exec(mode_buffer);
        }
        return result + escape(mode_buffer.substr(last_index));
      }
  
      function processSubLanguage() {
        if (top.subLanguage && !languages[top.subLanguage]) {
          return escape(mode_buffer);
        }
        var result = top.subLanguage ? highlight(top.subLanguage, mode_buffer, true, top.continuation.top) : highlightAuto(mode_buffer);
        // Counting embedded language score towards the host language may be disabled
        // with zeroing the containing mode relevance. Usecase in point is Markdown that
        // allows XML everywhere and makes every XML snippet to have a much larger Markdown
        // score.
        if (top.relevance > 0) {
          relevance += result.relevance;
        }
        if (top.subLanguageMode == 'continuous') {
          top.continuation.top = result.top;
        }
        return buildSpan(result.language, result.value, false, true);
      }
  
      function processBuffer() {
        return top.subLanguage !== undefined ? processSubLanguage() : processKeywords();
      }
  
      function startNewMode(mode, lexeme) {
        var markup = mode.className ? buildSpan(mode.className, '', true) : '';
        if (mode.returnBegin) {
          result += markup;
          mode_buffer = '';
        } else if (mode.excludeBegin) {
          result += escape(lexeme) + markup;
          mode_buffer = '';
        } else {
          result += markup;
          mode_buffer = lexeme;
        }
        top = Object.create(mode, { parent: { value: top } });
      }
  
      function processLexeme(buffer, lexeme) {
  
        mode_buffer += buffer;
        if (lexeme === undefined) {
          result += processBuffer();
          return 0;
        }
  
        var new_mode = subMode(lexeme, top);
        if (new_mode) {
          result += processBuffer();
          startNewMode(new_mode, lexeme);
          return new_mode.returnBegin ? 0 : lexeme.length;
        }
  
        var end_mode = endOfMode(top, lexeme);
        if (end_mode) {
          var origin = top;
          if (!(origin.returnEnd || origin.excludeEnd)) {
            mode_buffer += lexeme;
          }
          result += processBuffer();
          do {
            if (top.className) {
              result += '</span>';
            }
            relevance += top.relevance;
            top = top.parent;
          } while (top != end_mode.parent);
          if (origin.excludeEnd) {
            result += escape(lexeme);
          }
          mode_buffer = '';
          if (end_mode.starts) {
            startNewMode(end_mode.starts, '');
          }
          return origin.returnEnd ? 0 : lexeme.length;
        }
  
        if (isIllegal(lexeme, top))
          throw new Error('Illegal lexeme "' + lexeme + '" for mode "' + (top.className || '<unnamed>') + '"');
  
        /*
        Parser should not reach this point as all types of lexemes should be caught
        earlier, but if it does due to some bug make sure it advances at least one
        character forward to prevent infinite looping.
        */
        mode_buffer += lexeme;
        return lexeme.length || 1;
      }
  
      var language = getLanguage(name);
      if (!language) {
        throw new Error('Unknown language: "' + name + '"');
      }
  
      compileLanguage(language);
      var top = continuation || language;
      var result = '';
      for (var current = top; current != language; current = current.parent) {
        if (current.className) {
          result += buildSpan(current.className, result, true);
        }
      }
      var mode_buffer = '';
      var relevance = 0;
      try {
        var match, count, index = 0;
        while (true) {
          top.terminators.lastIndex = index;
          match = top.terminators.exec(value);
          if (!match)
            break;
          count = processLexeme(value.substr(index, match.index - index), match[0]);
          index = match.index + count;
        }
        processLexeme(value.substr(index));
        for (var current = top; current.parent; current = current.parent) { // close dangling modes
          if (current.className) {
            result += '</span>';
          }
        };
        return {
          relevance: relevance,
          value: result,
          language: name,
          top: top
        };
      } catch (e) {
        if (e.message.indexOf('Illegal') != -1) {
          return {
            relevance: 0,
            value: escape(value)
          };
        } else {
          throw e;
        }
      }
    }
  
    /*
    Highlighting with language detection. Accepts a string with the code to
    highlight. Returns an object with the following properties:
  
    - language (detected language)
    - relevance (int)
    - value (an HTML string with highlighting markup)
    - second_best (object with the same structure for second-best heuristically
      detected language, may be absent)
  
    */
    function highlightAuto(text, languageSubset) {
      languageSubset = languageSubset || options.languages || Object.keys(languages);
      var result = {
        relevance: 0,
        value: escape(text)
      };
      var second_best = result;
      languageSubset.forEach(function (name) {
        if (!getLanguage(name)) {
          return;
        }
        var current = highlight(name, text, false);
        current.language = name;
        if (current.relevance > second_best.relevance) {
          second_best = current;
        }
        if (current.relevance > result.relevance) {
          second_best = result;
          result = current;
        }
      });
      if (second_best.language) {
        result.second_best = second_best;
      }
      return result;
    }
  
    /*
    Post-processing of the highlighted markup:
  
    - replace TABs with something more useful
    - replace real line-breaks with '<br>' for non-pre containers
  
    */
    function fixMarkup(value) {
      if (options.tabReplace) {
        value = value.replace(/^((<[^>]+>|\t)+)/gm, function (match, p1, offset, s) {
          return p1.replace(/\t/g, options.tabReplace);
        });
      }
      if (options.useBR) {
        value = value.replace(/\n/g, '<br>');
      }
      return value;
    }
  
    /*
    Applies highlighting to a DOM node containing code. Accepts a DOM node and
    two optional parameters for fixMarkup.
    */
    function highlightBlock(block) {
      var text = options.useBR ? block.innerHTML
        .replace(/\n/g, '').replace(/<br>|<br [^>]*>/g, '\n').replace(/<[^>]*>/g, '')
        : block.textContent;
      var language = blockLanguage(block);
      if (language == 'no-highlight')
        return;
      var result = language ? highlight(language, text, true) : highlightAuto(text);
      var original = nodeStream(block);
      if (original.length) {
        var pre = document.createElementNS('http://www.w3.org/1999/xhtml', 'pre');
        pre.innerHTML = result.value;
        result.value = mergeStreams(original, nodeStream(pre), text);
      }
      result.value = fixMarkup(result.value);
  
      block.innerHTML = result.value;
      block.className += ' hljs ' + (!language && result.language || '');
      block.result = {
        language: result.language,
        re: result.relevance
      };
      if (result.second_best) {
        block.second_best = {
          language: result.second_best.language,
          re: result.second_best.relevance
        };
      }
    }
  
    var options = {
      classPrefix: 'hljs-',
      tabReplace: null,
      useBR: false,
      languages: undefined
    };
  
    /*
    Updates highlight.js global options with values passed in the form of an object
    */
    function configure(user_options) {
      options = inherit(options, user_options);
    }
  
    /*
    Applies highlighting to all <pre><code>..</code></pre> blocks on a page.
    */
    function initHighlighting() {
      if (initHighlighting.called)
        return;
      initHighlighting.called = true;
  
      var blocks = document.querySelectorAll('pre code:not(.comment code)');
      Array.prototype.forEach.call(blocks, highlightBlock);
    }
  
    /*
    Attaches highlighting to the page load event.
    */
    function initHighlightingOnLoad() {
      addEventListener('DOMContentLoaded', initHighlighting, false);
      addEventListener('load', initHighlighting, false);
    }
  
    var languages = {};
    var aliases = {};
  
    function registerLanguage(name, language) {
      var lang = languages[name] = language(this);
      if (lang.aliases) {
        lang.aliases.forEach(function (alias) { aliases[alias] = name; });
      }
    }
  
    function listLanguages() {
      return Object.keys(languages);
    }
  
    function getLanguage(name) {
      return languages[name] || languages[aliases[name]];
    }
  
    /* Interface definition */
  
    this.highlight = highlight;
    this.highlightAuto = highlightAuto;
    this.fixMarkup = fixMarkup;
    this.highlightBlock = highlightBlock;
    this.configure = configure;
    this.initHighlighting = initHighlighting;
    this.initHighlightingOnLoad = initHighlightingOnLoad;
    this.registerLanguage = registerLanguage;
    this.listLanguages = listLanguages;
    this.getLanguage = getLanguage;
    this.inherit = inherit;
  
    // Common regexps
    this.IDENT_RE = '[a-zA-Z][a-zA-Z0-9_]*';
    this.UNDERSCORE_IDENT_RE = '[a-zA-Z_][a-zA-Z0-9_]*';
    this.NUMBER_RE = '\\b\\d+(\\.\\d+)?';
    this.C_NUMBER_RE = '(\\b0[xX][a-fA-F0-9]+|(\\b\\d+(\\.\\d*)?|\\.\\d+)([eE][-+]?\\d+)?)'; // 0x..., 0..., decimal, float
    this.BINARY_NUMBER_RE = '\\b(0b[01]+)'; // 0b...
    this.RE_STARTERS_RE = '!|!=|!==|%|%=|&|&&|&=|\\*|\\*=|\\+|\\+=|,|-|-=|/=|/|:|;|<<|<<=|<=|<|===|==|=|>>>=|>>=|>=|>>>|>>|>|\\?|\\[|\\{|\\(|\\^|\\^=|\\||\\|=|\\|\\||~';
  
    // Common modes
    this.BACKSLASH_ESCAPE = {
      begin: '\\\\[\\s\\S]', relevance: 0
    };
    this.APOS_STRING_MODE = {
      className: 'string',
      begin: '\'', end: '\'',
      illegal: '\\n',
      contains: [this.BACKSLASH_ESCAPE]
    };
    this.QUOTE_STRING_MODE = {
      className: 'string',
      begin: '"', end: '"',
      illegal: '\\n',
      contains: [this.BACKSLASH_ESCAPE]
    };
    this.PHRASAL_WORDS_MODE = {
      begin: /\b(a|an|the|are|I|I'm|isn't|don't|doesn't|won't|but|just|should|pretty|simply|enough|gonna|going|wtf|so|such)\b/
    };
    this.C_LINE_COMMENT_MODE = {
      className: 'comment',
      begin: '//', end: '$',
      contains: [this.PHRASAL_WORDS_MODE]
    };
    this.C_BLOCK_COMMENT_MODE = {
      className: 'comment',
      begin: '/\\*', end: '\\*/',
      contains: [this.PHRASAL_WORDS_MODE]
    };
    this.HASH_COMMENT_MODE = {
      className: 'comment',
      begin: '#', end: '$',
      contains: [this.PHRASAL_WORDS_MODE]
    };
    this.NUMBER_MODE = {
      className: 'number',
      begin: this.NUMBER_RE,
      relevance: 0
    };
    this.C_NUMBER_MODE = {
      className: 'number',
      begin: this.C_NUMBER_RE,
      relevance: 0
    };
    this.BINARY_NUMBER_MODE = {
      className: 'number',
      begin: this.BINARY_NUMBER_RE,
      relevance: 0
    };
    this.CSS_NUMBER_MODE = {
      className: 'number',
      begin: this.NUMBER_RE + '(' +
        '%|em|ex|ch|rem' +
        '|vw|vh|vmin|vmax' +
        '|cm|mm|in|pt|pc|px' +
        '|deg|grad|rad|turn' +
        '|s|ms' +
        '|Hz|kHz' +
        '|dpi|dpcm|dppx' +
        ')?',
      relevance: 0
    };
    this.REGEXP_MODE = {
      className: 'regexp',
      begin: /\//, end: /\/[gim]*/,
      illegal: /\n/,
      contains: [
        this.BACKSLASH_ESCAPE,
        {
          begin: /\[/, end: /\]/,
          relevance: 0,
          contains: [this.BACKSLASH_ESCAPE]
        }
      ]
    };
    this.TITLE_MODE = {
      className: 'title',
      begin: this.IDENT_RE,
      relevance: 0
    };
    this.UNDERSCORE_TITLE_MODE = {
      className: 'title',
      begin: this.UNDERSCORE_IDENT_RE,
      relevance: 0
    };
  }();
  
  hljs.registerLanguage('cpp', function (hljs) {
    var CPP_KEYWORDS = {
      keyword: 'false int float while private char catch export virtual operator sizeof ' +
        'dynamic_cast|10 typedef const_cast|10 const struct for static_cast|10 union namespace ' +
        'unsigned long throw volatile static protected bool template mutable if public friend ' +
        'do return goto auto void enum else break new extern using true class asm case typeid ' +
        'short reinterpret_cast|10 default double register explicit signed typename try this ' +
        'switch continue wchar_t inline delete alignof char16_t char32_t constexpr decltype ' +
        'noexcept nullptr static_assert thread_local restrict _Bool complex _Complex _Imaginary',
      built_in: 'std string cin cout cerr clog stringstream istringstream ostringstream ' +
        'auto_ptr deque list queue stack vector map set bitset multiset multimap unordered_set ' +
        'unordered_map unordered_multiset unordered_multimap array shared_ptr abort abs acos ' +
        'asin atan2 atan calloc ceil cosh cos exit exp fabs floor fmod fprintf fputs free frexp ' +
        'fscanf isalnum isalpha iscntrl isdigit isgraph islower isprint ispunct isspace isupper ' +
        'isxdigit tolower toupper labs ldexp log10 log malloc memchr memcmp memcpy memset modf pow ' +
        'printf putchar puts scanf sinh sin snprintf sprintf sqrt sscanf strcat strchr strcmp ' +
        'strcpy strcspn strlen strncat strncmp strncpy strpbrk strrchr strspn strstr tanh tan ' +
        'vfprintf vprintf vsprintf'
    };
    return {
      aliases: ['c', 'h', 'c++', 'h++'],
      keywords: CPP_KEYWORDS,
      illegal: '</',
      contains: [
        hljs.C_LINE_COMMENT_MODE,
        hljs.C_BLOCK_COMMENT_MODE,
        hljs.QUOTE_STRING_MODE,
        {
          className: 'string',
          begin: '\'\\\\?.', end: '\'',
          illegal: '.'
        },
        {
          className: 'number',
          begin: '\\b(\\d+(\\.\\d*)?|\\.\\d+)(u|U|l|L|ul|UL|f|F)'
        },
        hljs.C_NUMBER_MODE,
        {
          className: 'preprocessor',
          begin: '#', end: '$',
          keywords: 'if else elif endif define undef warning error line pragma',
          contains: [
            {
              begin: 'include\\s*[<"]', end: '[>"]',
              keywords: 'include',
              illegal: '\\n'
            },
            hljs.C_LINE_COMMENT_MODE
          ]
        },
        {
          className: 'stl_container',
          begin: '\\b(deque|list|queue|stack|vector|map|set|bitset|multiset|multimap|unordered_map|unordered_set|unordered_multiset|unordered_multimap|array)\\s*<', end: '>',
          keywords: CPP_KEYWORDS,
          contains: ['self']
        },
        {
          begin: hljs.IDENT_RE + '::'
        }
      ]
    };
  });
  
  hljs.registerLanguage('SystemVerilog', function (hljs) {
    return {
      case_insensitive: true,
      keywords: {
        keyword:
          'alias always always_comb always_ff always_latch and assert assign assume automatic before ' +
          'begin bind bins binsof bit break buf bufif0 bufif1 byte case ' +
          'casex casez cell chandle class clocking cmos config const constraint context ' +
          'continue cover covergroup coverpoint cross deassign default defparam design disable dist ' +
          'do edge else end endcase endclass endclocking endconfig endfunction endgenerate endgroup ' +
          'endinterface endmodule endpackage endprimitive endprogram endproperty endspecify endsequence endtable endtask enum ' +
          'event expect export extends extern final first_match for force foreach forever ' +
          'fork forkjoin function generate genvar highz0 highz1 if iff ifnone ignore_bins ' +
          'illegal_bins import incdir include initial inout input inside instance int integer ' +
          'interface intersect join join_any join_none large liblist library local localparam logic ' +
          'longint macromodule matches medium modport module nand negedge new nmos nor ' +
          'noshowcancelled not notif0 notif1 null or output package packed parameter pmos ' +
          'posedge primitive priority program property protected pull0 pull1 pulldown pullup pulsestyle_onevent ' +
          'pulsestyle_ondetect pure rand randc randcase randsequence rcmos real realtime ref reg ' +
          'release repeat return rnmos rpmos rtran rtranif0 rtranif1 scalared sequence shortint ' +
          'shortreal showcancelled signed small solve specify specparam static string strong0 strong1 ' +
          'struct super supply0 supply1 table tagged task this throughout time timeprecision ' +
          'timeunit tran tranif0 tranif1 tri tri0 tri1 triand trior trireg type ' +
          'typedef union unique unsigned use var vectored virtual void wait wait_order ' +
          'wand weak0 weak1 while wildcard wire with within wor xnor xor'
      },
      illegal: '{',
      contains: [
        hljs.C_BLOCK_COMMENT_MODE,
        {
          className: 'comment',
          begin: '--', end: '$'
        },
        hljs.C_BLOCK_COMMENT_MODE,
        {
          className: 'comment',
          begin: '//', end: '$'
        },
        hljs.QUOTE_STRING_MODE,
        hljs.C_NUMBER_MODE,
        {
          className: 'literal',
          begin: '\'(U|X|0|1|Z|W|L|H|-)\'',
          contains: [hljs.BACKSLASH_ESCAPE]
        },
        {
          className: 'attribute',
          begin: '\'[A-Za-z](_?[A-Za-z0-9])*',
          contains: [hljs.BACKSLASH_ESCAPE]
        }
      ]
    }; // return
  });
  
  hljs.registerLanguage('bash', function (hljs) {
    var VAR = {
      className: 'variable',
      variants: [
        { begin: /\$[\w\d#@][\w\d_]*/ },
        { begin: /\$\{(.*?)\}/ }
      ]
    };
    var QUOTE_STRING = {
      className: 'string',
      begin: /"/, end: /"/,
      contains: [
        hljs.BACKSLASH_ESCAPE,
        VAR,
        {
          className: 'variable',
          begin: /\$\(/, end: /\)/,
          contains: [hljs.BACKSLASH_ESCAPE]
        }
      ]
    };
    var APOS_STRING = {
      className: 'string',
      begin: /'/, end: /'/
    };
  
    return {
      aliases: ['sh', 'zsh'],
      lexemes: /-?[a-z\.]+/,
      keywords: {
        keyword:
          'if then else elif fi for break continue while in do done exit return set ' +
          'declare case esac export exec',
        literal:
          'true false',
        built_in:
          'printf echo read cd pwd pushd popd dirs let eval unset typeset readonly ' +
          'getopts source shopt caller type hash bind help sudo',
        operator:
          '-ne -eq -lt -gt -f -d -e -s -l -a' // relevance booster
      },
      contains: [
        {
          className: 'shebang',
          begin: /^#![^\n]+sh\s*$/,
          relevance: 10
        },
        {
          className: 'function',
          begin: /\w[\w\d_]*\s*\(\s*\)\s*\{/,
          returnBegin: true,
          contains: [hljs.inherit(hljs.TITLE_MODE, { begin: /\w[\w\d_]*/ })],
          relevance: 0
        },
        hljs.HASH_COMMENT_MODE,
        hljs.NUMBER_MODE,
        QUOTE_STRING,
        APOS_STRING,
        VAR
      ]
    };
  });
  
  hljs.registerLanguage('eLanguage', function (hljs) {
    return {
      case_insensitive: true,
      keywords: {
        keyword:
          'all of and as_a assert assume attribute before break export case ' +
          'change check compute computed consume continue cover cross cycle default define ' +
          'delay detach do down to each edges else emit event exec ' +
          'expect extend fail fall for force from gen global if ifdef ' +
          'ifndef index is also c first inline instance not a an ' +
          'empty only undefined item keep keeping key like line list matching ' +
          'me nand new nor in now on or pass prev print ' +
          'range ranges release repeat return reverse rise routine select soft start ' +
          'state machine step struct sync sys that then transition true try ' +
          'type unit until using var when while with bits generate wait ' +
          'kill port simple_port event_port method_port import',
        type:
          'int bit bool byte file string time uint'
      },
      illegal: '{',
      contains: [
        hljs.C_BLOCK_COMMENT_MODE,
        {
          className: 'comment',
          begin: '--', end: '$'
        },
        hljs.C_BLOCK_COMMENT_MODE,
        {
          className: 'comment',
          begin: '//', end: '$'
        },
        hljs.QUOTE_STRING_MODE,
        hljs.C_NUMBER_MODE,
        {
          className: 'literal',
          begin: '\'(U|X|0|1|Z|W|L|H|-)\'',
          contains: [hljs.BACKSLASH_ESCAPE]
        },
        {
          className: 'attribute',
          begin: '\'[A-Za-z](_?[A-Za-z0-9])*',
          contains: [hljs.BACKSLASH_ESCAPE]
        }
      ]
    }; // return
  });
  
  hljs.registerLanguage('vhdl', function (hljs) {
    return {
      case_insensitive: true,
      keywords: {
        keyword:
          'abs access after alias all and architecture array assert attribute begin block ' +
          'body buffer bus case component configuration constant context cover disconnect ' +
          'downto default else elsif end entity exit fairness file for force function generate ' +
          'generic group guarded if impure in inertial inout is label library linkage literal ' +
          'loop map mod nand new next nor not null of on open or others out package port ' +
          'postponed procedure process property protected pure range record register reject ' +
          'release rem report restrict restrict_guarantee return rol ror select sequence ' +
          'severity shared signal sla sll sra srl strong subtype then to transport type ' +
          'unaffected units until use variable vmode vprop vunit wait when while with xnor xor',
        typename:
          'boolean bit character severity_level integer time delay_length natural positive ' +
          'string bit_vector file_open_kind file_open_status std_ulogic std_ulogic_vector ' +
          'std_logic std_logic_vector unsigned signed boolean_vector integer_vector ' +
          'real_vector time_vector'
      },
      illegal: '{',
      contains: [
        hljs.C_BLOCK_COMMENT_MODE,        // VHDL-2008 block commenting.
        {
          className: 'comment',
          begin: '--', end: '$'
        },
        hljs.QUOTE_STRING_MODE,
        hljs.C_NUMBER_MODE,
        {
          className: 'literal',
          begin: '\'(U|X|0|1|Z|W|L|H|-)\'',
          contains: [hljs.BACKSLASH_ESCAPE]
        },
        {
          className: 'attribute',
          begin: '\'[A-Za-z](_?[A-Za-z0-9])*',
          contains: [hljs.BACKSLASH_ESCAPE]
        }
      ]
    }; // return
  });
  
  hljs.registerLanguage('matlab', function (hljs) {
    var COMMON_CONTAINS = [
      hljs.C_NUMBER_MODE,
      {
        className: 'string',
        begin: '\'', end: '\'',
        contains: [hljs.BACKSLASH_ESCAPE, { begin: '\'\'' }]
      }
    ];
  
    return {
      keywords: {
        keyword:
          'break case catch classdef continue else elseif end enumerated events for function ' +
          'global if methods otherwise parfor persistent properties return spmd switch try while',
        built_in:
          'sin sind sinh asin asind asinh cos cosd cosh acos acosd acosh tan tand tanh atan ' +
          'atand atan2 atanh sec secd sech asec asecd asech csc cscd csch acsc acscd acsch cot ' +
          'cotd coth acot acotd acoth hypot exp expm1 log log1p log10 log2 pow2 realpow reallog ' +
          'realsqrt sqrt nthroot nextpow2 abs angle complex conj imag real unwrap isreal ' +
          'cplxpair fix floor ceil round mod rem sign airy besselj bessely besselh besseli ' +
          'besselk beta betainc betaln ellipj ellipke erf erfc erfcx erfinv expint gamma ' +
          'gammainc gammaln psi legendre cross dot factor isprime primes gcd lcm rat rats perms ' +
          'nchoosek factorial cart2sph cart2pol pol2cart sph2cart hsv2rgb rgb2hsv zeros ones ' +
          'eye repmat rand randn linspace logspace freqspace meshgrid accumarray size length ' +
          'ndims numel disp isempty isequal isequalwithequalnans cat reshape diag blkdiag tril ' +
          'triu fliplr flipud flipdim rot90 find sub2ind ind2sub bsxfun ndgrid permute ipermute ' +
          'shiftdim circshift squeeze isscalar isvector ans eps realmax realmin pi i inf nan ' +
          'isnan isinf isfinite j why compan gallery hadamard hankel hilb invhilb magic pascal ' +
          'rosser toeplitz vander wilkinson'
      },
      illegal: '(//|"|#|/\\*|\\s+/\\w+)',
      contains: [
        {
          className: 'function',
          beginKeywords: 'function', end: '$',
          contains: [
            hljs.UNDERSCORE_TITLE_MODE,
            {
              className: 'params',
              begin: '\\(', end: '\\)'
            },
            {
              className: 'params',
              begin: '\\[', end: '\\]'
            }
          ]
        },
        {
          className: 'transposed_variable',
          begin: '[a-zA-Z_][a-zA-Z_0-9]*(\'+[\\.\']*|[\\.\']+)', end: '',
          relevance: 0
        },
        {
          className: 'matrix',
          begin: '\\[', end: '\\]\'*[\\.\']*',
          contains: COMMON_CONTAINS,
          relevance: 0
        },
        {
          className: 'cell',
          begin: '\\{',
          contains: COMMON_CONTAINS,
          illegal: /:/,
          variants: [
            { end: /\}'[\.']*/ },
            { end: /\}/, relevance: 0 }
          ]
        },
        {
          className: 'comment',
          begin: '\\%', end: '$'
        }
      ].concat(COMMON_CONTAINS)
    };
  });
  
  // jshint multistr:true
  
  (function (w, d) {
    'use strict';
  
    var TABLE_NAME = 'hljs-ln',
      LINE_NAME = 'hljs-ln-line',
      CODE_BLOCK_NAME = 'hljs-ln-code',
      NUMBERS_BLOCK_NAME = 'hljs-ln-numbers',
      NUMBER_LINE_NAME = 'hljs-ln-n',
      DATA_ATTR_NAME = 'data-line-number',
      BREAK_LINE_REGEXP = /\r\n|\r|\n/g;
  
    if (w.hljs) {
      w.hljs.initLineNumbersOnLoad = initLineNumbersOnLoad;
      w.hljs.lineNumbersBlock = lineNumbersBlock;
      w.hljs.lineNumbersValue = lineNumbersValue;
  
      addStyles();
    } else {
      w.console.error('highlight.js not detected!');
    }
  
    function isHljsLnCodeDescendant(domElt) {
      var curElt = domElt;
      while (curElt) {
        if (curElt.className && curElt.className.indexOf('hljs-ln-code') !== -1) {
          return true;
        }
        curElt = curElt.parentNode;
      }
      return false;
    }
  
    function getHljsLnTable(hljsLnDomElt) {
      var curElt = hljsLnDomElt;
      while (curElt.nodeName !== 'TABLE') {
        curElt = curElt.parentNode;
      }
      return curElt;
    }
  
    // Function to workaround a copy issue with Microsoft Edge.
    // Due to hljs-ln wrapping the lines of code inside a <table> element,
    // itself wrapped inside a <pre> element, window.getSelection().toString()
    // does not contain any line breaks. So we need to get them back using the
    // rendered code in the DOM as reference.
    function edgeGetSelectedCodeLines(selection) {
      // current selected text without line breaks
      var selectionText = selection.toString();
  
      // get the <td> element wrapping the first line of selected code
      var tdAnchor = selection.anchorNode;
      while (tdAnchor.nodeName !== 'TD') {
        tdAnchor = tdAnchor.parentNode;
      }
  
      // get the <td> element wrapping the last line of selected code
      var tdFocus = selection.focusNode;
      while (tdFocus.nodeName !== 'TD') {
        tdFocus = tdFocus.parentNode;
      }
  
      // extract line numbers
      var firstLineNumber = parseInt(tdAnchor.dataset.lineNumber);
      var lastLineNumber = parseInt(tdFocus.dataset.lineNumber);
  
      // multi-lines copied case
      if (firstLineNumber != lastLineNumber) {
  
        var firstLineText = tdAnchor.textContent;
        var lastLineText = tdFocus.textContent;
  
        // if the selection was made backward, swap values
        if (firstLineNumber > lastLineNumber) {
          var tmp = firstLineNumber;
          firstLineNumber = lastLineNumber;
          lastLineNumber = tmp;
          tmp = firstLineText;
          firstLineText = lastLineText;
          lastLineText = tmp;
        }
  
        // discard not copied characters in first line
        while (selectionText.indexOf(firstLineText) !== 0) {
          firstLineText = firstLineText.slice(1);
        }
  
        // discard not copied characters in last line
        while (selectionText.lastIndexOf(lastLineText) === -1) {
          lastLineText = lastLineText.slice(0, -1);
        }
  
        // reconstruct and return the real copied text
        var selectedText = firstLineText;
        var hljsLnTable = getHljsLnTable(tdAnchor);
        for (var i = firstLineNumber + 1; i < lastLineNumber; ++i) {
          var codeLineSel = format('.{0}[{1}="{2}"]', [CODE_BLOCK_NAME, DATA_ATTR_NAME, i]);
          var codeLineElt = hljsLnTable.querySelector(codeLineSel);
          selectedText += '\n' + codeLineElt.textContent;
        }
        selectedText += '\n' + lastLineText;
        return selectedText;
        // single copied line case
      } else {
        return selectionText;
      }
    }
  
    // ensure consistent code copy/paste behavior across all browsers
    // (see https://github.com/wcoder/highlightjs-line-numbers.js/issues/51)
    document.addEventListener('copy', function (e) {
      // get current selection
      var selection = window.getSelection();
      // override behavior when one wants to copy line of codes
      if (isHljsLnCodeDescendant(selection.anchorNode)) {
        var selectionText;
        // workaround an issue with Microsoft Edge as copied line breaks
        // are removed otherwise from the selection string
        if (window.navigator.userAgent.indexOf('Edge') !== -1) {
          selectionText = edgeGetSelectedCodeLines(selection);
        } else {
          // other browsers can directly use the selection string
          selectionText = selection.toString();
        }
        e.clipboardData.setData('text/plain', selectionText);
        e.preventDefault();
      }
    });
  
    function addStyles() {
      var css = d.createElement('style');
      css.type = 'text/css';
      css.innerHTML = format(
        '.{0}{border-collapse:collapse}' +
        '.{0} td{padding:0}' +
        '.{1}:before{content:attr({2})}',
        [
          TABLE_NAME,
          NUMBER_LINE_NAME,
          DATA_ATTR_NAME
        ]);
      d.getElementsByTagName('head')[0].appendChild(css);
    }
  
    function initLineNumbersOnLoad(options, callback) {
      if (d.readyState === 'interactive' || d.readyState === 'complete') {
        documentReady(options, callback);
      } else {
        w.addEventListener('DOMContentLoaded', function () {
          documentReady(options, callback);
        });
      }
    }
  
    function documentReady(options, callback) {
      try {
        var blocks = d.querySelectorAll('code.hljs,code.nohighlight');
  
        for (var i in blocks) {
          if (blocks.hasOwnProperty(i)) {
            if (!isPluginDisabledForBlock(blocks[i])) {
              lineNumbersBlock(blocks[i], options, callback);
            }
          }
        }
      } catch (e) {
        w.console.error('LineNumbers error: ', e);
      }
    }
  
    function isPluginDisabledForBlock(element) {
      return element.classList.contains('nohljsln');
    }
  
    function lineNumbersBlock(element, options, callback) {
      if (typeof element !== 'object') return;
  
      async(function () {
        element.innerHTML = lineNumbersInternal(element, options);
        callback();
      });
    }
  
    function lineNumbersValue(value, options) {
      if (typeof value !== 'string') return;
  
      var element = document.createElement('code')
      element.innerHTML = value
  
      return lineNumbersInternal(element, options);
    }
  
    function lineNumbersInternal(element, options) {
  
      var internalOptions = mapOptions(element, options);
  
      duplicateMultilineNodes(element);
  
      return addLineNumbersBlockFor(element.innerHTML, internalOptions);
    }
  
    function addLineNumbersBlockFor(inputHtml, options) {
      var lines = getLines(inputHtml);
  
      // if last line contains only carriage return remove it
      if (lines[lines.length - 1].trim() === '') {
        lines.pop();
      }
  
      if (lines.length > 1 || options.singleLine) {
        var html = '';
  
        for (var i = 0, l = lines.length; i < l; i++) {
          html += format(
            '<tr>' +
            '<td class="border {0} {1}" {3}="{5}" style="border-right: 1px solid #ccc; padding-right: 5px; color: #999;">' +
            '<div class="{2}" {3}="{5}"></div>' +
            '</td>' +
            '<td class="{0} {4}" {3}="{5}">' +
            '{6}' +
            '</td>' +
            '</tr>',
            [
              LINE_NAME,
              NUMBERS_BLOCK_NAME,
              NUMBER_LINE_NAME,
              DATA_ATTR_NAME,
              CODE_BLOCK_NAME,
              i + options.startFrom,
              lines[i].length > 0 ? lines[i] : ' '
            ]);
        }
  
        return format('<table class="{0}">{1}</table>', [TABLE_NAME, html]);
      }
  
      return inputHtml;
    }
  
    /**
     * @param {HTMLElement} element Code block.
     * @param {Object} options External API options.
     * @returns {Object} Internal API options.
     */
    function mapOptions(element, options) {
      options = options || {};
      return {
        singleLine: getSingleLineOption(options),
        startFrom: getStartFromOption(element, options)
      };
    }
  
    function getSingleLineOption(options) {
      var defaultValue = false;
      if (!!options.singleLine) {
        return options.singleLine;
      }
      return defaultValue;
    }
  
    function getStartFromOption(element, options) {
      var defaultValue = 1;
      var startFrom = defaultValue;
  
      if (isFinite(options.startFrom)) {
        startFrom = options.startFrom;
      }
  
      // can be overridden because local option is priority
      var value = getAttribute(element, 'data-ln-start-from');
      if (value !== null) {
        startFrom = toNumber(value, defaultValue);
      }
  
      return startFrom;
    }
  
    /**
     * Recursive method for fix multi-line elements implementation in highlight.js
     * Doing deep passage on child nodes.
     * @param {HTMLElement} element
     */
    function duplicateMultilineNodes(element) {
      var nodes = element.childNodes;
      for (var node in nodes) {
        if (nodes.hasOwnProperty(node)) {
          var child = nodes[node];
          if (getLinesCount(child.textContent) > 0) {
            if (child.childNodes.length > 0) {
              duplicateMultilineNodes(child);
            } else {
              duplicateMultilineNode(child.parentNode);
            }
          }
        }
      }
    }
  
    /**
     * Method for fix multi-line elements implementation in highlight.js
     * @param {HTMLElement} element
     */
    function duplicateMultilineNode(element) {
      var className = element.className;
  
      if (! /hljs-/.test(className)) return;
  
      var lines = getLines(element.innerHTML);
  
      for (var i = 0, result = ''; i < lines.length; i++) {
        var lineText = lines[i].length > 0 ? lines[i] : ' ';
        result += format('<span class="{0}">{1}</span>\n', [className, lineText]);
      }
  
      element.innerHTML = result.trim();
    }
  
    function getLines(text) {
      if (text.length === 0) return [];
      return text.split(BREAK_LINE_REGEXP);
    }
  
    function getLinesCount(text) {
      return (text.trim().match(BREAK_LINE_REGEXP) || []).length;
    }
  
    ///
    /// HELPERS
    ///
  
    function async(func) {
      w.setTimeout(func, 0);
    }
  
    /**
     * {@link https://wcoder.github.io/notes/string-format-for-string-formating-in-javascript}
     * @param {string} format
     * @param {array} args
     */
    function format(format, args) {
      return format.replace(/\{(\d+)\}/g, function (m, n) {
        return args[n] !== undefined ? args[n] : m;
      });
    }
  
    /**
     * @param {HTMLElement} element Code block.
     * @param {String} attrName Attribute name.
     * @returns {String} Attribute value or empty.
     */
    function getAttribute(element, attrName) {
      return element.hasAttribute(attrName) ? element.getAttribute(attrName) : null;
    }
  
    /**
     * @param {String} str Source string.
     * @param {Number} fallback Fallback value.
     * @returns Parsed number or fallback value.
     */
    function toNumber(str, fallback) {
      if (!str) return fallback;
      var number = Number(str);
      return isFinite(number) ? number : fallback;
    }
  
  }(window, document));
  
  hljs.initHighlightingOnLoad();
  hljs.initLineNumbersOnLoad("", function() {
    try {
      var c=window.location.search;
      if (c.length>0&&c.charAt(0)=="?") {
        var b=$(".hljs-ln-n[data-line-number=" + parseInt(c.substr(1,c.length-1)) + "]").css("border-radius","4px").css("background-color","red").css("color","white");
        $("html, body").animate({scrollTop:Math.max(0,b.offset().top-($(window).height()>0?$(window).height()/3:0))},250)
      }
    }
    catch(a){}
  });
  
  