"use strict";

const AI_TARGET_CONTAINER_CLASSNAME = 'comment-for-ai';
const AI_TARGET_RESULT_CLASSNAME = 'ai-generated';

const EDITOR_ACTIONS_BTN_CLASSNAME = 'ai-container-floating-actions'
const EDITOR_ACTIONS_GROUP_BTN_CLASSNAME = 'ai-container-floating-actions-group'

// This is dynamically loaded in the layout, we save it when the page loads
const AI_ASSISTANT_IMG_SRC = { src: "" };

const GENERATE_MISSING_COMMENT_HOVER = 'Generate missing comment';
const SAVE_COMMENTS_HOVER = 'Save comments';
const STOP_GENERATION_HOVER = 'Stop generating';
const REGENERATE_COMMENT_HOVER = 'Regenerate comment';
const DISCARD_COMMENT_HOVER = 'Discard changes';

// Enum for generate btn states
const GENERATE_BTN_STATE = Object.freeze({
    GENERATE:   "generate",
    STOP:  "stop",
    ACCEPT: "accept"
});
let generateBtnCpy;

/*
    Assuming the format of a button as:

    <a class=`${BTN_CLASSES}`>
      <span class=`${BNT_SPAN_CLASSES}`></span>
    </a> 
*/ 
const AI_ASSISTANT_BTN_SPAN_CLASSES = 'fa';
const AI_ASSISTANT_BTN_CLASSES = 'btn btn-neutral floating';

const GENERATE_BTN_SPAN_CLASSES = 'fa';
const GENERATE_BTN_CLASSES = 'btn btn-neutral';

const CHECK_BTN_SPAN_CLASSES = 'fa fa-check';
const CHECK_BTN_CLASSES = 'btn';

const REGENERATE_CLS = 'regenerate-btn';
const REGENERATE_BTN_SPAN_CLASSES = 'fa fa-refresh';
const REGENERATE_BTN_STYLE_CLASSES = 'btn btn-warning';

const DISCARD_CLS = 'discard-btn';
const DISCARD_BTN_SPAN_CLASSES = 'fa fa-remove';
const DISCARD_BTN_STYLE_CLASSES = 'btn btn-danger';

const CANCEL_BTN_SPAN_CLASSES = 'fa fa-stop-circle-o';
const CANCEL_BTN_CLASSES = 'btn btn-danger';

function createFaBtn(btnClasses, btnSpanClasses, btnHover, action = null) {
    const btn = document.createElement('a');

    // Make the button accessible to keyboard only use
    btn.setAttribute('role', 'button');
    btn.setAttribute('tabindex', '0');
    btn.addEventListener('keydown', (event) => {
      if (event.key === 'Enter') {
        wrapper.click();
      }
    });

    btn.setAttribute('class', btnClasses);
    btn.setAttribute('title', btnHover);
    btn.innerHTML = `<span class="${btnSpanClasses}"></span>`;

    if (typeof action === 'function')
        btn.onclick = action;

    return btn;
}

function changeFaBtnStyling(btn, btnClasses, btnHover, spanClasses,) {
    btn.setAttribute('class', btnClasses);
    btn.setAttribute('title', btnHover);
    btn.querySelector('span').setAttribute('class', spanClasses);
}

function styleBtnAsAccept(btn) {
    changeFaBtnStyling(btn, CHECK_BTN_CLASSES, SAVE_COMMENTS_HOVER, CHECK_BTN_SPAN_CLASSES);
}

function styleBtnAsStop(btn) {
    changeFaBtnStyling(btn, CANCEL_BTN_CLASSES, STOP_GENERATION_HOVER, CANCEL_BTN_SPAN_CLASSES);
}

function handleIndividualRegenerate(event, comment) {
    event.stopPropagation();

    generateDocumentationForIndividualComment(comment);

    const btnGroup = comment.getElementsByClassName(EDITOR_ACTIONS_GROUP_BTN_CLASSNAME)[0];
    hideElements([btnGroup]);
}

function handleIndividualDiscard(event, comment) {
    event.stopPropagation();

    const rawComment = comment.dataset.raw ?? "";
    const editor = getEditorForComment(comment);
    const preview = comment.querySelector('.preview');

    removeCommentFromEditList(comment);
    
    if (preview) {
        updatePreviewEditor(rawComment, preview);
    }

    if (editor) {
        replaceLexicalEditorContent(editor, rawComment);
    }

    // We check raw comment to determine if it was regenerated
    if (!rawComment && comment.classList.contains(AI_TARGET_RESULT_CLASSNAME)) {
        // Clear old editor
        const editor = getEditorForComment(comment);
        if (editor) {
            editor.setRootElement(null);
        }

        // Refresh ai-generated container
        const info = comment.dataset.info ? comment.dataset.info : "";
        const newAiContainer = makeCommentForAi(info, comment.dataset.start, comment.dataset.above);
        comment.replaceWith(newAiContainer);

    }
}

// ========================

/*
    Assuming the format of a comment for ai container as:

    <div class data-info data-start title>
        <a class>
		    <span class>
		    <img src>
        </a>
    </div>
*/
function makeCommentForAi(info, start, above) {
    const container = document.createElement('div');
    container.classList.add(AI_TARGET_CONTAINER_CLASSNAME);

    container.innerHTML =   `<a class="${AI_ASSISTANT_BTN_CLASSES}" title="${GENERATE_MISSING_COMMENT_HOVER}">
		                        <span class="${AI_ASSISTANT_BTN_SPAN_CLASSES}"></span>
		                        <img src="${AI_ASSISTANT_IMG_SRC.src}">
                            </a>`;

    wrapIndividualCommentGenerateButton(container);
    container.dataset.info = info;
    container.dataset.start = start;
    container.dataset.above = above;

    return container;
}

let lastMouseDownPos = { x: 0, y: 0 };
document.addEventListener('mousedown', (e) => {
    lastMouseDownPos = { x: e.clientX, y: e.clientY };
});

document.addEventListener('contextmenu', event => {
  const target = event.target;

  if (target.closest('div.comment div.editor')) {
    return;
  }

  event.preventDefault();
});

function elementContainsPoint(el, x, y) {
    const rect = el.getBoundingClientRect();
    return (
        x >= rect.left &&
        x <= rect.right &&
        y >= rect.top &&
        y <= rect.bottom
    );
}

let aiTargetContainer = null;
// TODO: refactor this?
function getAIContainer() {
    if (aiTargetContainer) {
        return aiTargetContainer;
    }

    return document.querySelector(`.${AI_TARGET_CONTAINER_CLASSNAME}`) || null;
}

function addPanZoom(element) {
    let panZoom = svgPanZoom(element, {
        viewportSelector: 'svg > g',
        fit: true,
        center: true,
        dblClickZoomEnabled: false,
        mouseWheelZoomEnabled: false,
        zoomEnabled: true,
        panEnabled: true
    });

    $(window).on('resize', function () {
        panZoom.resize();
        panZoom.fit();
        panZoom.center();
    });
}

function wrapInitialCommentInEditor(comment) {
    const originalHtml = comment.innerHTML;
    const rawComment = comment?.dataset?.raw?.replace(/\\n/g, "\n") ?? "";

    const wrapper = makeCommentStructure(comment, rawComment);
    wrapper.dataset.raw = rawComment ?? "";

    const preview = wrapper.querySelector('.preview');
    if (preview) {
        preview.innerHTML = originalHtml;
    }

    if (rawComment) {
        wrapper.querySelector('div.panes > div.editor').textContent = rawComment;
    }

    if (comment.id === 'ai-generated-comment') {
        wrapper.querySelector('div.panes > div.editor').id = 'ai-generated-comment';
    }

    comment.replaceWith(wrapper);
}



function wrapIndividualCommentGenerateButton(comment) {
    const floatingBtn = comment.querySelector('.floating');
    if (!floatingBtn) {
        return;
    }

    floatingBtn.title = GENERATE_MISSING_COMMENT_HOVER;

    comment.querySelector('img').src = AI_ASSISTANT_IMG_SRC.src;

    floatingBtn.onclick = (event) => {
        event.preventDefault();
        generateDocumentationForIndividualComment(comment);
    };
}

window.addEventListener('load', function () {
    AI_ASSISTANT_IMG_SRC.src = document.querySelector('#generate-button img').src;
    Object.freeze(AI_ASSISTANT_IMG_SRC);
    const generateBtn = getGenerateBtn();
    generateBtn.dataset.state = GENERATE_BTN_STATE.GENERATE;
    generateBtnCpy = generateBtn.cloneNode(true);


    const diagrams = document.getElementsByClassName('diagram');
    for (const element of diagrams) {
        if (element.classList.contains("fsm") || element.classList.contains("custom")) {
            continue;
        }

        const svgElements = element.getElementsByTagName('svg');
        if (svgElements.length == 0) {
            continue;
        }

        addPanZoom(svgElements[0]);
    }

    let fsmContainers = document.getElementsByClassName('fsm')
    for (const element of fsmContainers) {
        addPanZoom(element.querySelector("svg"));
    }

    $('table[id*="fsm-table"] > tbody > tr td:last-child').click(function (event) {
        event.stopPropagation();
    });

    const commentEls = Array.from(document.getElementsByClassName('comment'));
    commentEls.forEach(comment => wrapInitialCommentInEditor(comment));

    const generateDocumentationButton = getGenerateBtn();
    const aiContainers = document.getElementsByClassName(AI_TARGET_CONTAINER_CLASSNAME);
    generateDocumentationButton.addEventListener("click", handleGenerateBtnClick);
    if (aiContainers.length == 0) {
        hideElements([generateDocumentationButton]);
    }

    const elements = Array.from(document.getElementsByClassName(AI_TARGET_CONTAINER_CLASSNAME));
    elements.forEach(comment => wrapIndividualCommentGenerateButton(comment));

}, false);

function isInTable(element) {
    if (!element) {
        return false;
    }

    if (element.nodeName === 'TD') {
        return true;
    }

    return isInTable(element.parentNode);
}

function onCommentClick(event) {
    const element = event.currentTarget; // or event.target depending on your use

    const inTable = isInTable(element);
    if (inTable)
        event.stopPropagation();
    const borderStyle = inTable ? 'round-border-editor' : 'round-bottom-border';

    const editor = element.querySelector('.editor');
    const preview = element.querySelector('.preview');
    const toolbarHeader = element.querySelector('.toolbar-header');

    if (editor.dataset.lexicalEditor !== 'true') {
        createMarkdownEditor(element);
        // Enter editor mode on first creation
        hideElements([preview]);
        revealElements([editor]);
    } else {
        // Switch to editor and show toolbar
        element.querySelector('.toggle').click();
    }

    revealElements([toolbarHeader]);
    element.classList.add('no-hover');
    element.classList.add('active');
    preview.classList.add(borderStyle);
    editor.classList.add(borderStyle);

    const onOutsideClick = (event) => {
        // Check if the click was outside the div
        if (!element.contains(event.target)) {
            // If the release button event is outside but the click originated from within we ignore the event 
            // This can happen when overshooting text selection outside the border
            if (elementContainsPoint(element, lastMouseDownPos.x, lastMouseDownPos.y)) {
                return;
            }

            // If we are in edit mode we hit the toggle button to trigger preview mode
            if (!editor.classList.contains('hidden')) {
                element.querySelector('.toggle').click();
            }

            hideElements([toolbarHeader]);
            element.classList.remove('no-hover');
            element.classList.remove('active');
            preview.classList.remove(borderStyle);
            editor.classList.remove(borderStyle);

            element.addEventListener('click', onCommentClick);
            document.removeEventListener('click', onOutsideClick, true);
        }
    }

    // Add event to hide toolbar and enter preview mode if user clicks outside the editor
    // This should be handled with care in relation to other on click events
    document.addEventListener('click', onOutsideClick, true);

    element.removeEventListener('click', onCommentClick);
}

function action(number) {
    console.log('action ' + number + ' was executed');
}

function makeCommentStructure(comment, originalRawComment) {
    const wrapper = document.createElement('div');
    wrapper.dataset.info = comment.dataset.info ?? ("raw comment" + originalRawComment); // TODO: Make existing comments also contain information for regenerating the description
    
    wrapper.setAttribute('data-start', comment.getAttribute('data-start'));
    wrapper.setAttribute('data-end', comment.getAttribute('data-end'));
    wrapper.setAttribute('data-above', comment.getAttribute('data-above'));

    // Make the wrapper accessible to keyboard only use
    wrapper.setAttribute('role', 'button');
    wrapper.setAttribute('tabindex', '0');
    wrapper.addEventListener('keydown', (event) => {
        if (event.key === 'Enter') {
            wrapper.click();
        }
    });

    wrapper.classList.add('comment');
    wrapper.innerHTML = `
    <div class="toolbar-header hidden">
        <div class="toolbar" style="display: flex;">
            <button class="toggle toolbar-btn">Preview</button>
            <div class="style-btns" style="display: flex;">
                <div class="separator-line">
                    <div class="centered-vertical-line"></div>
                </div>
                <button class="bold toolbar-btn" title="Bold"><b>B</b></button>
                <button class="italic toolbar-btn" title="Italic"><i>I</i></button>
                <button class="underline toolbar-btn" title="Underline"><u>U</u></button>
                <button class="code toolbar-btn" title="Code Block">&lt;/&gt;</button>
                <button class="quote toolbar-btn" title="Quote">❝</button>

                <div class="separator-line">
                    <div class="centered-vertical-line"></div>
                </div>
                <button class="h1-btn toolbar-btn" title="Heading">H1</button>
                <button class="ul-btn toolbar-btn" title="Unordered list">UL</button>
            </div>
        </div>
    </div>
    <div class="floating-text-format-popup hidden" style="opacity: 0;">
        <div class="toolbar" style="display: flex;">
            <div class="style-floating-btns" style="display: flex;">
                <button class="bold toolbar-btn floating-btn" title="Bold"><b>B</b></button>
                <button class="italic toolbar-btn floating-btn" title="Italic"><i>I</i></button>
                <button class="underline toolbar-btn floating-btn" title="Underline"><u>U</u></button>
                <button class="code toolbar-btn floating-btn" title="Code Block">&lt;/&gt;</button>
                <button class="quote toolbar-btn floating-btn" title="Quote">❝</button>
                <div class="separator-line">
                    <div class="centered-vertical-line"></div>
                </div>
                <button class="h1-btn toolbar-btn floating-btn" title="Heading">H1</button>
                <button class="ul-btn toolbar-btn floating-btn" title="Unordered list">UL</button>
            </div>
        </div>
    </div>

    <div class="panes">
        <div class="editor hidden" contenteditable="true"></div>
        <div class="preview"></div>
    </div>`;

    if (isInTable(comment)) {
        wrapper.querySelector('.toolbar-header').classList.add('never-show');
    }

    if (originalRawComment) {
        wrapper.querySelector('div.panes > div.editor').textContent = originalRawComment;
    }

    wrapper.addEventListener('click', onCommentClick);

    return wrapper;
}

function createActionsBtnGroup(comment) {
    const regenerateLocalBtn = createFaBtn(REGENERATE_CLS + ' ' + REGENERATE_BTN_STYLE_CLASSES + ' ' + EDITOR_ACTIONS_BTN_CLASSNAME, REGENERATE_BTN_SPAN_CLASSES, 
                                REGENERATE_COMMENT_HOVER, (event) => { handleIndividualRegenerate(event, comment) });
    const discardLocalBtn = createFaBtn(DISCARD_CLS + ' ' + DISCARD_BTN_STYLE_CLASSES + ' ' + EDITOR_ACTIONS_BTN_CLASSNAME, DISCARD_BTN_SPAN_CLASSES, 
                                DISCARD_COMMENT_HOVER, (event) => { handleIndividualDiscard(event, comment) });

    const btnContainer = document.createElement('div');
    btnContainer.classList.add(EDITOR_ACTIONS_GROUP_BTN_CLASSNAME);

    btnContainer.append(regenerateLocalBtn);
    btnContainer.append(discardLocalBtn);

    return btnContainer;
}

$(document).ready(function () {
    const commentContainer = document.getElementById('ai-generated-container');
    if (!commentContainer) {
        return;
    }

    commentContainer.addEventListener('keydown', (event) => {
        if (event.key === 'Enter') {
            event.preventDefault();
            document.execCommand('insertLineBreak');
        }
    });
});

function updatePreviewEditor(markdown, previewContainer) {
    if (typeof window.vscodeOpen === "function" && vscodeOpen()) {
        const handler = (event) => {
            if (event.data.type === "DVTBridgeResult") {
                previewContainer.innerHTML = event.data.html;
                requestAnimationFrame(() => embedSvgZoom(previewContainer));
                window.removeEventListener("message", handler);
            }
        };

        window.addEventListener("message", handler);
        window['DVTBridge']('parse', markdown);
    } else {
        const result = window['DVTBridge']('parse', markdown);
        previewContainer.innerHTML = result;
        requestAnimationFrame(() => embedSvgZoom(previewContainer));
    }
}

function embedSvgZoom(container) {
    const svgs = container.querySelectorAll('svg');
    svgs.forEach(s => svgPanZoom(s, {
        viewportSelector: 'svg > g',
        fit: true,
        center: true,
        dblClickZoomEnabled: false,
        mouseWheelZoomEnabled: false,
        zoomEnabled: true,
        panEnabled: true
    }));
    fixMarkdownAdmonitions();
}

let remainingComments = 0;

// Key: start+data-info -> Value: { start, info, end }
let editedComments = new Map();

function getEditedCommentsForAccept() {
    const processed = []; // of list of lists that contain [content, start, end, insertAbove]
    for (const [, tuple] of editedComments) {
        const current = [tuple.content, tuple.start, tuple.end, tuple.insertAbove];
        processed.push(current);
    }

    return processed;
}

function changeStyleAfterGeneration(generatedRawComment = "") {
    remainingComments--;
    if (remainingComments == 0) {
        changeGenerateBtnToAcceptBtn();
    }

    let comment = getAIContainer();
    if (comment) {
        const originalHtml = comment.innerHTML;
        const wrapper = makeCommentStructure(comment, generatedRawComment);

        const actionsBtnGroup = createActionsBtnGroup(wrapper);
        wrapper.append(actionsBtnGroup);
        
        const preview = wrapper.querySelector('.preview');
        if (preview) {
            preview.innerHTML = originalHtml;
        }

        // Keep raw data if it was regenerated so we can discard to it if we have to
        if (comment.dataset.raw) {
            wrapper.dataset.raw = comment.dataset.raw;
        }

        wrapper.classList.add('ai-generated');

        if (generatedRawComment.length > 0) {
            addCommentToEditList(comment, generatedRawComment);
        }
        
        comment.replaceWith(wrapper);
    }
}

function addCommentToEditList(comment, rawContent) {
    const start = comment.getAttribute('data-start');
    const end = comment.getAttribute('data-end');
    const info = comment.getAttribute('data-info');
    const insertAbove = comment.getAttribute('data-above');

    // key is concatenation between start and data-info in cases where comments share the same startLine
    // e.g.: multiple ports declared on the same line
    editedComments.set(start + info, { content: rawContent, start: start, end: end, insertAbove: insertAbove });

    const btn = getGenerateBtn();
    if (btn.dataset.state !== GENERATE_BTN_STATE.STOP) {
        changeGenerateBtnToAcceptBtn();
    }
}

function removeCommentFromEditList(comment) {
    const start = comment.getAttribute('data-start');
    const info = comment.getAttribute('data-info');
    editedComments.delete(start + info);
    const commentsForAI = document.getElementsByClassName(AI_TARGET_CONTAINER_CLASSNAME);
    if (editedComments.size === 0 ) {
        if (commentsForAI.length !== 0) {
            changeGenerateBtnToGenerateBtn();
        } else {
            const generateBtn = getGenerateBtn();
            hideElements([generateBtn]);
        }
    }
}

const cancelAutoScroll = { value: true };
function scrollToDiv(target) {
    const stop = () => { cancelAutoScroll.value = true; cleanup(); };
    const cleanup = () => {
        window.removeEventListener('wheel', stop);
        window.removeEventListener('touchstart', stop);
        window.removeEventListener('keydown', stop);
    };

    window.addEventListener('wheel', stop);
    window.addEventListener('touchstart', stop);
    window.addEventListener('keydown', stop);

    const startY = window.scrollY;
    const endY = Math.max(target.getBoundingClientRect().top + startY - 50, 0);
    const duration = 600;                     // ms
    const start = performance.now();

    function step(now) {
        if (cancelAutoScroll.value) {
            return;
        }               // user stopped us
        const elapsed = now - start;
        const progress = Math.min(elapsed / duration, 1);
        // ease‑in‑out (optional)
        const ease = progress < .5
            ? 2 * progress * progress
            : -1 + (4 - 2 * progress) * progress;
        window.scrollTo(0, startY + (endY - startY) * ease);

        if (progress < 1) {
            requestAnimationFrame(step);
        }
        else {
            cleanup();
        }                         // finished normally
    }

    requestAnimationFrame(step);
}

function disableClicks() {
    document.addEventListener("click", handler, true);

    function handler(e) {
        e.stopPropagation();
        e.preventDefault();
    }
}

async function acceptDocumentationGeneration() {
    hideElements([getGenerateBtn()]);
    disableClicks();
    
    window['DVTBridge']('accept', getEditedCommentsForAccept());
}

async function stopDocumentationGeneration() {
    changeGenerateBtnToAcceptBtn();
    window['DVTBridge']('stop');
}

function lockAllIndividualComments() {
    const buttons = document.querySelectorAll('div.comment-for-ai');
    buttons.forEach(button => {
        button.title = "Another message is being generated!";
        button.classList.add('locked');
    });
}

function unlockAllIndividualComments() {
    const buttons = document.querySelectorAll('div.comment-for-ai.locked');
    buttons.forEach(button => {
        button.title = "";
        button.classList.remove('locked');
    });
}

function generateDocumentationForIndividualComment(comment) {
    lockAllIndividualComments();
    remainingComments = 1;
    aiTargetContainer = comment;

    const floatingBtn = comment.querySelector('.floating');
    if (floatingBtn) {
        floatingBtn.remove();
    }

    changeGenerateBtnToStopBtn();
    //hideElements([getGenerateBtn()]);
        
    const declarations = getElementsDeclarationList([comment]);
    window['DVTBridge']('generate', declarations);
}

function getElementsDeclarationList(elements) {
    const declarations = [];
    for (const elem of elements) {
        const info = elem.dataset.info;
        if (!info) {
            console.error("Element: " + elem + " has no info");
        }
        
        declarations.push(info);
    }

    return declarations;
}

function getGenerateBtn() {
    const btn = document.getElementById("generate-button");
    if (!btn) {
        console.error('No main generate button found');
        return null;
    } else {
		btn.style.display = 'block';
	}

    return btn;
}

function hideElements(elements) {
    for (const elem of elements) {
        elem?.classList?.add('hidden');
    }
}

function revealElements(elements) {
    for (const elem of elements) {
        elem?.classList?.remove('hidden');
    }
}

function handleGenerateBtnClick() {
    const btn = getGenerateBtn();
    switch (btn.dataset.state) {
        case GENERATE_BTN_STATE.GENERATE:
            generateAllDocumentation();
            break;

        case GENERATE_BTN_STATE.STOP:
            stopDocumentationGeneration();
            streamEndHandler("")
            break;

        case GENERATE_BTN_STATE.ACCEPT:
            acceptDocumentationGeneration();
            break;

        default:
            error.log('Invalid generate button state: ' + btn.dataset.state);
    }
}

/*
    Resets the styling
*/
function changeGenerateBtnToGenerateBtn() {
    let generateDocumentationButton = getGenerateBtn();
    generateDocumentationButton.replaceWith(generateBtnCpy.cloneNode(true));

    generateDocumentationButton = getGenerateBtn()
    generateDocumentationButton.addEventListener("click", handleGenerateBtnClick);
    generateDocumentationButton.dataset.state = GENERATE_BTN_STATE.GENERATE;
}

function changeGenerateBtnToAcceptBtn() {
    const btn = getGenerateBtn();
    btn.querySelector('img')?.remove();
    btn.querySelector(".title")?.remove();
    styleBtnAsAccept(btn);
    btn.dataset.state = GENERATE_BTN_STATE.ACCEPT;
}

function changeGenerateBtnToStopBtn() {
    const btn = getGenerateBtn();
    btn.querySelector('img')?.remove();
    btn.querySelector(".title")?.remove();
    styleBtnAsStop(btn);
    btn.dataset.state = GENERATE_BTN_STATE.STOP;
    
}

async function generateAllDocumentation() {
    cancelAutoScroll.value = false;

    changeGenerateBtnToStopBtn();

    const elements = document.getElementsByClassName(AI_TARGET_CONTAINER_CLASSNAME);
    const declarations = getElementsDeclarationList(elements);

    remainingComments = declarations.length;
    window['DVTBridge']('generate', declarations);
}


let isMarkdown = false;

async function processChunk(chunk) {
    let processedChunk = chunk;
    if (isMarkdown) {
        processedChunk = window['DVTBridge']('parse', chunk);
    }

    document.getElementsByClassName(AI_TARGET_CONTAINER_CLASSNAME)[0].innerHTML = processedChunk;

    if (isMarkdown) {
        fixMarkdownAdmonitions();
    }
}

function saveHTMLChunk(container, chunk) {
    container.innerHTML = chunk;

    if (isMarkdown) {
        fixMarkdownAdmonitions();
    }
}

let activeLexicalEditor = null;

function improveExistingDescription(editor, action, text) {
    activeLexicalEditor = editor
    window['DVTBridge']('improve', action, text);
}

function appendToLexical(textChunk) {
    if (activeLexicalEditor === null) {
        return;
    }

    appendToLexicalEditor(activeLexicalEditor, textChunk);
}

window.addEventListener("message", event => {
    const message = event.data;
    const generateBtn = getGenerateBtn();
    const targetComment = getAIContainer();

    switch (message.type) {
        case "stream_start":
            isMarkdown = message.isMarkdown;

            scrollToDiv(getAIContainer());

            if (targetComment) {
                targetComment.innerHTML = '';
                targetComment.classList.add('ai-generated');
            }
            generateBtn.style.display = 'flex';
            break;

        case "stream": {
            if (message.html.length > 0) {
                saveHTMLChunk(targetComment, message.html);
            } else if (targetComment) {
                targetComment.innerHTML += message.content;
            } else if (activeLexicalEditor) {
                appendToLexical(message.content)

            }
            break;
        }

        case "stream_end": {
            if (message.entireMessage) {
                saveHTMLChunk(targetComment, message.html);
            }
            streamEndHandler(message.entireMessage)
        }
    }
});

function streamEndHandler(entireMessage) {
    changeStyleAfterGeneration(entireMessage);
    revealElements([getGenerateBtn()]);
    aiTargetContainer = null;
    unlockAllIndividualComments();
    updateWaveDromStyles();
}

function updateWaveDromStyles() {
    $('svg.WaveDrom').each(function() {
        const style = $(this).find('style');
        if (style.length) {
            style.html(style.html().replace(/text\s*{/g, '.WaveDrom text {'));
        }
    });
}
