const SCALE_RESOLUTION = 10;
const ZOOM_OUT_AMOUNT = 1.8;

const diagram = document.getElementById("diagram-div");
const separator = document.getElementById("separator-div");
const table = document.getElementById("table-div");
const preferences = document.getElementById("preferences-div");
const diagramMenu = document.getElementById("diagram-menu");

const contextMenuOptions = [
	document.getElementById("zoom-in-menu-option"),
	document.getElementById("zoom-out-menu-option"),
	document.getElementById("fit-canvas-menu-option"),
	document.getElementById("preferences-menu-option")
];

const bitfieldTooltip = document.getElementById("vscode-bitfield-tooltip");
const nameText = document.getElementById("vscode-bitfield-tooltip-name");
const accessText = document.getElementById("vscode-bitfield-tooltip-access");
const resetText = document.getElementById("vscode-bitfield-tooltip-reset");
	
var svg, svgImage, svgElement;
var initTransform, initScale, initXOff, initYOff;

var vscode;
if (typeof acquireVsCodeApi === "function")
	vscode = acquireVsCodeApi();
        
var scale = 1;
var isPanning = false;
var xoff = 0;
var yoff = 0;
var start = {
    x: 0,
    y: 0
};

var menuSourcePoint = {
    x: 0,
    y: 0
};
var selectedOptionIndex;

var edgeSelection = {
	edge: undefined,
	arrow: undefined,
	label: undefined
};
var rectSelection;
var mouseDownSepCoords;

preprocessDiagram();
if (svgImage !== undefined) {
    addDiagramListeners();
    addKeysListeners();
}

function preprocessDiagram() {
	var svgImages = document.getElementsByTagName("svg");
	if (svgImages === undefined)
		return;
		
	for (let item of svgImages) {
		let parent = item.parentNode;
		if (parent === undefined)
			continue;
		
		if (parent.id !== "diagram-div")
			continue;
		
		svgImage = item;
		svg = gySVG(svgImage);
		svgElement = getElementOfInterest();
	}
	
    svgImage.style.width = "100%";
	svgImage.style.height = "100%";
	
    var bbox = svgImage.getBBox();
    svgImage.setAttribute('viewBox', `${bbox.x} ${bbox.y} ${bbox.width} ${bbox.height}`);

	zoomOutFixedAmountAboutCenter(1.2);
    
    initScale = scale;
    initXOff = xoff;
    initYOff = yoff;
    initTransform = svgElement.style.transform;
	
    if (window.registerModelMap !== undefined)
        diagram.setAttribute("style", "padding-block-start: 1em; padding-block-end: 1em");
    
    /* Deactivate text selection */
    $('svg').find('tspan').each( function(index, value) {
        $(value).attr('cursor', 'default');
    });
    
    $('svg').find('text').each( function(index, value) {
        $(value).attr('cursor', 'default');
    });
}

function getElementOfInterest() {
	const waves = $('g[id^="waves_0"]')[0];
	if (waves === undefined)
		return svgImage.getElementsByTagName("g")[0];
		
	return waves;
}

function addDiagramListeners() {
    svgImage.addEventListener("wheel", zoomIncrementedScaleAboutPoint);
    svgImage.addEventListener("mousedown", onMouseDown);
    svgImage.addEventListener("mousemove", onMouseMove);
    svgImage.addEventListener("mouseup", onMouseUp);
    svgImage.addEventListener("click", highlightComponent);
    
    if (typeof registerModelMap !== 'undefined') {
    	VSCodeManageBitfieldSVGTooltip();
	}
}

function addKeysListeners() {
    if (document === undefined)
        return;
        
    document.addEventListener('keydown', (event) => {
        switch(event.key) {
            case "ArrowLeft":
                if (isMenuVisible())
                    break;
                    
                panFixedAmount(-50, 0);
                break;
            case "ArrowUp":
                if (!isMenuVisible()) {
                    panFixedAmount(0, -50);
                    break;
                }
                
                selectPreviousMenuOption();
                break;
            case "ArrowRight":
                if (isMenuVisible())
                    break;
                    
                panFixedAmount(50, 0);
                break;
            case "ArrowDown":
                if (!isMenuVisible()) {
                    panFixedAmount(0, 50);
                    break;
                }
                
                selectNextMenuOption();
                break;
            case "0":
                zoomToFitCanvas();
                break;
            case "+":
                zoomInFixedAmountAboutCenter();
                break;
            case "-":
                zoomOutFixedAmountAboutCenter(ZOOM_OUT_AMOUNT);
                break;
            case "F10":
                if (diagram === undefined || !event.shiftKey)
                    break;
            
                var diagramDivCenterX = diagram.offsetLeft + (diagram.offsetWidth / 2);
                var diagramDivCenterY = diagram.offsetTop + (diagram.offsetHeight / 2);
                handleContextMenu(diagramDivCenterX, diagramDivCenterY);
                
                break;
            case "Escape":
                if (isMenuVisible()) {
	                hideContextMenu();
	                break;
                }
                
                if (preferences.style.display === "flex") {
                	hidePreferences();
                	break;
            	}
            case "Enter":
                if (!isMenuVisible())
                    break;
                
                if (selectedOptionIndex === 0)
                    zoomInFixedAmountAboutCenter();
                else if (selectedOptionIndex === 1)
                    zoomOutFixedAmountAboutCenter(ZOOM_OUT_AMOUNT);
                else if (selectedOptionIndex === 2)
                    zoomToFitCanvas();
                else if (selectedOptionIndex === 3)
                	showPreferences();
                	
                hideContextMenu(); 
                break;
            default:
                break;
        }
    });
}

function selectNextMenuOption(){
    if (selectedOptionIndex == contextMenuOptions.length - 1 || selectedOptionIndex === undefined)
        selectMenuOption(0);
    else
        selectMenuOption(selectedOptionIndex + 1);
}

function selectPreviousMenuOption(){
    if (selectedOptionIndex == 0 || selectedOptionIndex === undefined)
        selectMenuOption(contextMenuOptions.length - 1);
    else
        selectMenuOption(selectedOptionIndex - 1);
}

function zoomIncrementedScaleAboutPoint(event) {
    /* Convert from the screen coordinate system to the SVG coordinate system. */
    var svgSysCoords = getCoordsInSVGSpace(event.clientX, event.clientY);
    
    /* Get scroll direction & set zoom level. */
    var recomputedScale = scale;
    var delta = (event.wheelDelta ? event.wheelDelta : -event.deltaY);
    if (delta > 0)
        recomputedScale *= 1.1;
    else
        recomputedScale /= 1.1;
        
    zoomFixedScaleAboutPoint(recomputedScale, svgSysCoords.x, svgSysCoords.y)
}
    
function zoomFixedScaleAboutPoint(recomputedScale, svgSysCoordsX, svgSysCoordsY) {
    /* Take the scale into account with the offset. */
    var xs = (svgSysCoordsX - xoff) / scale;
    var ys = (svgSysCoordsY - yoff) / scale;
    
    scale = recomputedScale;
    
    /* Reverse the offset amount with the new scale. */
    xoff = svgSysCoordsX - xs * scale;
    yoff = svgSysCoordsY - ys * scale;

    setTransform();
}

function getCoordsInSVGSpace(coordX, coordY) {
    var pt = svgImage.createSVGPoint();
    pt.x = coordX;
    pt.y = coordY;
    return pt.matrixTransform(svgImage.getScreenCTM().inverse());
}

function setTransform() {
    if (svgElement === undefined)
		return;
		
    svgElement.style.transform = "translate(" + xoff + "px, " + yoff + "px) scale(" + scale + ")";
}

function resetTransform() {
    if (svgElement === undefined)
		return;
		
    svgElement.style.transform = initTransform;
}
    
function onMouseDown(event) {
    switch (event.which) {
        case 1: /* Left click */
            event.preventDefault();
            
            var svgSysCoords = getCoordsInSVGSpace(event.clientX, event.clientY);
            start = {
                x: svgSysCoords.x - xoff,
                y: svgSysCoords.y - yoff
            };
            
            isPanning = true;
            
            diagramMenu.style.display = "none";
            
            break;
        case 3: /* Right click */
            event.preventDefault();
             
            handleContextMenu(event.pageX, event.pageY);
            
            break;
        default:
            break;
    }
}

function onMouseMove(event){
    event.preventDefault();
            
    if (!isPanning)
        return;
    
    var svgSysCoords = getCoordsInSVGSpace(event.clientX, event.clientY);
    
    xoff = (svgSysCoords.x - start.x);
    yoff = (svgSysCoords.y - start.y);
    
    setTransform();
    svgImage.style.cursor = "grabbing";
}

function onMouseUp(event){
    if (!isPanning)
        return;
         
    svgImage.style.cursor = "default";
    isPanning = false;
}

function getElementsAroundPoint(initialX, initialY) {
	let elements = new Set();
	for (let x = initialX - 6; x <= initialX + 6; x++) {
		for (let y = initialY - 6; y <= initialY + 6; y++) {
			let elemsAtPos = getAllElementsFromPoint(x, y);
			if (elemsAtPos.length === 0)
				continue;
				
			elemsAtPos.forEach(elem => elements.add(elem));
		}
	}
	
	return elements;
}

function getAllElementsFromPoint(x, y) {
    var elements = [];
    var elementsOfInterest = [];
    var display = [];
    var item = document.elementFromPoint(x, y);
    
    while (item && item.nodeName !== "svg") {
    	if (selectedRect(item) || selectedEdge(item) || selectedArrowHead(item))
    		elementsOfInterest.push(item);
    		
    	elements.push(item);
        display.push(item.style.display);
        item.style.display = "none";
        item = document.elementFromPoint(x, y);
    }
    
    // restore display property
    for (var i = 0; i < elements.length; i++) {
        elements[i].style.display = display[i];
    }
    
    return elementsOfInterest;
}

function highlightComponent(event) {
	let elem;
	
	if (event.target.nodeName === "svg") {
		let x = event.clientX;
		let y = event.clientY;
		elem = getClosestElem(x, y, getElementsAroundPoint(x, y));
	} else {
		elem = event.target;
	}
	
	if (elem === undefined) {
		resetSelection();
		return;
	}
		
	if (selectedRect(elem)) {
		if (isLabel(elem)) {
			for (let e of window.edges) {
				if (e === undefined)
					continue;
				
				if (e.labelText !== elem.textContent)
					continue;
				
				if (e.labelX !== parseInt(elem.getAttribute("x")))
					continue;
				
				/*if (e.labelY !== parseInt(elem.getAttribute("y")))
					continue;*/
				
				highlightEdge(getEdge(e.line), getArrow(e.arrow), getLabel(e.labelText, e.labelX, e.labelY));
			}
			
			return;
		}
			
		let rect = svg.querySelector("#" + elem.getAttribute("rectid"));
		if (rect === undefined)
			return;
					
		highlightRect(rect);
		return;
	}
	
	if (selectedEdge(elem)) {
		let linePoints = elem.getAttribute("d");
		if (linePoints === undefined)
			return;
		
		let line = "[" + linePoints.replace(new RegExp("[a-zA-Z]", "g"), '').replace(new RegExp(" ", "g"), ', ') + "]";
		for (let e of window.edges) {
			if (e === undefined)
				continue;
			
			if (e.line !== line)
				continue;
				
			highlightEdge(getEdge(e.line), getArrow(e.arrow), getLabel(e.labelText, e.labelX, e.labelY));
			return;
		}
	}
	
	if (selectedArrowHead(elem)) {
		let arrowPoints = elem.getAttribute("points");
		if (arrowPoints === undefined)
			return;
			
		for (let e of window.edges) {
			if (e === undefined)
				continue;
			
			if (arrowPoints.trim() !== e.arrow.replace(new RegExp("\\[", "g"), '').replace(new RegExp("\\]", "g"), '').replace(new RegExp(", ", "g"), ' '))
				continue;
			
			highlightEdge(getEdge(e.line), getArrow(e.arrow), getLabel(e.labelText, e.labelX, e.labelY));
			return;
		}
	}
	
	resetSelection();
}

function getClosestElem(x, y, elements) {
	let distances = [];
	elements.forEach(elem => {
		let point = getPositionAtCenter(elem);
		distances.push({
			element: elem, 
			distance: Math.hypot(x - point.x, y - point.y)
		});
	});
	
	let minDist = Number.MAX_VALUE;
	let closestElem;
	distances.forEach(item => {
		if (item.distance <= minDist) {
			closestElem = item;
			minDist = closestElem.distance;
		}
	});
	
	if (closestElem === undefined)
		return undefined;
	
	return closestElem.element;
}

function getPositionAtCenter(element) {
	const {top, left, width, height} = element.getBoundingClientRect();
	return {
		x: left + width / 2,
		y: top + height / 2
	};
}

function selectedRect(element) {
	if (element.tagName === "text" && element.getAttribute("rectid") !== undefined)
		return true;
		
	return false;
}

function isLabel(element) {
	for (let e of window.edges) {
		if (e === undefined)
			continue;
		
		if (e.labelText === element.textContent)
			return true;
	}
	
	return false;
}

function selectedEdge(element) {
	if (element.tagName === "path")
		return true;
	
	return false;
}

function selectedArrowHead(element) {
	if (element.tagName === "polygon")
		return true;
	
	return false;
}

function getArrow(arrow) {
	if (arrow === undefined)
		return undefined;
		
	return svg.querySelector("polygon[points=\" " + arrow.replace(new RegExp("\\[", "g"), '').replace(new RegExp("\\]", "g"), '').replace(new RegExp(", ", "g"), ' ') + "\"][stroke-linecap=\"round\"]");
}

function getLabel(labelText, labelX, labelY) {
	if (labelText === undefined)
		return undefined;
	
	return svg.querySelectorAll('text').find(el => el.textContent() === labelText && el.getAttribute("x") === parseInt(labelX));
}

function getEdge(edge) {
	if (edge === undefined)
		return undefined;
	
	return svg.querySelectorAll('path').find(el => {
		let path = el.d();
		if (path === undefined)
			return;
		
		return edge === "[" + path.replace(new RegExp("[a-zA-Z]", "g"), '').replace(new RegExp(" ", "g"), ', ') + "]";
	});
}

function highlightRect(element) {
    if (element === undefined)
    	return;
    
    resetSelection();
	
	rectSelection = element;
	rectSelection.classList.add("selected-types");
}

function highlightEdge(edge, arrow, label) {
	resetSelection();
    
	edgeSelection.edge = edge;
	edgeSelection.arrow = arrow;
	edgeSelection.label = label;
    
	if (edgeSelection.edge !== undefined)
		edgeSelection.edge.classList.add("selected-edge-line");
	
	if (edgeSelection.arrow !== undefined)
		edgeSelection.arrow.classList.add("selected-edge-arrow");
	
	if (edgeSelection.label !== undefined)
		edgeSelection.label.classList.add("selected-edge-label");
}

function resetSelection() {
	resetRectSelection();
	resetEdgeSelection();
}

function resetRectSelection() {
	if (rectSelection === undefined)
		return;
		
	rectSelection.classList.remove("selected-types");
}

function resetEdgeSelection() {
	if (edgeSelection.edge !== undefined)
		edgeSelection.edge.classList.remove("selected-edge-line")
	
	if (edgeSelection.arrow !== undefined)
		edgeSelection.arrow.classList.remove("selected-edge-arrow");
	
	if (edgeSelection.label !== undefined)
		edgeSelection.label.classList.remove("selected-edge-label");
}

function handleContextMenu(coordX, coordY) {
    if (diagramMenu.style.display === "none")
        showContextMenu(coordX, coordY);
    else
        hideContextMenu();
}

function showContextMenu(coordX, coordY) {
    if (diagramMenu === undefined)
        return;
        
    diagramMenu.style.display = "block";
    diagramMenu.style.left = coordX + "px";
    diagramMenu.style.top = coordY + "px";
    menuSourcePoint.x = coordX;
    menuSourcePoint.y = coordY;
}

function hideContextMenu() {
    if (diagramMenu === undefined)
        return;
        
    for(let option of contextMenuOptions) {
        if (option === undefined)
            return;

        option.classList.remove("keyboard-hovered");
    }
    
    selectedOptionIndex = undefined;
    diagramMenu.style.display = "none";
}

function zoomInFixedAmountAboutPoint() {
    var zoomPoint = getCoordsInSVGSpace(menuSourcePoint.x, menuSourcePoint.y);
    if (zoomPoint === undefined)
        return;
        
    zoomFixedScaleAboutPoint(scale * 1.8, zoomPoint.x, zoomPoint.y);
    diagramMenu.style.display = "none";
}

function zoomOutFixedAmountAboutPoint() {
    var zoomPoint = getCoordsInSVGSpace(menuSourcePoint.x, menuSourcePoint.y);
    if (zoomPoint === undefined)
        return;
       
    zoomFixedScaleAboutPoint(scale / 1.8, zoomPoint.x, zoomPoint.y);
    diagramMenu.style.display = "none";
}

function zoomToFitCanvas() {
    scale = initScale;
    xoff = initXOff;
    yoff = initYOff;
    isPanning = false;
    start = {
        x: 0,
        y: 0
    };

    resetTransform();
    diagramMenu.style.display = "none";
}

function mouseDownOnSeparator(e) {
    mouseDownSepCoords = {
    	e,
        offsetTop:  separator.offsetTop,
        diagramHeight:  diagram.offsetHeight,
        tableHeight: table.offsetHeight,
    };
    
    document.onmousemove = mouseMoveOnSeparator;
    document.onmouseup = function(event) {
        document.onmousemove = document.onmouseup = null;
    }
}

function mouseMoveOnSeparator(e) {
    var delta = {
    	x: e.clientX - mouseDownSepCoords.e.clientX,
        y: e.clientY - mouseDownSepCoords.e.clientY
    };
    
    delta.y = Math.min(Math.max(delta.y, -mouseDownSepCoords.diagramHeight), mouseDownSepCoords.tableHeight);

    separator.style.top = mouseDownSepCoords.offsetTop + delta.y + "px";
    diagram.style.height = (mouseDownSepCoords.diagramHeight + delta.y) + "px";
    table.style.height = (mouseDownSepCoords.tableHeight - delta.y) + "px";
}

function getSVGCenter() {
    if (svgImage === undefined || svgImage.viewBox === undefined || svgImage.viewBox.baseVal === undefined)
        return undefined;
        
    return {
        x: svgImage.viewBox.baseVal.x + (svgImage.viewBox.baseVal.width / 2),
        y: svgImage.viewBox.baseVal.y + (svgImage.viewBox.baseVal.height / 2)
    };
}

function zoomInFixedAmountAboutCenter() {
    var zoomPoint = getSVGCenter();
    if (zoomPoint === undefined)
        return;
        
    zoomFixedScaleAboutPoint(scale * 1.8, zoomPoint.x, zoomPoint.y);
    
    if (diagramMenu === undefined)
        return;
        
    diagramMenu.style.display = "none";
}

function zoomOutFixedAmountAboutCenter(amount) {
    var zoomPoint = getSVGCenter();
    if (zoomPoint === undefined)
        return;
    zoomFixedScaleAboutPoint(scale / amount, zoomPoint.x, zoomPoint.y);
    
    if (diagramMenu === undefined)
        return;
        
    diagramMenu.style.display = "none";
}

function panFixedAmount(dx, dy) {
    xoff += dx;
    yoff += dy;
    
    setTransform();
}

function isMenuVisible() {
    if (diagramMenu === undefined)
        return;
        
    return diagramMenu.style.display === "block";
}

function selectMenuOption(selectedOption) {
    for (var i = 0; i < contextMenuOptions.length; i++) {
        let option = contextMenuOptions[i];
        if (option === undefined)
            continue;

        if (i === selectedOption) {
            option.classList.add("keyboard-hovered");
            selectedOptionIndex = i;
            continue;
        }

        option.classList.remove("keyboard-hovered");
    }
}

function clickOnRegisterName(event) {
    let registerPosData = { type: "goToRegister" };
    
    registerPosData.file = registerFilePath;
    registerPosData.startLine = registerStartLine;
    registerPosData.startCharacter = registerStartCharacter;
    registerPosData.endLine = registerEndLine;
    registerPosData.endCharacter = registerEndCharacter;
    
    if (vscode !== undefined)
	    vscode.postMessage(registerPosData);
}

ProcessWaveDromDiagram();

function showPreferences() {
    if (preferences === undefined || diagramMenu === undefined || vscode === undefined)
        return;
    
    vscode.postMessage({type: "showPreferences"});
    
    preferences.style.display = "flex";
    diagramMenu.style.display = "none";
}

function hidePreferences() {
	if (preferences === undefined)
        return;
        
    preferences.style.display = "none";
}

function increaseValue(element) {
	let parent = element.parentElement;
	if (parent === undefined)
		return;
		
	let counter = parent.firstElementChild;
	if (counter === undefined)
		return;
	
	let newValue = Number(counter.value) + 1;
	counter.value = newValue;
	
	actualPrefs.set(counter.name, newValue);
	
	if (newValue > 1) {
		let decrBtn = element.previousElementSibling;
		if (decrBtn === undefined)
			return;
			
		decrBtn.classList.remove("inactive-button");
		decrBtn.setAttribute("onclick", "decreaseValue(this)");
	}
}

function decreaseValue(element) {
	let parent = element.parentElement;
	if (parent === undefined)
		return;
		
	let counter = parent.firstElementChild;
	if (counter === undefined)
		return;
	
	let newValue = Number(counter.value) - 1;
	counter.value = newValue;
	
	actualPrefs.set(counter.name, newValue);
	
	if (newValue == 1) {
		element.classList.add("inactive-button");
		element.removeAttribute("onclick");
	} else {
		element.classList.remove("inactive-button");
		element.setAttribute("onclick", "decreaseValue(this)");
	}
}

function applyConfig(selectedNew) {
	if (vscode === undefined)
		return;
		
    vscode.postMessage({
    	type: "applyConfig",
    	showNew: selectedNew ? true : false,
    	prefs: Array.from(actualPrefs, ([name, value]) => ({ name, value })),
    	element: window.element
	});
	
	if (selectedNew)
		resetConfig();
	
	hidePreferences();
}

function resetConfig() {
    window.preferences.forEach((pref) => {
    	initPrefs(pref.name, pref.value);
	});
 
    actualPrefs = new Map(originalPrefs);
    
    hidePreferences();
}

function goToUMLDiagramsSettings() {
    if (vscode !== undefined)
	    vscode.postMessage({ type: "goToUMLDiagramsSettings" });
}

function VSCodeManageBitfieldSVGTooltip() {
	$('rect').within('svg').on('mouseenter', function(event) {
		showBitfieldTooltip(event, bitfieldTooltip, registerModelMap);
	}).on('mousemove', function(event) {
		moveBitfieldTooltip(event, bitfieldTooltip);
	}).on('mouseleave', function(event) {
		let relatedTarget = event.relatedTarget;
		if (relatedTarget === null)
			return;
			
		if (relatedTarget.tagName !== "tspan")
			bitfieldTooltip.style.display = "none";
	});
	
	$('text > tspan').within('svg').on('mousemove', function(event) {
		moveBitfieldTooltip(event, bitfieldTooltip);
	}).on('click', function(event) {
		if (event.target.parentElement.tagName === "text") {
			let target = event.target;
			if (target === null)
				return;
				
			let parent = target.parentElement;
			if (parent === null)
				return;

			$("g > rect[x=" + parent.getAttribute('x') + "]").within('svg').each(function(index, value) {
				var event = document.createEvent('MouseEvent');
			    event.initEvent('click', true, false);
			    $(value)[0].dispatchEvent(event);
		    });
		}
	});
}

function showBitfieldTooltip(event, bitfieldTooltip, registerModelMap) {
	bitfieldTooltip.style.left = event.pageX + 10 + "px";
	bitfieldTooltip.style.top = event.pageY + 20 + "px";
	bitfieldTooltip.style.display = "flex";
	
	var rectX = parseInt(event.target.getAttribute('x'));
	        
    if (typeof registerModelMap[rectX] !== 'undefined') {
        nameText.textContent = "Name: " + registerModelMap[rectX][0];
        accessText.textContent = "Access: " + registerModelMap[rectX][3];
        resetText.textContent = "Reset: " + registerModelMap[rectX][5];
        
        var type = parseInt(registerModelMap[rectX][9]);
        switch(type) {
            case 1:
                bitfieldTooltip.style.backgroundColor = window.DVTApplyCustomDarkTheme ? "gray" : "lightgray";
                break;
            case 2:
                bitfieldTooltip.style.backgroundColor = window.DVTApplyCustomDarkTheme ? "pink" : "lightpink";
                break;
            case 3:
                bitfieldTooltip.style.backgroundColor = window.DVTApplyCustomDarkTheme ? "green" : "lightgreen";
                break;
            case 4:
                bitfieldTooltip.style.backgroundColor = window.DVTApplyCustomDarkTheme ? "blue" : "lightblue";
                break;
            case 5:
                bitfieldTooltip.style.backgroundColor = window.DVTApplyCustomDarkTheme ? "orange" : "lemonchiffon";
                break;
            default:
                break;
        }
        
        bitfieldTooltip.style.color = window.DVTApplyCustomDarkTheme ? "white" : "black";
    } else {
        bitfieldTooltip.style.display = "none";
    }
}

function moveBitfieldTooltip(event, bitfieldTooltip) {
	bitfieldTooltip.style.left = event.pageX + 10 + "px";
	bitfieldTooltip.style.top = event.pageY + 20 + "px";
}