/*
Verilog Autoconfig Lexer
*/

header {
package ro.amiq.dvt.buildconfig;

import java.util.List;
import java.util.Set;
import java.util.Map;
import java.util.ArrayList;
import java.util.StringTokenizer;
import java.io.File;
import java.io.CharArrayReader;
import java.io.InputStreamReader;

import ro.amiq.dvt.IDVTConstants;
import ro.amiq.dvt.startup.core.DVTLogger;
import ro.amiq.dvt.utils.FileInputStreamUtils;
import ro.amiq.dvt.utils.expr.ExprEval;
import ro.amiq.dvt.utils.parser.Comment;
}

class AutoConfigVlogLexer extends Lexer;

options {
    k = 3;
    charVocabulary='\u0000'..'\uFFFE';
}

tokens {
	DEFINE;
	IMPORT;
	INCLUDE;
	MACRO_CALL;
}

{

    @Override
    public void reportError(RecognitionException ex) {
    	super.reportError(ex);
    }

    @Override
    public void userExceptionHandler(Exception ex) throws RecognitionException, CharStreamException, TokenStreamException {
        if (ex instanceof RecognitionException)
            throw (RecognitionException) ex;
        if (ex instanceof CharStreamException)
            throw (CharStreamException) ex;
        if (ex instanceof TokenStreamException)
            throw (TokenStreamException) ex;
        DVTLogger.INSTANCE.logError(ex);
    }

}

INVALID_CHARS
	:
		( '\u0000'..'\u0008' | '\u000B' | '\u000E'..'\u001F' | '\u007F'..'\uFFFE' )+
		{ $setType(Token.SKIP); }
	;

VALID_CHAR
	:
		( "\r\n"            ) => "\r\n"                   { newline(); $setType(Token.SKIP); }
	| 	( "\r"              ) => "\r"                     { newline(); $setType(Token.SKIP); }
	| 	( "\n"              ) => "\n"                     { newline(); $setType(Token.SKIP); }
	|   ( "``"              ) => "``"                     { $setType(Token.SKIP); }
    |	( '"'	  			) => STRING                   { $setType(Token.SKIP); }
    |	( "//"	  			) => SL_COMMENT               { $setType(Token.SKIP); }
    |	( "/*"	  			) => ML_COMMENT               { $setType(Token.SKIP); }
    |	( IMPORT 			) => IMPORT		  			  { $setType(IMPORT); }
    |	( DEFINE_DEFINITION ) => DEFINE_DEFINITION		  { $setType(DEFINE); }
    |	( INCLUDE           ) => INCLUDE				  { $setType(INCLUDE); }
    |	( PACKAGE_DECL      ) => PACKAGE_DECL		      { $setType(PACKAGE_DECL); }
    | 	( IFDEF             ) => IFDEF					  { $setType(IFDEF); }
    | 	( DEFINE_REPLACE    ) => DEFINE_REPLACE			  { if (LA(1) == '`') $setType(Token.SKIP); else $setType(MACRO_CALL); }
	| 	~( '\u0000'..'\u0008' | '\u000B' | '\u000E'..'\u001F' | '\u007F'..'\uFFFF' ) { $setType(Token.SKIP); }
	;

// Whitespace -- ignored
protected
WS
    :
    	( options { greedy = true; } : ' ' | '\t' | '\f' | '\r' | '\n' )+ { $setType(Token.SKIP); }
    ;

// Single-line commentsML_COMMENT
protected
SL_COMMENT
    :
    	"//" ( options { generateAmbigWarnings = false; } : ( '\r' ~( '\n' | '\uFFFF' ) ) => '\r' { newline(); } ~( '\n' | '\uFFFF' ) | ~( '\n' | '\r' | '\uFFFF' ) )*
    	( '\n' | { LA(1)=='\r' && LA(2)=='\n' }? "\r\n" | '\uFFFF' ) { newline(); }
    ;

// Multiple-line comments
protected
ML_COMMENT
    :
		"/*"
		(/* '\r' '\n' can be matched in one alternative or by matching
		    '\r' in one iteration and '\n' in another. I am trying to
		    handle any flavor of newline that comes in, but the language
		    that allows both "\r\n" and "\r" and "\n" to all be valid
		    newline is ambiguous. Consequently, the resulting grammar
		    must be ambiguous. I'm shutting this warning off.
		 */
		options { generateAmbigWarnings = false; }
		:
		  { LA(2) != '/' }? '*'
		| "\r\n"   { newline(); }
		| '\r'     { newline(); }
		| '\n'     { newline(); }
		| ~( '*' | '\n' | '\r' | '\uFFFF' )
		)*
		"*/"
    ;

protected
ID options { testLiterals = true; }
    :
     	( 'a'..'z' | 'A'..'Z' | '_' ) ( options { greedy = true; } : 'a'..'z' | 'A'..'Z' | '0'..'9' | '_' | '$' )*
    ;

// String literals
protected
STRING
    :
    	'"'! ( ( '\\' ( DEFINE_WS )* ( '\r' | '\n' ) ) => STRING_NEWLINE | ( "\\\\" ) => "\\\\" | ( "\\\"" ) => "\\\"" | ~( '"' | '\n' | '\r' | '\uFFFF' ) | NEWLINE )* '"'!
    ;

protected
NEWLINE
    :
    	( { LA(1)=='\r' && LA(2)=='\n' }? "\r\n" | '\r' | '\n' ) { newline(); $setText(" "); }
    ;

protected
STRING_NEWLINE
    :
    	'\\' ( DEFINE_WS )* NEWLINE { $setText(" "); }
    ;

protected
DEFINE_WS
    :
    	' ' | '\t' | '\f'
    ;

protected
DEFINE_WS_OR_COMMENT
    :
    	( options { greedy = true; } : DEFINE_WS | ML_COMMENT )
    ;

protected
DEFINE_WS_NL_OR_COMMENT
    :
    	DEFINE_WS
    |	ML_COMMENT
    | 	SL_COMMENT
    | 	{ LA(1)=='\r' && LA(2)=='\n' }? "\r\n"   { newline(); }
    | 	'\r'                   { newline(); }
    | 	'\n'                   { newline(); }
    ;

protected
DEFINE_DEFINITION
    :
        '`'! ( DEFINE_WS! )* "define"! ( DEFINE_WS_OR_COMMENT! )+ ID
    ;

protected
DEFINE_REPLACE
    :
    	'`'! ( DEFINE_WS! )* ID ( DEFINE_WS_OR_COMMENT! )*
    ;

protected
DEFINE_WS_NL_OR_COMMENT_OPT
	:
		( DEFINE_WS_NL_OR_COMMENT )*
	;

protected
IFDEF
	:
		'`'! ( DEFINE_WS! )* ( "if"! ( 'n'! )? "def"! ) ( options { greedy = true; } : DEFINE_WS_NL_OR_COMMENT! )+ ID
	;

protected
INCLUDE
    :
    	'`'! ( DEFINE_WS! )* "include"! DEFINE_WS_NL_OR_COMMENT_OPT! STRING
    ;

protected
IMPORT
    :
    	"import"! ( WS! )+ ID ( WS! )* "::"!
    ;

protected
PACKAGE_DECL
    :
    	"package"! ( DEFINE_WS_NL_OR_COMMENT! )+ (("static"! | "automatic"!) ( DEFINE_WS_NL_OR_COMMENT! )+ )? ID ( DEFINE_WS_NL_OR_COMMENT! )* ";"!
    ;

protected
DEFINE_WS_OR_COMMENT_OPT
    :
    	( DEFINE_WS_OR_COMMENT ) => DEFINE_WS_OR_COMMENT DEFINE_WS_OR_COMMENT_OPT
    |   /* nothing */
    ;





