/*
Command file lexer
*/

header {
    package ro.amiq.dvt.buildconfig;

	import java.util.List;
	import java.util.Stack;
    import java.util.HashSet;
    import java.util.HashMap;
	import java.util.ArrayList;
	import java.util.StringTokenizer;

	import java.io.File;
	import java.io.ByteArrayOutputStream;
	import java.io.FileInputStream;
	import java.io.BufferedInputStream;
	import java.io.InputStream;
	import java.io.StringReader;
	import java.io.BufferedReader;
	import ro.amiq.dvt.startup.core.DVTLogger;

	import antlr.ANTLRException;
	import antlr.Parser;
	import antlr.RecognitionException;
	import antlr.TokenStreamException;
}

class BuildConfigLexer extends Lexer;

options {
    k = 2;
    exportVocab=BuildConfigLexer;
}

tokens {
	COMMENT;
}

{

	private IBuildConfigLexerListener parser;
	private int offset = 0;
	private boolean continueParsingException;
	private boolean keepStringDoubleQuotes;

    // SIMI: getText is not working if the comment is in Japanese
    private String getUnicodeText(int start) {
        // don't change even if the code looks stupid
        int length = text.length() - start;
        byte[] buffer = new byte[length];
        for (int i = 0; i < length; i++)
            buffer[i] = (byte) text.charAt(start + i);
        return new String(buffer);
    }

    @Override
    public void match(char arg0) throws MismatchedCharException, CharStreamException {
        if (inputState.guessing()==0)
            offset++;
        super.match(arg0);
    }

    @Override
    public void match(String arg0) throws MismatchedCharException, CharStreamException {
        if (inputState.guessing()==0)
            offset += arg0.length();
        super.match(arg0);
    }

    @Override
    public void match(BitSet arg0) throws MismatchedCharException, CharStreamException {
        if (inputState.guessing()==0)
            offset++;
        super.match(arg0);
    }

    @Override
    public void matchRange(char arg0, char arg1) throws MismatchedCharException, CharStreamException {
        if (inputState.guessing()==0)
            offset++;
        super.matchRange(arg0, arg1);
    }

    @Override
    public void matchNot(char arg0) throws MismatchedCharException, CharStreamException {
        if (inputState.guessing()==0)
            offset++;
        super.matchNot(arg0);
    }

	// override the token object class
    public void setTokenObjectClass(String cl) {
		tokenObjectClass = BuildConfigToken.class;
    }

    protected Token makeToken(int t) {
    	Token token = super.makeToken(t);
    	if (token instanceof BuildConfigToken) {
    		((BuildConfigToken)token).setOffset(offset);
    		((BuildConfigToken)token).setFilename(getFilename());
    	}
    	return token;
    }

    protected void checkForError() throws CharStreamException {
    	if (inputState.guessing() == 0) {
    		if (LA(1) == '\\' && LA(2) == '"' && LA(3) == '"') {
    			parser.reportProblem(BuildConfigParser.REPORT_ERROR, "This directive contains the illegal character sequence '\\\"\"' and will be ignored", getFilename(), getLine());
    			continueParsingException = true;
    		}
    	}
    }

	@Override
	public void userExceptionHandler(Exception ex) throws RecognitionException, CharStreamException, TokenStreamException {
		if (ex instanceof RecognitionException)
			throw (RecognitionException) ex;
		if (ex instanceof CharStreamException)
			throw (CharStreamException) ex;
		if (ex instanceof TokenStreamException)
			throw (TokenStreamException) ex;
		DVTLogger.INSTANCE.logError(ex);
	}

	public void setParser(IBuildConfigLexerListener parser) {
		this.parser = parser;
	}

	public boolean getContinueParsingException() {
		boolean temp = continueParsingException;
		continueParsingException = false;
		return temp;
	}
	
	public void setKeepStringDoubleQuotes(boolean keepStringDoubleQuotes) {
		this.keepStringDoubleQuotes = keepStringDoubleQuotes;
	}
	
}

protected
VOCAB
    :
      '\3'..'\377'
    ;

OTHERS
    :
      '\177'..'\377'
    {
        String temp = getUnicodeText(_begin);
        if (temp != null && temp.length() > 0) {
			throw new RecognitionException(
					"Unexpected char: '" + temp + "' (code 0x" + Integer.toHexString((int) temp.charAt(0)).toUpperCase() + ")",
					getFilename(), getLine(), getColumn(), -1, -1);
        }
    }
    ;

// Whitespace -- ignored
WS_NEWLINE
    :
      ( ' '
      | '\t'
      | '\f'
        // handle newlines
      | ( options { generateAmbigWarnings = false; }
        :
          "\r\n"        { newline(); } // Evil DOS
        | '\r'          { newline(); } // Macintosh
        | '\n'          { newline(); } // Unix (the right way)
        )
      )+
      {
		  $setType(Token.SKIP);
	  }
    ;

// Single-line comments
protected
SL_COMMENT returns [boolean skip = true]
    :
      ( "//" {parser.registerComment("//",getLine());} | "--" {parser.registerComment("--",getLine());} | { LA(1) == '#' && LA(2) == '#' && LA(3) == '#' }? "###" { skip = false; parser.registerComment("#",getLine());} | '#' {parser.registerComment("#",getLine());} )
      ( options { greedy = true; } : ~( '\n' | '\r' ) )* ( '\n' | '\r' ( options { greedy = true; } : '\n' )? | /* EOF */ )
      {
		  newline();
      }
    ;

protected
ML_COMMENT
{
	int startLine = 0;
}
    :
      "/*"! { startLine = getLine(); }
      (/* '\r' '\n' can be matched in one alternative or by matching
          '\r' in one iteration and '\n' in another.  I am trying to
          handle any flavor of newline that comes in, but the language
          that allows both "\r\n" and "\r" and "\n" to all be valid
          newline is ambiguous.  Consequently, the resulting grammar
          must be ambiguous.  I'm shutting this warning off.
       */
      options { generateAmbigWarnings = false; }
      :
        { LA(2) != '/' }? '*'
      | "\r\n"   { newline(); }
      | '\r'     { newline(); }
      | '\n'     { newline(); }
      | ~( '*' | '\n' | '\r' )
      )*
      "*/"!

    exception
  	catch [RecognitionException ex] {
		throw new RecognitionException(
				"Multiline comment started at line " + startLine + " not found until end of file.",
				ex.getFilename(), ex.getLine(), ex.getColumn(), -1, -1);
  	}
    ;

protected FORMATTED_NUMBER
	:
      NUMBER_BASE ( options { greedy = true; }: NUMBER_FORMAT )?
	| NUMBER_FORMAT
	;

// Base number (possible to form numbers using base
//number followed by formatted number). The 'general' number
//begining with '0' or containing dot, +/-, or exponent cannot be used
//together with formatted number.
protected NUMBER_BASE
 	:
 	  ( '0'..'9' ) ( options { greedy = true; } : '0'..'9' | '_' )* ( options { greedy = true; } : '.' ( options { greedy = true; } : '0'..'9' | '_' )* )?
 	    ( options { greedy = true; } :
 	      ( ( ( "step" ) => "step" | "s" | "ms" | "us" | "ns" | "ps" | "fs" ) )
 		| ( ( 'e' | 'E' ) ( options { greedy = true; } : '+' | '-' )? ( options { greedy = true; } : '0'..'9' )+ )
 		)?
	;

//Number formatted as DEC, HEX, OCT, BIN
protected NUMBER_FORMAT
{
	String tmp = "";
	char type = '?';
	char charZ = ' ';
	char charX = ' ';
	char charN = ' ';
	boolean report = true;
	boolean undefined = false;
}
 	:
 	  '\'' ( ( 's' | 'S' ) { type = 's'; } )?
      (
        ( 'd' | 'D' ) { type = 'd'; }
      | ( 'b' | 'B' ) { type = 'b'; }
      | ( 'h' | 'H' ) { type = 'h'; }
      | ( 'o' | 'O' ) { type = 'o'; }
      )
      (
        ( options { greedy = true; } : { type != 'd' || LA(1) != '?' }? nfb:NUMBER_FORMAT_DIGIT[type]
        {
            if (type == 'd') {
                char c = nfb.getText().charAt(0);
                if (c == 'x' || c == 'X') {
                    char tc = (charN != ' ')? charN: (charZ != ' ')? charZ: charX;
                    if (tc != ' ' && report) {
                    	parser.reportProblem(BuildConfigParser.REPORT_ERROR, "Near '" + tc + "': X is not allowed in decimal constants", getFilename(), getLine());
                    	report = false;
                    }
                    charX = c;
                } else if (c == 'z' || c == 'Z' || c == '?') {
                    char tc = (charN != ' ')? charN: (charZ != ' ')? charZ: charX;
                    if (tc != ' ' && report) {
                 	   	c = (c == 'z')? 'Z' : c;
                    	parser.reportProblem(BuildConfigParser.REPORT_ERROR, "Near '" + tc + "': " + c + " is not allowed in decimal constants", getFilename(), getLine());
                 	   	report = false;
                    }
                    charZ = c;
                } else if (c >= 0 && c <= '9') {
                    char tc = (charZ != ' ')? charZ: charX;
                    if (tc != ' ' && report) {
                    	parser.reportProblem(BuildConfigParser.REPORT_ERROR, "Near '" + tc + "': " + c + " is not allowed in decimal constants", getFilename(), getLine());
                    	report = false;
                    }
                    charN = c;
                }
            }
        }
        )+
    )
    ;

protected
NUMBER_FORMAT_DIGIT [char type]
    :
      ( '0'..'9' | 'A'..'F' | 'a'..'f' | '?' | '_' | 'z' | 'Z' | 'x' | 'X' )
    {
        try {
            String tmp = $getText;
            char c = (tmp != null && tmp.length() == 1)? tmp.charAt(0) : 0;
            String numberType = null;

            switch (type) {
            case 'd':
                if (!(c >= '0' && c <= '9') && c != 'z' && c != 'Z' && c != 'x' && c != 'X' && c != '?' && c != '_')
                    numberType = "decimal";
                break;
            case 'b':
                if (!(c >= '0' && c <= '1') && c != 'z' && c != 'Z' && c != 'x' && c != 'X' && c != '?' && c != '_')
                    numberType = "binary";
                break;
            case 'o':
                if (!(c >= '0' && c <= '7') && c != 'z' && c != 'Z' && c != 'x' && c != 'X' && c != '?' && c != '_')
                    numberType = "octal";
                break;
            default:
                if (!(c >= '0' && c <= '9') && c != 'z' && c != 'Z' && c != 'x' && c != 'X' && c != '?' && c != '_' && !(c >= 'a' && c <= 'f') && !(c >= 'A' && c <= 'F'))
                    numberType = "hexadecimal";
                break;
            }
            if (numberType != null)
            	parser.reportProblem(BuildConfigParser.REPORT_ERROR, "'" + c + "' is not allowed in " + numberType + " number", getFilename(), getLine());
        } catch (Exception e) {
        }
    }
    ;

protected
WS
    :
      ' ' | '\t' | '\f'
    ;

protected
NEWLINE
	:
	  ( ( "\r\n" ) => "\r\n" | '\r' | '\n' )
	{
    	newline();
    	$setText(" ");
	}
    ;

protected
ESCAPED_NEWLINE
	:
	  '\\' ( WS )* NEWLINE
	{
    	$setText(" ");
	}
	;
	
protected
STRING_DOUBLE_QUOTES:
	  { keepStringDoubleQuotes }? STRING_DOUBLE_QUOTES_KEEP_QUOTES
	| { !keepStringDoubleQuotes }? STRING_DOUBLE_QUOTES_REMOVE_QUOTES
	;

protected
STRING_DOUBLE_QUOTES_REMOVE_QUOTES
{
	int line = -1;
}
    :
	  '"'!
    (
    	( '\\' ( WS )* ( '\r' | '\n' ) ) => ESCAPED_NEWLINE
      | ( "\\\\" ) => "\\\\"!
      | ( "\\\"" ) => '\\'! '"' // unescape the quote, e.g "\" ... \"" => "...", see: #DVT-16145 
      | ~( '"' | '\r' | '\n' )
    )* ( '"'! | { userMark = userMark(); line = getLine(); } ( '\r'! | '\n'! )
    {
    	userRewind(userMark);
    	parser.reportProblem(BuildConfigParser.REPORT_ERROR, "This directive contains unclosed string literal (use '\\' to continue on the next line)", getFilename(), line);
    } )
    ;

protected
STRING_DOUBLE_QUOTES_KEEP_QUOTES
{
	int line = -1;
}
    :
	  '"'
    (
    	( '\\' ( WS )* ( '\r' | '\n' ) ) => ESCAPED_NEWLINE
      | ( "\\\\" ) => "\\\\"!
      | ( "\\\"" ) => "\\\""!
      | ~( '"' | '\r' | '\n' )
    )* ( '"' | { userMark = userMark(); line = getLine(); } ( '\r'! | '\n'! )
    {
    	userRewind(userMark);
    	parser.reportProblem(BuildConfigParser.REPORT_ERROR, "This directive contains unclosed string literal (use '\\' to continue on the next line)", getFilename(), line);
    } )
    ;

protected
ESCAPED_STRING_DOUBLE_QUOTES
{
	int line = -1;
}
	:
	  // Irun accepts \\\"<string>\\\" or \...\"<string>\...\" any number of '\', even different, and translates to "<string>"  	
	  ( '\\'! )+ '"'
      ( options { generateAmbigWarnings = false; }
      :
    	  ( '\\' ( WS )* ( '\r' | '\n' ) ) => ESCAPED_NEWLINE
    	| { LA(2) != '"' }? '\\'!
        | ~( '\\' | '\n' | '\r' )
      )* ( ( '\\'! )+ '"' | { userMark = userMark(); line = getLine(); } ( '\r'! | '\n'! )
    {
    	userRewind(userMark);
    	String content = getText();
    	content += "\"";
    	setText(content);
    	parser.reportProblem(BuildConfigParser.REPORT_ERROR, "This directive contains unclosed string literal (use '\\' to continue on the next line)", getFilename(), line);
    } )
  ;

protected
STRING_SINGLE_QUOTES
    :
	  '\''!
	  (
	    ( '\\' ( WS )* ( '\r' | '\n' ) ) => ESCAPED_NEWLINE
	  | ( "\\\\" ) => "\\\\"
	  | ( "\\'" ) => "\\'"
	  | ~( '\'' | '\n' | '\r' )
	  | NEWLINE
	  )*
	  '\''!
    ;
    
protected
ESCAPED_SINGLE_QUOTES
	:
	  (( '\\'! ) + '\''!
	  {
    	String content = getText();
    	content += "\'";
    	setText(content);
	  }
	  ) 
    ;
   

DIRECTIVE
{
	boolean skip = true;
}
    :
        ( "//" )   => SL_COMMENT { $setType(Token.SKIP); }
      | ( "#" )    => skip = SL_COMMENT { if (skip) $setType(Token.SKIP); else $setType(COMMENT); }
      | ( "--" )   => SL_COMMENT { $setType(Token.SKIP); }
      | ( "/*" )   => ML_COMMENT { $setType(Token.SKIP); }
      | ( ( "\\" )* "\\\"" ) => { checkForError(); } ESCAPED_STRING_DOUBLE_QUOTES
      | ( '"' )    => { checkForError(); } STRING_DOUBLE_QUOTES
      | ( '\'' )   => STRING_SINGLE_QUOTES
      | ( "\\\'" ) => ESCAPED_SINGLE_QUOTES
      | (
          ( ( "\\" )* "\\\"" ) => { checkForError(); } ESCAPED_STRING_DOUBLE_QUOTES
        | ( FORMATTED_NUMBER ) => FORMATTED_NUMBER
    	| { checkForError(); } STRING_DOUBLE_QUOTES
    	| STRING_SINGLE_QUOTES
    	| ( "\\\'" ) => ESCAPED_SINGLE_QUOTES
        | ~( ' ' | '\t' | '\f' | '\n' | '\r' | '"' | '\'' | '\177'..'\377' )
    	)+
      {
      	  String directive = getText();
      	  if (directive.endsWith("\\"))
      	      directive = directive.substring(0, directive.length() - 1);
      	  if (directive.length() == 0)
      	      $setType(Token.SKIP);
      }
    ;

