header {
package ro.amiq.dvt.logviewer.query.parsing;

import java.util.*;
import org.apache.lucene.search.*;
import ro.amiq.dvt.logviewer.LogViewerConst;
import ro.amiq.dvt.logviewer.indexing.*;
import ro.amiq.dvt.logviewer.query.*;
}

class LogViewerQueryParser extends Parser;
options {
    k = 2;
    buildAST = false;
	importVocab = LogViewerQueryLexer;
}

{
	private String filterText;
    private Map<String, LogViewerAttributeType> validAttributes;
    private List<LogViewerEntryStructure> entryStructures;
    private boolean canSort;
    private boolean allowCommands;
    private StringBuilder statusMessage;
    private Set<String> sortAttributes = new LinkedHashSet<>();
    private Set<String> hideAttributes = new HashSet<>();

    public LogViewerQueryParser(
            LogViewerQueryLexer lexer,
            String filterText,
            Map<String, LogViewerAttributeType> validAttributes,
            List<LogViewerEntryStructure> entryStructures,
            boolean canSort,
            boolean allowCommands,
            StringBuilder statusMessage
            ) {
        this(lexer);

		this.filterText = filterText;
        this.validAttributes  = validAttributes;
        this.entryStructures  = entryStructures;
        this.canSort = canSort;
        this.allowCommands = allowCommands;
        this.statusMessage = statusMessage;
    }

    public void reportError(RecognitionException e) {
		throw new LogViewerInvalidQuerySyntaxException();
    }
}

expression returns[LogViewerQueryWrapper wrapper]
	{
		Query query = null;
		wrapper = null;
	}
    : query=logicalOr EOF
        {
        	Sort sort = LogViewerQueryParserUtils.createSort(sortAttributes, validAttributes, statusMessage);
        	query = LogViewerQueryParserUtils.addQueryChecksForSort(query, sort, validAttributes);
        	
        	hideAttributes = LogViewerQueryParserUtils.validateHideAttributes(hideAttributes, validAttributes, statusMessage);
        	
            wrapper = new LogViewerQueryWrapper(
                    filterText,
                    false,
                    query,
                    sort,
                    hideAttributes);
        }
    ;

logicalOr returns [Query q]
	{
		Query auxQuery = null;
		q = null;
	}
    : auxQuery=logicalAnd!
        { q = auxQuery; }
      ( OR auxQuery=logicalAnd!
        {
			BooleanQuery.Builder bld = new BooleanQuery.Builder();
			if (q != null)
				bld.add(q, BooleanClause.Occur.SHOULD);

			if (auxQuery != null) {
				bld.add(auxQuery, BooleanClause.Occur.SHOULD);
				q = bld.build();
		    }
        }
      )*
    ;

logicalAnd returns [Query q]
	{
		Query auxQuery = null;
		q = null;
	}
    : auxQuery=clause! { q = auxQuery; }
      ( AND
      	auxQuery=clause!
			{
				BooleanQuery.Builder bld = new BooleanQuery.Builder();
				if (q != null)
					bld.add(q, BooleanClause.Occur.MUST);

				if (auxQuery != null) {
					bld.add(auxQuery, BooleanClause.Occur.MUST);
					q = bld.build();
				}
			}
      )*
    ;

clause returns [Query q]
	{
		Query auxQuery = null;
		q = null;
	}
    : LPAREN! auxQuery=logicalOr! RPAREN!
        { q = auxQuery; }
    | auxQuery=comparison
    	{ q = auxQuery; }
   	| auxQuery=command
    	{ q = auxQuery; }
    ;

comparison returns [Query q]
	{
		String attr = null;
		String op = null;
		String val = null;
		q = null;
	}
    : attr=attribute! op=operator! val=value!
        {
        	q = LogViewerQueryParserUtils.createQuery(attr, op, val, validAttributes, entryStructures);
        }
    ;

command returns [Query q]
	{
	    Set<String> attributes = null;
	    q = null;
	}
    : SORTBY attributes=attrList
	    {
			if (!allowCommands)
				throw new LogViewerInvalidQueryException(LogViewerConst.searchUnsupportedSortbyMessage());

	    	if (canSort)
	    		sortAttributes.addAll(attributes);
	    	else
	    		statusMessage.append(LogViewerConst.UNSUPPORTED_SORTBY_CLAUSE_MESSAGE);
	    }
    | HIDE attributes=attrList
	    {
			if (!allowCommands)
				throw new LogViewerInvalidQueryException(LogViewerConst.searchUnsupportedHideMessage());

	    	hideAttributes.addAll(attributes);
	    }
    ;

attribute returns [String text]
	{
		text = null;
	}
    : w:WORD { text = w.getText(); };

operator returns [String text]
	{
		text = null;
	}
    : EQ { text = "="; }
    | NE { text = "!="; }
    | LT { text = "<"; }
    | LE { text = "<="; }
    | GT { text = ">"; }
    | GE { text = ">="; }
    ;
    
value returns [String text]
	{
		text = null;
	}
    : s:STRING { text = s.getText(); }
    | w:WORD   { text = w.getText(); }
    ;

attrList returns [Set<String> attributes]
	{
	    attributes = new LinkedHashSet<String>();
	    String attributeName = null;
	}
    : attributeName=attribute { attributes.add(attributeName); }
      ( COMMA attributeName=attribute { attributes.add(attributeName); } )*
    ;
